/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2011/10/19
 */
public class LispComplexImpl extends LispComplex {

	//
	private LispReal real, imag;

	//
	/*package*/ LispComplexImpl(LispReal r, LispReal i) {
		this.real = r;
		this.imag = i;
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#getRealDouble()
	 */
	public double getRealDouble() {
		return real.getRealDouble();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getReal()
	 */
	public LispReal getReal() {
		return real;
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getImagDouble()
	 */
	public double getImagDouble() {
		return imag.getRealDouble();
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.Datum#getImag()
	 */
	public LispReal getImag() {
		return imag;
	}

	/*
	 * (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#isEqualTo(net.morilib.lisp.LispNumber)
	 */
	@Override
	public boolean isEqualTo(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;

			return ((real.isEqualTo(c.getReal())) &&
					(imag.isEqualTo(c.getImag())));
		} else if(x instanceof LispReal) {
			return false;
		} else {
			return super.isEqualTo(x);
		}
		//throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispComplex#toExact()
	 */
	@Override
	public LispComplex toExact() {
		return newComplex(real.toExact(), imag.toExact());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispComplex#toInexact()
	 */
	@Override
	public LispComplex toInexact() {
		return newComplex(real.toInexact(), imag.toInexact());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispComplex#conjugate()
	 */
	@Override
	public LispComplex conjugate() {
		return new LispComplexImpl(real, imag.uminus());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#add(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber add(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;

			return LispComplex.newComplex(
					real.add(c.getReal()),
					imag.add(c.getImag()));
		} else if(x instanceof LispReal) {
			return newComplex(real.add(x.getReal()), imag);
		} else if(x instanceof LispQuaternion) {
			return LispQuaternion.add(this, (LispQuaternion)x);
		} else if(x instanceof LispOctonion) {
			return LispOctonion.add(this, (LispOctonion)x);
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#sub(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber sub(LispNumber x) {
		if(x instanceof LispComplex) {
			LispComplex c = (LispComplex)x;

			return LispComplex.newComplex(
					real.subtract(c.getReal()),
					imag.subtract(c.getImag()));
		} else if(x instanceof LispReal) {
			return newComplex(real.subtract(x.getReal()), imag);
		} else if(x instanceof LispQuaternion) {
			return LispQuaternion.sub(this, (LispQuaternion)x);
		} else if(x instanceof LispOctonion) {
			return LispOctonion.sub(this, (LispOctonion)x);
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#mul(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber mul(LispNumber x) {
		if(x instanceof LispComplex) {
			LispReal xr = ((LispComplex)x).getReal();
			LispReal xi = ((LispComplex)x).getImag();

			if(xr.signum() == 0) {
				return newComplex(
						imag.uminus().multiply(xi),
						real.multiply(xi));
			} else {
				return newComplex(
						real.multiply(xr).subtract(imag.multiply(xi)),
						imag.multiply(xr).add(real.multiply(xi)));
			}
		} else if(x instanceof LispReal) {
			LispReal r = x.getReal();

			return newComplex(real.multiply(r), imag.multiply(r));
		} else if(x instanceof LispQuaternion) {
			return LispQuaternion.mul(this, (LispQuaternion)x);
		} else if(x instanceof LispOctonion) {
			return LispOctonion.mul(this, (LispOctonion)x);
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#div(net.morilib.lisp.LispNumber)
	 */
	@Override
	public LispNumber div(LispNumber x) {
		if(x instanceof LispComplex) {
			LispReal xr = ((LispComplex)x).getReal();
			LispReal xi = ((LispComplex)x).getImag();

			if(xr.signum() == 0) {
				return newComplex(
						imag.divide(xi), real.uminus().divide(xi));
			} else {
				LispReal nr = xr.multiply(xr).add(xi.multiply(xi));
				LispReal x1 =
					real.multiply(xr).add(imag.multiply(xi));
				LispReal x2 =
					imag.multiply(xr).subtract(real.multiply(xi));

				return newComplex(x1.divide(nr), x2.divide(nr));
			}
		} else if(x instanceof LispReal) {
			LispReal r = x.getReal();

			return newComplex(real.divide(r), imag.divide(r));
		} else if(x instanceof LispQuaternion) {
			return LispQuaternion.div(this, (LispQuaternion)x);
		} else if(x instanceof LispOctonion) {
			return LispOctonion.div(this, (LispOctonion)x);
		}
		throw new IllegalArgumentException(x.toString());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#uminus()
	 */
	@Override
	public LispComplexImpl uminus() {
		return new LispComplexImpl(real.uminus(), imag.uminus());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#isExact()
	 */
	@Override
	public boolean isExact() {
		return real.isExact();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.LispNumber#norm()
	 */
	@Override
	public LispReal norm() {
		return (LispReal)LispMath.sqrt(
				(LispReal)real.mul(real).add(imag.mul(imag)));
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object x) {
		if(x instanceof LispComplexImpl) {
			LispComplexImpl c = (LispComplexImpl)x;

			return real.equals(c.real) && imag.equals(c.imag);
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		int l = 17;

		l = 37 * (l + real.hashCode());
		l = 37 * (l + imag.hashCode());
		return l;
	}

}
