/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.math.BigInteger;

import net.morilib.lisp.sos.LispType;
import net.morilib.util.Inclementor;
import net.morilib.util.InclementorBoundsException;

public final class LispCharacter extends Atom
implements Comparable<LispCharacter>, Inclementor<LispCharacter> {
	
	
	public static final LispCharacter NEWLINE =
		new LispCharacter('\n');
	
	public static final LispCharacter SPACE = new LispCharacter(' ');
	
	//
	private char character;
	
	
	public LispCharacter(char ch) {
		character = ch;
	}
	
	
	private static String getName(char ch) {
		if(ch == '\n') {
			return "newline";
		} else if(ch == ' ') {
			return "space";
		} else {
			return Character.toString(ch);
		}
	}
	
	
	public static LispCharacter getByName(String name) {
		if("newline".equals(name)) {
			return NEWLINE;
		} else if("space".equals(name)) {
			return SPACE;
		} else {
			return null;
		}
	}
	
	/**
	 * @return the character
	 */
	public char getCharacter() {
		return character;
	}
	
	@Override
	public String getResult() {
		return "#\\" + getName(character);
	}

	@Override
	public String print() {
		return Character.toString(character);
	}

	@Override
	public LispString toLispString() {
		return new LispString("#\\" + getName(character));
	}
	
	
	public boolean equals(Object d) {
		if(d instanceof LispCharacter) {
			return character == ((LispCharacter)d).character;
		}
		return false;
	}
	
	
	public int hashCode() {
		int l = 17;
		
		l = 37 * l + (int)character;
		return l;
	}
	
	
	public String toString() {
		return Character.toString(character);
	}
	
	
	public boolean isTypeCharacter() {
		return true;
	}


	public int compareTo(LispCharacter o) {
		return (character < o.character) ? -1 :
			((character > o.character) ? 1 : 0);
	}


	public boolean equalIncliment(Inclementor<?> i) {
		if(i instanceof LispCharacter) {
			return character == ((LispCharacter)i).character;
		} else {
			return character == i.toInt();
		}
	}


	public boolean equalInt(int i) {
		return character == i;
	}


	public LispCharacter getObject() {
		return this;
	}


	public boolean isZero() {
		return character == 0;
	}


	public Inclementor<LispCharacter> suc() {
		if(character >= Character.MAX_VALUE) {
			throw new InclementorBoundsException();
		}
		return new LispCharacter((char)(character + 1));
	}


	public Inclementor<LispCharacter> suc(int step) {
		if((character + step) >= Character.MAX_VALUE) {
			throw new InclementorBoundsException();
		}
		return new LispCharacter((char)(character + 1));
	}


	public int toInt() {
		return character;
	}


	public boolean hasNext() {
		return character < Character.MAX_VALUE;
	}


	public Inclementor<LispCharacter> getZero() {
		return new LispCharacter((char)0);
	}


	public boolean equalInt(BigInteger i) {
		return i.equals(BigInteger.valueOf(character));
	}
	
	
	public LispType getType() {
		return LispType.CHAR;
	}

}
