/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PushbackReader;
import java.io.Reader;

public class InputPort extends Datum {
	
	//
	private PushbackReader reader;
	private boolean standard;
	private Parser parser;
	private boolean reachedEOF = false;
	private boolean closed = false;
	private LispMessage msg;
	
	//
	private InputPort(Reader rd, boolean std, LispMessage msg) {
		this.reader     = new PushbackReader(rd);
		this.standard   = std;
		this.msg        = msg;
		this.parser     = new Parser(this.reader, msg);
	}
	
	
	/*package*/ InputPort(Reader rd, LispMessage msg) {
		this(rd, false, msg);
	}
	
	
	/*package*/ InputPort(String fname, LispMessage msg) {
		try {
			Reader rd;
			
			rd = new InputStreamReader(new FileInputStream(fname));
			this.reader     = new PushbackReader(rd);
			this.standard   = false;
			this.msg        = msg;
			this.parser     = new Parser(this.reader, msg);
		} catch (FileNotFoundException e) {
			throw new LispIOException(e);
		}
		
	}
	
	
	/*package*/ static InputPort getStandard(LispMessage msg) {
		return new InputPort(
				new InputStreamReader(System.in), true, msg);
	}
	
	
	public Datum readChar() {
		if(closed) {
			throw msg.getError("err.port.closed");
		} else if(reachedEOF) {
			return EOFObject.EOF;
		}
		
		try {
			int ch = reader.read();
			
			if(ch < 0) {
				reachedEOF = true;
				return EOFObject.EOF;
			} else {
				return new LispCharacter((char)ch);
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	
	
	public Datum peekChar() {
		if(closed) {
			throw msg.getError("err.port.closed");
		} else if(reachedEOF) {
			return EOFObject.EOF;
		}
		
		try {
			int ch = reader.read();
			
			if(ch < 0) {
				reachedEOF = true;
				return EOFObject.EOF;
			} else {
				reader.unread(ch);
				return new LispCharacter((char)ch);
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	
	
	/*
	public Datum readLine() {
		if(closed) {
			throw msg.getError("err.port.closed");
		} else if(reachedEOF) {
			return EOFObject.EOF;
		}
		
		try {
			String line = lineReader.readLine();
			
			if(line == null) {
				reachedEOF = true;
				return EOFObject.EOF;
			} else {
				return new LispString(line);
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	*/
	
	
	public Datum read() {
		if(closed) {
			throw msg.getError("err.port.closed");
		} else if(reachedEOF) {
			return EOFObject.EOF;
		}
		
		try {
			/*String line;
			
			parser.clear();
			
			line = lineReader.readLine();
			if(line == null) {
				reachedEOF = true;
				return EOFObject.EOF;
			}
			
			do {
				parser.read(line);
				if(parser.parse()) {
					return parser.getDatum();
				}
			} while((line = lineReader.readLine()) != null);
			
			// lineがnull
			reachedEOF = true;
			throw new ReadException("unexpected EOF");*/
			parser.clear();
			if(parser.parse()) {
				Datum res = parser.getDatum();
				
				if(res != null) {
					return res;
				} else {
					reachedEOF = true;
					return EOFObject.EOF;
				}
			} else {
				reachedEOF = true;
				if(!parser.isReadBefore()) {
					return EOFObject.EOF;
				} else {
					throw msg.getReadError("err.read.eof");
					//return EOFObject.EOF;
				}
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	
	
	public void close() {
		try {
			if(!closed && !standard) {
				reader.close();
				closed = true;
			}
		} catch (IOException e) {
			throw new LispIOException(e);
		}
	}
	
	
	public boolean isStandard() {
		return standard;
	}
	
	
	public boolean isReachedEOF() {
		return reachedEOF;
	}
	
	
	public boolean isTypePort() {
		return true;
	}
	
	
	/*package*/ Parser getParser() {
		return parser;
	}
	
}
