/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: strtool.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<string.h>
#include	<ctype.h>
#include	"strlist.h"
#include	"strtool.h"

int 
strtool_len(const char *str)
{
	if (!str)
	{
		goto ERROR;
	}
	return (strlen(str));
ERROR:
	return (0);
}

int 
strtool_isempty(const char *str)
{
	if (!str || *str == '\0')
	{
		return (1);
	}
	return (0);
}

char           *
strtool_toupper(char *str)
{
	char           *src = str;

	if (!strtool_isempty(str))
	{
		while (*str)
		{
			*str = toupper(*str);
			str++;
		}
	}
	return (src);
}

char           *
strtool_tolower(char *str)
{
	char           *src = str;

	if (!strtool_isempty(str))
	{
		while (*str)
		{
			*str = tolower(*str);
			str++;
		}
	}
	return (src);
}

char           *
strtool_dup(const char *src)
{
	if (strtool_isempty(src))
	{
		goto ERROR;
	}
	return (strdup(src));
ERROR:
	return (NULL);
}

char           *
strtool_ndup(const char *src, const int size)
{
	char           *dst = NULL;

	if (strtool_isempty(src) || size <= 0 || size > strlen(src))
	{
		goto ERROR;
	}
	dst = (char *) calloc(size + 1, sizeof(char));
	if (!dst)
	{
		goto ERROR;
	}
	memcpy(dst, src, size);

	return (dst);
ERROR:
	return (NULL);
}

int 
strtool_ncat(char **str1, const char *str2, const int size)
{
	int             str1_len = 0;
	int             str2_len = 0;
	char           *tmp = NULL;

	if (!str1)
	{
		goto ERROR;
	}
	str1_len = strtool_len(*str1);
	str2_len = strtool_len(str2);

	if ((!str1_len && !str2_len) || size > str2_len || size < 1)
	{
		goto ERROR;
	}
	if (str1_len)
	{
		tmp = (char *) calloc(str1_len + size + 1, sizeof(char));
		strcpy(tmp, *str1);
		strncat(tmp, str2, size);
	} else
	{
		tmp = strtool_ndup(str2, size);
	}

	free(*str1);
	*str1 = tmp;

	return (0);
ERROR:
	return (1);
}

int 
strtool_cat(char **str1, const char *str2)
{
	return (strtool_ncat(str1, str2, strtool_len(str2)));
}

int 
strtool_strip(char **str)
{
	char           *sp, *ep;
	char           *tp;

	if (strtool_isempty(*str))
	{
		goto ERROR;
	}
	tp = *str;

	sp = tp;
	ep = tp + strlen(tp) - 1;

	while (sp != ep && (*sp == ' ' || *sp == '\t'))
	{
		sp++;
	}

	while (sp != ep && (*ep == ' ' || *ep == '\t'))
	{
		ep--;
	}


	if (sp == ep)
	{
		*tp = '\0';
	} else
	{
		*(ep + 1) = '\0';

		while (sp <= ep + 1)
		{
			*tp = *sp;
			sp++;
			tp++;
		}
	}


	return (0);
ERROR:
	return (1);
}

strlist_t      *
strtool_split(const char *str, const char *delim, const int time)
{
	strlist_t      *dst = NULL;
	char           *sp, *ep, *tmp;

	if (strtool_isempty(str) || strtool_isempty(delim) || time < 0)
	{
		goto ERROR;
	}
	dst = strlist_create();

	sp = (char *) str;

	while (1)
	{
		ep = strstr(sp, delim);

		if (!ep || (time && strlist_len(dst) >= time))
		{
			strlist_add(dst, sp);
			break;
		}
		tmp = strtool_ndup(sp, ep - sp);

		strlist_add(dst, tmp);

		free(tmp);

		sp = ep + strlen(delim);
	}

	return (dst);
ERROR:
	strlist_destroy(dst);
	return (NULL);
}

char           *
strtool_sprintf(const char *fmt,...)
{
	char            buf[2048];
	va_list         ap;
	int             err;

	if (!fmt)
	{
		goto ERROR;
	}
	memset(buf, 0, sizeof(buf));

	va_start(ap, fmt);

	err = vsnprintf(buf, sizeof(buf), fmt, ap);

	va_end(ap);

	if (err <= 0 || err > sizeof(buf))
	{
		goto ERROR;
	}
	return (strdup(buf));
ERROR:
	return (NULL);
}

char           *
strtool_to_hex(const char *str)
{
	char           *dst = NULL;
	int             n, len;

	len = strtool_len(str);
	if (len == 0)
	{
		goto ERROR;
	}
	dst = (char *) calloc(len * 2 + 1, sizeof(char));
	if (!dst)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		sprintf(dst + n * 2, "%x", (*(str + n) >> 4) & 0x0F);
		sprintf(dst + n * 2 + 1, "%x", *(str + n) & 0x0F);
	}

	return (dst);
ERROR:
	return (NULL);
}

int 
strtool_isequal(const char *str1, const char *str2)
{
	if (!str1 || !str2)
	{
		goto ERROR;
	}
	if (strcmp(str1, str2) != 0)
	{
		goto ERROR;
	}
	return (1);
ERROR:
	return (0);
}

int 
strtool_isdigit(const char *str)
{
	int             n, len;

	len = strtool_len(str);
	if (len <= 0)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		if (!isdigit(str[n]))
		{
			goto ERROR;
		}
	}

	return (1);
ERROR:
	return (0);
}

int 
strtool_add_fence(char **src, char fence)
{
	char           *dst = NULL;

	if (!src || !strtool_len(*src) || !fence)
	{
		goto ERROR;
	}
	dst = (char *) calloc(strlen(*src) + 3, sizeof(char));
	if (!dst)
	{
		goto ERROR;
	}
	*(dst + 0) = fence;
	strcpy(dst + 1, *src);
	*(dst + strlen(*src) + 1) = fence;

	free(*src);

	*src = dst;

	return (0);
ERROR:
	return (1);
}

int 
strtool_remove_fence(char **src, char fence)
{
	char           *tmp = NULL;

	if (!src || strtool_len(*src) < 3 || !fence)
	{
		goto ERROR;
	}
	for (tmp = *src; *(tmp + 1) && *(tmp + 1) != fence; tmp++)
	{
		*tmp = *(tmp + 1);
	}

	*(tmp) = '\0';

	return (0);
ERROR:
	return (1);
}

int 
strtool_remove_prefix(char *src, const char *prefix)
{
	int             len;

	if (!src || !prefix)
	{
		goto ERROR;
	}
	if (strlen(src) < strlen(prefix))
	{
		goto ERROR;
	}
	len = strlen(prefix);

	while (*(src + len))
	{
		*src = *(src + len);
		src++;
	}

	*src = '\0';

	return (0);
ERROR:
	return (1);
}

int 
strtool_cmp(const char *str1, const char *str2)
{
	if (!str1 && !str2)
	{
		return (0);
	}
	if (!str1 && str2)
	{
		return (-1);
	}
	if (str1 && !str2)
	{
		return (1);
	}
	return (strcmp(str1, str2));
}
