/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: strlist.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	"dlist.h"
#include	"strlist.h"
#include	"strtool.h"

struct _str_array_
{
	dlist_t        *dlist;
};

void            strlist_free(void *data);

strlist_t      *
strlist_create(void)
{
	strlist_t      *obj = NULL;

	obj = (strlist_t *) calloc(1, sizeof(strlist_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->dlist = dlist_create();
	if (!obj->dlist)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	strlist_destroy(obj);
	return (NULL);
}

int 
strlist_destroy(strlist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	dlist_destroy(obj->dlist, strlist_free);

	free(obj);

	return (0);
ERROR:
	return (1);
}

void 
strlist_free(void *data)
{
	if (data)
	{
		free(data);
	}
}

char           *
strlist_pop(strlist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	return ((char *) dlist_pop(obj->dlist));
ERROR:
	return (NULL);
}

int 
strlist_push(strlist_t * obj, const char *data)
{
	char           *tmp = NULL;
	int             err;

	if (!obj || strtool_isempty(data))
	{
		goto ERROR;
	}
	tmp = strdup(data);

	err = dlist_push(obj->dlist, (void *) tmp);

	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	free(tmp);
	return (1);
}

char           *
strlist_dequeue(strlist_t * obj)
{
	return ((char *) dlist_dequeue(obj->dlist));
}

int 
strlist_queue(strlist_t * obj, const char *data)
{
	char           *tmp = NULL;
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	tmp = strdup(data);

	err = dlist_queue(obj->dlist, (void *) tmp);

	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	free(tmp);
	return (1);
}

int 
strlist_add(strlist_t * obj, const char *data)
{
	return (strlist_push(obj, data));
}

int 
strlist_add_next(strlist_t * obj, const int offset, const char *data)
{
	char           *tmp = NULL;
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	tmp = strdup(data);

	err = dlist_add_next(obj->dlist, offset, (void *) tmp);

	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	free(tmp);
	return (1);
}

char           *
strlist_get(const strlist_t * obj, const int offset)
{
	char           *tmp = NULL;

	tmp = strlist_refer(obj, offset);

	if (!tmp)
	{
		goto ERROR;
	}
	return (strdup(tmp));
ERROR:
	return (NULL);
}

char           *
strlist_refer(const strlist_t * obj, const int offset)
{
	if (!obj)
	{
		goto ERROR;
	}
	return ((char *) dlist_get_data_by_offset(obj->dlist, offset));
ERROR:
	return (NULL);
}

strlist_t      *
strlist_dup(const strlist_t * obj)
{
	if (!obj)
	{
		return (NULL);
	}
	return (strlist_ndup(obj, strlist_len(obj)));
}

strlist_t      *
strlist_ndup(const strlist_t * obj, const int len)
{
	strlist_t      *dst = NULL;
	int             n;

	if (!obj || len <= 0 || len > strlist_len(obj))
	{
		goto ERROR;
	}
	dst = strlist_create();
	if (!dst)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		strlist_push(dst, strlist_refer(obj, n));
	}

	return (dst);
ERROR:
	return (NULL);
}

int 
strlist_cat(strlist_t * dst, const strlist_t * src)
{
	int             err;

	if (!dst || !src)
	{
		goto ERROR;
	}
	err = strlist_ncat(dst, src, strlist_len(src));
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
strlist_ncat(strlist_t * dst, const strlist_t * src, const int len)
{
	int             n;
	char           *tmp;

	if (!dst || !src || len > strlist_len(src))
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		tmp = strlist_get(src, n);
		strlist_push(dst, tmp);
		free(tmp);
	}

	return (0);
ERROR:
	return (1);
}

char           *
strlist_join_all(const strlist_t * obj, const char *sep)
{
	return (strlist_join(obj, sep, 0, strlist_len(obj)));
}

char           *
strlist_join(
	     const strlist_t * obj,
	     const char *sep,
	     const int from,
	     const int to)
{
	char           *dst = NULL;
	int             n, len;

	if (!obj || !sep)
	{
		goto ERROR;
	}
	if (from < 0 || to <= from || to > strlist_len(obj))
	{
		goto ERROR;
	}
	for (n = from, len = to; n < len; n++)
	{
		strtool_cat(&dst, (char *) strlist_refer(obj, n));
		if (n + 1 != len)
		{
			strtool_cat(&dst, sep);
		}
	}

	return (dst);
ERROR:
	return (NULL);
}


int 
strlist_is_exist(const strlist_t * obj, const char *data)
{
	char           *tmp;
	int             n, len;

	if (obj && !strtool_isempty(data))
	{
		len = strlist_len(obj);

		for (n = 0; n < len; n++)
		{
			tmp = dlist_get_data_by_offset(obj->dlist, n);

			if (tmp && strcmp(tmp, data) == 0)
			{
				return (1);
			}
		}
	}
	return (0);
}

int 
strlist_get_offset(const strlist_t * obj, const char *data)
{
	int             n, len;
	char           *tmp;

	if (!(len = strlist_len(obj)) || strtool_isempty(data))
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		tmp = strlist_refer(obj, n);

		if (strcmp(tmp, data) == 0)
		{
			if (n == 0)
			{
				return (0);
			} else
			{
				break;
			}
		}
	}

	if (!n || n >= len)
	{
		goto ERROR;
	}
	return (n);
ERROR:
	return (-1);
}

int 
strlist_len(const strlist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (dlist_len(obj->dlist));
ERROR:
	return (0);
}
