/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: mx.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>

#ifdef	Mac
#define	BIND_8_COMPAT
#include	<sys/types.h>
#endif

#include	<netinet/in.h>
#include	<arpa/nameser.h>
#include	<resolv.h>
#include	"strtool.h"
#include	"dlist.h"
#include	"mx.h"

struct mx_record
{
	short           preference;
	char           *domain;
};

struct mx
{
	dlist_t        *dlist;
};

mx_t           *mx_create(void);
void            mx_free(void *data);
int             mx_add_record(mx_t * obj, const short pref, const char *domain);

mx_t           *
mx_create(void)
{
	mx_t           *obj = NULL;

	obj = (mx_t *) calloc(1, sizeof(mx_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->dlist = dlist_create();
	if (!obj->dlist)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	mx_destroy(obj);
	return (NULL);
}

int 
mx_destroy(mx_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	dlist_destroy(obj->dlist, mx_free);

	free(obj);

	return (0);
ERROR:
	return (1);
}

void 
mx_free(void *data)
{
	struct mx_record *obj = (struct mx_record *) data;

	if (!obj)
	{
		return;
	}
	free(obj->domain);

	free(obj);

	return;
}

mx_t           *
mx_get_records(const char *domain)
{
	mx_t           *obj = NULL;
	unsigned char   ans[PACKETSZ];
	unsigned char  *ptr;
	HEADER         *hdr = (HEADER *) ans;
	char            tmp[MAXDNAME];
	int             ans_len;
	int             n, len, cnt, err;
	short           type, class, rdlength, preference;
	long            ttl;

	obj = mx_create();
	if (!obj)
	{
		goto ERROR;
	}
	ans_len = res_query(domain, C_IN, T_MX, ans, sizeof(ans));
	if (ans_len < sizeof(HEADER) || hdr->rcode != 0)
	{
		goto ERROR;
	}
	/* skip Question Section */
	cnt = ntohs(hdr->qdcount);
	ptr = ans + sizeof(HEADER);
	for (n = 0; n < cnt; n++)
	{
		len = dn_expand(ans, ans + ans_len, ptr, tmp, sizeof(tmp));
		ptr += len + INT16SZ + INT16SZ;
	}

	/* Answer Section */
	cnt = ntohs(hdr->ancount);
	for (n = 0; n < cnt; n++)
	{
		len = dn_expand(ans, ans + ans_len, ptr, tmp, sizeof(tmp));
		ptr += len;

		GETSHORT(type, ptr);
		GETSHORT(class, ptr);
		GETLONG(ttl, ptr);
		GETSHORT(rdlength, ptr);

		/*
		 * fprintf(stdout, "%d\t%d\t%ld\t%d", type, class, ttl,
		 * rdlength);
		 */

		if (type != T_MX)
		{
			/* fprintf(stdout, "\n"); */
			ptr += rdlength;
			continue;
		}
		GETSHORT(preference, ptr);

		len = dn_expand(ans, ans + ans_len, ptr, tmp, sizeof(tmp));

		/* fprintf(stdout, "\t%d\t%s\n", preference, tmp); */

		err = mx_add_record(obj, preference, tmp);

		if (err)
		{
			goto ERROR;
		}
		ptr += len;
	}

	return (obj);
ERROR:
	mx_destroy(obj);
	return (NULL);
}

int 
mx_add_record(mx_t * obj, const short preference, const char *domain)
{
	struct mx_record *tmp, *data = NULL;
	int             err, n, len;

	if (!obj || strtool_isempty(domain))
	{
		goto ERROR;
	}
	len = mx_len(obj);

	data = (struct mx_record *) calloc(1, sizeof(struct mx_record));
	if (!data)
	{
		goto ERROR;
	}
	data->preference = preference;
	data->domain = strdup(domain);

	if (len == 0)
	{
		err = dlist_push(obj->dlist, (void *) data);
	} else if (len == 1)
	{
		tmp = dlist_get_data_by_offset(obj->dlist, 0);
		if (!tmp)
		{
			goto ERROR;
		}
		if (tmp->preference > preference)
		{
			err = dlist_queue(obj->dlist, (void *) data);
		} else
		{
			err = dlist_push(obj->dlist, (void *) data);
		}
	} else
	{
		for (n = 0; n < len; n++)
		{
			tmp = dlist_get_data_by_offset(obj->dlist, n);
			if (!tmp)
			{
				goto ERROR;
			}
			if (tmp->preference > preference)
			{
				break;
			}
		}

		if (n == 0)
		{
			err = dlist_queue(obj->dlist, (void *) data);
		} else
		{
			err = dlist_add_next(obj->dlist, n - 1, (void *) data);
		}
	}

	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	mx_free((void *) data);
	return (1);
}

char           *
mx_refer(const mx_t * obj, const int offset)
{
	struct mx_record *data;

	if (!obj)
	{
		goto ERROR;
	}
	data = (struct mx_record *) dlist_get_data_by_offset(obj->dlist, offset);
	if (!data)
	{
		goto ERROR;
	}
	return (data->domain);
ERROR:
	return (NULL);
}

char           *
mx_get(const mx_t * obj, const int offset)
{
	char           *tmp = NULL;

	tmp = mx_refer(obj, offset);
	if (!tmp)
	{
		goto ERROR;
	}
	return (strdup(tmp));
ERROR:
	return (NULL);
}

int 
mx_len(const mx_t * obj)
{
	if (!obj)
	{
		return (0);
	}
	return (dlist_len(obj->dlist));
}
