/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: multilist.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	"dlist.h"
#include	"strlist.h"
#include	"strtool.h"
#include	"multilist.h"

struct multi
{
	char           *name;
	strlist_t      *value;
};

struct multilist
{
	dlist_t        *dlist;
};

struct multi   *multilist_calloc(void);
void            multilist_free(void *data);

multilist_t    *
multilist_create(void)
{
	multilist_t    *obj = NULL;

	obj = (multilist_t *) calloc(1, sizeof(multilist_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->dlist = dlist_create();
	if (!obj->dlist)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	multilist_destroy(obj);
	return (NULL);
}

int 
multilist_destroy(multilist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	dlist_destroy(obj->dlist, multilist_free);

	free(obj);

	return (0);
ERROR:
	return (1);
}

struct multi   *
multilist_calloc(void)
{
	struct multi   *obj = NULL;

	obj = (struct multi *) calloc(1, sizeof(struct multi));
	if (!obj)
	{
		goto ERROR;
	}
	obj->name = NULL;
	obj->value = NULL;

	return (obj);
ERROR:
	return (NULL);
}

void 
multilist_free(void *data)
{
	struct multi   *tmp = (struct multi *) data;

	if (tmp)
	{
		free(tmp->name);
		strlist_destroy(tmp->value);
		free(tmp);
	}
}

int 
multilist_pop(
	      multilist_t * obj,
	      char **name,
	      strlist_t ** value)
{
	struct multi   *data = NULL;

	data = dlist_pop(obj->dlist);
	if (!data)
	{
		goto ERROR;
	}
	*name = data->name;
	*value = data->value;

	free(data);

	return (0);
ERROR:
	return (1);
}

int 
multilist_push(
	       multilist_t * obj,
	       const char *name,
	       const strlist_t * value)
{
	struct multi   *data = NULL;
	int             err;

	if (!obj || strtool_isempty(name) || strlist_len(value) == 0)
	{
		goto ERROR;
	}
	data = multilist_calloc();
	if (!data)
	{
		goto ERROR;
	}
	data->name = strdup(name);
	data->value = strlist_dup(value);

	err = dlist_push(obj->dlist, (void *) data);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	multilist_free((void *) data);
	return (1);
}

int 
multilist_add(
	      multilist_t * obj,
	      const char *name,
	      const strlist_t * value)
{
	return (multilist_push(obj, name, value));
}

int 
multilist_get(
	      const multilist_t * obj,
	      const int offset,
	      char **name,
	      strlist_t ** value)
{
	int             err;
	char           *t_name;
	strlist_t      *t_value;

	if (!name || !value)
	{
		goto ERROR;
	}
	err = multilist_refer(obj, offset, &t_name, &t_value);
	if (err)
	{
		goto ERROR;
	}
	*name = strdup(t_name);
	*value = strlist_dup(t_value);

	return (0);
ERROR:
	return (1);
}

int 
multilist_refer(
		const multilist_t * obj,
		const int offset,
		char **name,
		strlist_t ** value)
{
	struct multi   *data = NULL;

	if (!obj || !name || !value)
	{
		goto ERROR;
	}
	data = (struct multi *) dlist_get_data_by_offset(obj->dlist, offset);

	if (!data)
	{
		goto ERROR;
	}
	*name = data->name;
	*value = data->value;

	return (0);
ERROR:
	return (1);
}

int 
multilist_dequeue(
		  multilist_t * obj,
		  char **name,
		  strlist_t ** value)
{
	struct multi   *data = NULL;

	if (!obj || !name || !value)
	{
		goto ERROR;
	}
	data = dlist_dequeue(obj->dlist);
	if (!data)
	{
		goto ERROR;
	}
	*name = data->name;
	*value = data->value;

	free(data);

	return (0);
ERROR:
	return (1);
}

int 
multilist_queue(
		multilist_t * obj,
		const char *name,
		const strlist_t * value)
{
	struct multi   *data = NULL;

	if (!obj || strtool_isempty(name) || strlist_len(value) == 0)
	{
		goto ERROR;
	}
	data = multilist_calloc();
	if (!data)
	{
		goto ERROR;
	}
	data->name = strdup(name);
	data->value = strlist_dup(value);

	return (dlist_push(obj->dlist, (void *) data));
ERROR:
	return (1);
}

int 
multilist_add_value_by_name(
			    multilist_t * obj,
			    const char *name,
			    const char *value)
{
	strlist_t      *tmp;

	if (!obj || strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	tmp = multilist_get_value_by_name(obj, name);
	if (tmp)
	{
		strlist_add(tmp, value);
	} else
	{
		tmp = strlist_create();

		strlist_add(tmp, value);

		multilist_add(obj, name, tmp);

		strlist_destroy(tmp);
	}

	return (0);
ERROR:
	return (1);
}

strlist_t      *
multilist_get_value_by_name(
			    const multilist_t * obj,
			    const char *name)
{
	strlist_t      *value = NULL;

	value = multilist_refer_value_by_name(obj, name);
	if (!value)
	{
		return (NULL);
	}
	return (strlist_dup(value));
}

strlist_t      *
multilist_refer_value_by_name(
			      const multilist_t * obj,
			      const char *name)
{
	int             n, len;
	char           *tmp_name;
	strlist_t      *value;

	if (!obj || strtool_isempty(name))
	{
		goto ERROR;
	}
	len = multilist_len(obj);

	for (n = 0; n < len; n++)
	{
		multilist_refer(obj, n, &tmp_name, &value);
		if (strcmp(tmp_name, name) == 0)
		{
			return (value);
		}
	}
ERROR:
	return (NULL);
}

multilist_t    *
multilist_dup(const multilist_t * obj)
{
	if (!obj)
	{
		return (NULL);
	}
	return (multilist_ndup(obj, multilist_len(obj)));
}

multilist_t    *
multilist_ndup(const multilist_t * obj, const int len)
{
	multilist_t    *dst = NULL;
	char           *name;
	strlist_t      *value;
	int             n;

	if (!obj || len < 0 || len > multilist_len(obj))
	{
		goto ERROR;
	}
	dst = multilist_create();
	if (!dst)
	{
		goto ERROR;
	}
	for (n = 0; n < len; n++)
	{
		multilist_refer(obj, n, &name, &value);
		multilist_add(dst, name, value);
	}

	return (dst);
ERROR:
	return (NULL);
}

int 
multilist_len(const multilist_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (dlist_len(obj->dlist));
ERROR:
	return (0);
}
