/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: log.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<stdarg.h>
#include	<unistd.h>
#include	<time.h>
#include	<errno.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<fcntl.h>
#include	"filetool.h"
#include	"strlist.h"
#include	"strtool.h"
#include	"sem.h"
#include	"log.h"

struct log
{
	char           *file;
	int             fd;
	int             rotate_size;
	int             level;
	sem_t          *l_write;
};

int             log_need_rotate(log_t * obj);
int             log_rotate(log_t * obj);

log_t          *
log_open(const char *file)
{
	log_t          *obj = NULL;
	char           *tmp = NULL;
	char           *fname = NULL;

	if (strtool_isempty(file))
	{
		goto ERROR;
	}
	obj = (log_t *) calloc(1, sizeof(log_t));

	if (!obj)
	{
		goto ERROR;
	}
	obj->file = NULL;
	obj->fd = -1;
	obj->rotate_size = 32767;
	obj->level = LOG_LVL_DEBUG;
	obj->l_write = NULL;

	obj->file = strdup(file);

	obj->fd = open(file, O_WRONLY | O_CREAT | O_APPEND | O_SYNC, S_IRUSR | S_IWUSR);
	if (obj->fd == -1)
	{
		goto ERROR;
	}
	fname = filetool_get_fname_from_path(file);

#ifdef	Mac
	tmp = strdup(file);
#else
	tmp = strtool_sprintf("/tmp/%s", fname);

	if ((mknod(tmp, 0666 | S_IFREG, 0) == -1) && errno != EEXIST)
	{
		goto ERROR;
	}
#endif

	obj->l_write = sem_create(tmp);
	if (!obj->l_write)
	{
		goto ERROR;
	}
	free(tmp);
	free(fname);

	return (obj);
ERROR:
	free(tmp);
	free(fname);
	log_close(obj);
	return (NULL);
}

void 
log_close(log_t * obj)
{
	if (obj)
	{
		free(obj->file);
		if (obj->fd != -1)
		{
			close(obj->fd);
		}
		sem_destroy(obj->l_write);
		free(obj);
	}
}

int 
log_set_level(log_t * obj, const int level)
{
	if (!obj)
	{
		return (0);
	}
	obj->level = level;

	return (1);
}

int 
log_write(const int level, log_t * obj, char *fmt,...)
{
	char            date[64];
	char           *tmp = NULL;
	char            dst[2048];
	va_list         ap;
	time_t          t;
	struct tm      *tm;
	int             err;

	if (!obj || strtool_isempty(fmt))
	{
		goto ERROR;
	}
	err = sem_lock(obj->l_write);
	if (err)
	{
		goto ERROR;
	}
	if (log_need_rotate(obj))
	{
		err = log_rotate(obj);
		if (err)
		{
			goto ERROR;
		}
	}
	if (level > obj->level)
	{
		goto ERROR;
	}
	memset(date, 0, sizeof(date));

	time(&t);
	tm = localtime(&t);
	if (!tm)
	{
		goto ERROR;
	}
	strftime(date, sizeof(date), "%Y/%m/%d %H:%M:%S", tm);

	tmp = strtool_sprintf("[%s] ", date);
	write(obj->fd, tmp, strlen(tmp));

	va_start(ap, fmt);
	vsprintf(dst, fmt, ap);
	write(obj->fd, dst, strlen(dst));
	va_end(ap);

	fprintf(stdout, "%s\n", dst);
	write(obj->fd, "\n", 1);

	free(tmp);

	sem_unlock(obj->l_write);
	return (0);
ERROR:
	if (obj)
	{
		sem_unlock(obj->l_write);
	}
	return (1);
}

int 
log_need_rotate(log_t * obj)
{
	struct stat     st;
	static int      cnt = 0;
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	err = fstat(obj->fd, &st);
	if (err)
	{
		goto ERROR;
	}
	if (st.st_size < obj->rotate_size)
	{
		goto ERROR;
	} else
	{
		if (cnt)
		{
			return (1);
		}
		cnt++;

		close(obj->fd);

		obj->fd = open(obj->file, O_WRONLY | O_APPEND | O_SYNC);
		if (obj->fd == -1)
		{
			goto ERROR;
		}
		if (!log_need_rotate(obj))
		{
			goto ERROR;
		}
	}

	cnt = 0;

	return (1);
ERROR:
	cnt = 0;
	return (0);
}

int 
log_rotate(log_t * obj)
{
	char            tmp1[256];
	char            tmp2[256];
	int             n, max;

	if (!obj)
	{
		goto ERROR;
	}
	close(obj->fd);

	for (max = 1; max < 10; max++)
	{
		sprintf(tmp1, "%s.%d", obj->file, max);

		if (!filetool_is_exist(tmp1))
		{
			break;
		}
	}

	if (max > 1)
	{
		if (max > 9)
		{
			max = 9;
		}
		for (n = max; n > 0; n--)
		{
			sprintf(tmp1, "%s.%d", obj->file, n - 1);
			sprintf(tmp2, "%s.%d", obj->file, n);
			rename(tmp1, tmp2);
			/* printf("[%d] %s -> %s\n", getpid(), tmp1, tmp2); */
		}
	}
	sprintf(tmp2, "%s.1", obj->file);
	rename(obj->file, tmp2);
	/* printf("[%d] %s -> %s\n", getpid(), obj->file, tmp2); */

	obj->fd = open(obj->file,
		       O_WRONLY | O_CREAT | O_APPEND | O_SYNC,
		       S_IRUSR | S_IWUSR);

	if (obj->fd == -1)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}
