require 'amrita/template'
require 'shellwords'
require 'ropt'

require 'pinkyblog/util'
require 'pinkyblog/view-context'

module PinkyBlog
	class Menu
		attr_reader :items
	
		def initialize(items = [])
			@items = items
		end
		
		def get_current_item(context)
			
			if context.path_refered_by_menu then
				current_path = Util.normalize_path(context.path_refered_by_menu)
			else
				current_path = context.request.normalized_path
			end

			if current_path == '/' then
				return @items.first
			elsif current_path[0..2] == '/x/' then
				# エイリアス
				return @items.find{|x| x.alias_path == current_path}
			else
				# パスとクエリが一致→パスのみが一致、の順で検索
				found = @items.find{|x| x.path == current_path and x.query == context.request.GET}
				if found then
					return found
				else
					return @items.find{|x| x.path == current_path}
				end
			end
		end
		
		def to_model(context)

			re = []
			@items.each_with_index do |item, i|
				next if not context.master_mode? and not item.visible_by_guest?
				next if context.master_mode? and not item.visible_by_master?
				next if context.snapshot_mode? and not item.visible_on_snapshot?

				re << item.to_model(context, i)
				if item == get_current_item(context) then
					re.last[:class] = 'menu-on'
				end
				
				# トップページの場合の処理
				if i == 0 then
					re.last[:a]['href'] = context.route_to('/')
				end
			end
			
			re
		end
		
		
		def self.parse(text)
			items = []
		
			text = text.gsub("\r\n", "\n")
			text.each_with_index do |line, number|
				if line =~ /^(.+?)\s*\|\s*(.+)\s*$/ then
					caption, cmd = $1, $2
					cmd_args = Shellwords.shellwords(cmd)
					cmd_name = cmd_args.shift
					item = MenuItem.create(caption, cmd_name, cmd_args, number + 1)
					
					items << item if item
				end
			end
			
			unless items.find{|x| x.kind_of?(MenuItem::MasterMenu)} then
				raise MenuError, 'メニュー項目の中には、かならず「管理者メニュー」（mastermenu）を含んでいる必要があります。'
			end
			
			return self.new(items)
		end
	end
	
	class MenuError < StandardError
	end

	module MenuItem
		GUEST_ONLY = :guest_only
		MASTER_ONLY = :master_only
	
		class ArgumentParseError < StandardError
			attr_accessor :text_line_number, :menu_caption
		end
	
		def self.create(caption, cmd_name, cmd_args, line_number = nil)
			case cmd_name
			when 'top'
				return Top.new(caption, cmd_args, line_number)
			when 'about'
				return About.new(caption, cmd_args, line_number)
			when 'entry'
				return Entry.new(caption, cmd_args, line_number)
			when 'entries'
				return Entries.new(caption, cmd_args, line_number)
			when 'recent'
				return Recent.new(caption, cmd_args, line_number)
			when 'view'
				return View.new(caption, cmd_args, line_number)
			when 'list'
				return EntryList.new(caption, cmd_args, line_number)
			when 'search'
				return Search.new(caption, cmd_args, line_number)
			when 'newsfeed'
				return NewsFeed.new(caption, cmd_args, line_number)
			when 'login'
				return Login.new(caption, cmd_args, line_number)
			when 'logout'
				return Logout.new(caption, cmd_args, line_number)
			when 'master', 'mastermenu'
				return MasterMenu.new(caption, cmd_args, line_number)
			when 'addentry', 'write'
				return AddEntry.new(caption, cmd_args, line_number)
			when 'url', 'uri'
				return URI.new(caption, cmd_args, line_number)
			else
				return nil
			end
		end
	
	
		class Base
			attr_reader :caption, :visibility
			def initialize(caption, args = [], line_number = nil)
				@caption = caption
				@visibility = nil
				@accesskey = nil
				@line_number = line_number
				
				re = ROpt.parse(args, short_option_spec, *long_option_spec)
				if re then
					on_option_parsed(re)
				else
					raise_parse_error
				end
			end
			
			def path
				alias_path || normal_path
			end
			
			def normal_path
			end
			
			def alias_path
				(@alias ? "/x/#{@alias}" : nil)
			end
			
			def query
				{}
			end
			
			def query_string
				if @alias then
					''
				else
					real_query_string
				end
			end
			
			# エイリアスURIでは表示されない
			def real_query_string
				query.to_a.map{|k, v| "#{k}=#{v}"}.join('&')
			end
			
			def visible_by_guest?
				visibility != MASTER_ONLY
			end
			
			def visible_by_master?
				visibility != GUEST_ONLY
			end
			
			def visible_on_snapshot?
				false
			end
			
			def short_option_spec
				'mgk:'
			end
			
			def long_option_spec
				['master-only', 'guest-only', 'accesskey:', 'alias:']
			end
			
			def on_option_parsed(re)
				if re['master-only'] || re['m'] then
					@visibility = MASTER_ONLY
				elsif re['guest-only'] || re['g'] then
					@visibility = GUEST_ONLY
				end
				
				@accesskey = re['accesskey'] || re['k']
				
				@alias = re['alias']
				if @alias then
					unless @alias =~ /^[0-9a-zA-Z._!$&'()*+,;=~-]+$/ then 
						raise_parse_error
					end
				end
			end
			
			
			
			
			def to_model(context, index)
				model = {}
				
				model[:a] = Amrita.a({:href => Amrita::SanitizedString.new(Util.escape_html(build_href(context))), :accesskey => @accesskey}){@caption}
				model[:id] = sprintf("MENU%02d", index + 1)
				return model
			end
			
			def to_pan
				[path, @caption]
			end
			
			def build_href(context)
				qs = query_string
				context.route_to(path, (qs.empty? ? nil : qs)).to_s
			end
			
			private
			def raise_parse_error
				error = ArgumentParseError.new((@line_number ? "illegal menu definition, on line #{@line_number}" : ""))
				error.text_line_number = @line_number
				error.menu_caption = @caption
				raise error
			end
			

			
			
		end
		
		module TagOption
			def short_option_spec
				super << 't::x::'
			end
			
			def long_option_spec
				super << 'tag::' << 'exclude-tag::'
			end
			
			def on_option_parsed(re)
				super
				@tags = re['tag'] + re['t']
				@excluded_tags = re['exclude-tag'] + re['x']
			end
			
			def query
				re = super
				@tags.each_with_index do |tag, i|
					re["tags_#{i}"] = Util.encode_base64url(tag)
				end

				@excluded_tags.each_with_index do |tag, i|
					re["extags_#{i}"] = Util.encode_base64url(tag)
				end

				
				re
			end

		end
		
		module PageOption
			def long_option_spec
				super << 'page-menu-position:'
			end
			
			def on_option_parsed(re)
				super
				@page_menu_position = re['page-menu-position']
				case @page_menu_position
				when 'bottom', nil
				else
					raise_parse_error
				end
			end
			
			def query
				re = super
				if @page_menu_position then
					re['page_menu_position'] = @page_menu_position
				end
				
				re
			end

		end

		
		class Top < Base
			def normal_path
				'/top'
			end
			
			def visible_on_snapshot?
				true
			end
		end

		class About < Base
			def normal_path
				'/about'
			end
			
			def visible_on_snapshot?
				true
			end
		end
		
		class Entry < Base
			def short_option_spec
				super << 's'
			end
			
			def long_option_spec
				super << 'simple'
			end
			
			def on_option_parsed(re)
				super
				@simple = re['simple'] || re['s']
				@entry_id = re.args.first
				raise ArgumentParseError unless @entry_id
				raise_parse_error unless Util.validate_entry_id(@entry_id)
			end
		
			def normal_path
				"/entries/#{@entry_id}"
			end
			
			def query
				if @simple then
					super.merge({'simple' => '1'})
				else
					super
				end
			end
			
			def visible_on_snapshot?
				true
			end

		end
		
		class Entries < Base
			def on_option_parsed(re)
				super
				@entry_ids = re.args
				if @entry_ids.find{|x| !(Util.validate_entry_id(x))} then
					raise_parse_error
				end
			end
		
			def normal_path
				"/entries/#{@entry_ids.join(';')}"
			end
			
			def visible_on_snapshot?
				true
			end

		end


		
		class Recent < Base
			include TagOption
			include PageOption
			
			def short_option_spec
				super << 'n:'
			end
			
			def long_option_spec
				super << 'number:' << 'top-entry:'# << 'updated-range:' << 'created-range:'
			end
			
			def on_option_parsed(re)
				super
				n = re['number'] || re['n']
				@number = (n ? n.to_i : nil)
				@top_entry = re['top-entry']
				
			end
			
			def query
				re = super
				re['number'] = @number.to_s if @number
				re['top_entry_id'] = @top_entry if @top_entry
				re
			end

		
			def normal_path
				'/recent'
			end
		end
		
		class View < Base
			include PageOption
		
			def short_option_spec
				super << 'n:s:'
			end
			
			def long_option_spec
				super << 'number:' << 'sort:' << 'top-entry:'
			end
			
			def on_option_parsed(re)
				super
				n = re['number'] || re['n']
				@number = (n ? n.to_i : nil)
				@sort = re['sort'] || re['s'] || Sort::BY_MODIFIED
				@top_entry = re['top-entry']
				
				re.args.each do |cond_str|
					unless ViewCondition.parse(cond_str) then
						raise_option_parse_error
					end
				end
				@condition = re.args.join(' ')
				
			end
			
			def query
				re = super
				re['number'] = @number if @number
				re['sort'] = @sort if @sort
				re['top_entry_id'] = @top_entry if @top_entry
				re['condition'] = Util.encode_base64url(@condition)
				
				re
			end

		
			def normal_path
				'/view'
			end
			
		end

		
		class EntryList < Base
			include PageOption

			def normal_path
				'/entries'
			end
			
			
			def short_option_spec
				super << 's:r'
			end
			
			def long_option_spec
				super << 'sort-by:' << 'reverse'
			end
			
			def on_option_parsed(re)
				super
				@sort = re['sort-by'] || re['s']
				@reverse = true if re['reverse'] or re['r']
			end
			
			def query
				re = super
				re['sort'] = @sort if @sort
				re['order'] = Order::REVERSE if @reverse
				
				re
			end

			def visible_on_snapshot?
				true
			end

		end



		class Search < Base
			def normal_path
				'/search'
			end
		end

		class NewsFeed < Base
			def normal_path
				'/news_feed'
			end
		end

		class AddEntry < Base
			def normal_path
				'/master_menu/entry_add_form'
			end
			
			def visibility
				MASTER_ONLY
			end
			
			def short_option_spec
				super << 't::'
			end
			
			def long_option_spec
				super << 'tag::'
			end
			
			def on_option_parsed(re)
				super
				@tags = re['tag'] + re['t']
			end
			
			def query
				re = super
				@tags.each_with_index do |tag, i|
					re["tags_#{i}"] ||= []
					re["tags_#{i}"] << Util.encode_base64url(tag)
				end
				
				re
			end
		end

		
		class Login < Base
			def normal_path
				'/login'
			end
			
			def visibility
				GUEST_ONLY
			end
		end

		class Logout < Base
			def normal_path
				'/'
			end
			
			def query
				super.merge('logout' => '1')
			end
			
			def visibility
				MASTER_ONLY
			end
		end

		
		class MasterMenu < Base
			def normal_path
				'/master_menu'
			end
			
			def visibility
				MASTER_ONLY
			end
		end

		class URI < Base
			def on_option_parsed(re)
				super
				@uri = re.args.first
				raise ArgumentParseError unless @uri
			end
			
			def build_href(context)
				@uri
			end
		end


	end


	module ViewCondition
		PATTERN = %r{
			^(\!)?          # $1 = excluding symbol
			(ut|ct|tag|w)    # $2 = type
			\:
			(.+)             # $3 = parameter
		}x
		
		TIME_RANGE_PATTERN = /^([0-9][0-9-]*)?\.{2,}([0-9][0-9-]*)?$/
		TIME_PATTERN = /^([0-9][0-9-]*)$/
		
		def self.parse(str)
			if str =~ PATTERN then
				excluding = ($1 ? true : false)
				type = $2.downcase
				parameter = $3
				
				case type
				when 'ut'
					re = Updated.new(*parse_time(parameter))
				when 'ct'
					re = Created.new(*parse_time(parameter))
				when 'tag'
					re = Tag.new(parameter)
				when 'w'
					re = Word.new(parameter)
				else
					raise "unknown condition type - #{$2.inspect}"
				end
				
				re.excluding = excluding
				
				
				re
			else
				return nil
			end
		end
		
		def self.parse_time(str)
			if str =~ TIME_RANGE_PATTERN then
				left_str = $1; right_str = $2
			elsif str =~ TIME_PATTERN then
				left_str = $1; right_str = $1
			else
				raise "time parse failed - #{str.inspect}"
			end
			
			items = left_str.split(/\-/)
			items.map!{|x| x.to_i}
			left = Time.local(*items)
			
			items = right_str.split(/\-/)
			items.map!{|x| x.to_i}
			
			# 期間の終わりを表すため、日付最後の要素に+1、そして時刻から-1秒
			# （2009-03 → 2009-04 → 2009-03-31 23:59:59）
			items[-1] += 1
			right = Time.local(*items) - 1
			
			return [left, right]
		end
		
		class Base
			bool_attr_accessor :excluding
			
			public
			def include?(entry)
				if excluding? then
					!(basic_check(entry))
				else
					basic_check(entry)
				end
			end
			
			private
			def basic_check(entry)
			end
		end
		
		module TimeRangeCondition
			def initialize(left, right)
				@left = left
				@right = right
			end
		end
		
		class Updated < Base
			include TimeRangeCondition
			
			def basic_check(entry)
				entry.updated >= @left and entry.updated <= @right
			end
		end
		
		class Created < Base
			include TimeRangeCondition
			
			def basic_check(entry)
				entry.created >= @left and entry.created <= @right
			end
		end
		
		class Tag < Base
			def initialize(body)
				@body = body
			end

			def basic_check(entry)
				entry.tags.include?(@body)
			end
		end
		
		class Word < Base
			def initialize(body)
				@pattern = Regexp.new(Regexp.escape(body), Regexp::IGNORECASE)
			end
			
			def basic_check(entry)
				entry.body =~ @pattern
			end
		end
	end
end
