#-> Screen, Repository
require 'amrita/template'
require 'cgisup'
require 'pathname'
require 'html-gen'

require 'pinkyblog/const'
require 'pinkyblog/function'
require 'pinkyblog/menu-item'
require 'pinkyblog/section'
require 'pinkyblog/view-context'
require 'pinkyblog/option-check'

module PinkyBlog
	# ブラウザに表示する画面をあらわす抽象クラス
	class Screen
		include Amrita::ExpandByMember
		self.extend OptionCheck
		attr_reader :context
		

		
		def initialize(context, option_args = {})
			@context = context
			# OptionCheck
			self.class.assert_fulfill_requirement(option_args)
			self.class.set_defaults(option_args)
			
			option_args.each_pair do |key, value|
				instance_variable_set("@#{key}", value)
			end
		end
		
		def gen_sect(cls, opts = {})
			return cls.new(@context, opts)
		end
		
		def config
			@context.config
		end
		
		def module_handler
			@context.module_handler
		end
		
		def request
			@context.request
		end
		
		def master_mode?
			@context.master_mode?
		end
		
		def snapshot_mode?
			@context.snapshot_mode?
		end
		
		def menu_items
			items = []
			items << MenuItem.new(@context, '/', 'blog_top', 'トップページ')
			items << MenuItem.new(@context, '/about', 'about', 'このblogについて') if config.about_visible?
			items << MenuItem.new(@context, '/recent', 'recent_entries', '最近更新された記事') if config.recent_entries_visible? and not snapshot_mode?
			items << MenuItem.new(@context, '/entries', 'entry_list', '記事一覧')
			items << MenuItem.new(@context, '/search', 'search', '記事検索') unless snapshot_mode?
			items << MenuItem.new(@context, '/news_feed', 'news_feed', 'ニュースフィード配信') if config.news_feed_visible? and not snapshot_mode?
			items << MenuItem.new(@context, '/master_menu', 'master_menu', '管理者メニュー') if master_mode?
			return items
		end
		
		def menu
			model = []
			items = menu_items
			items.each_with_index do |item, i|
				model << item.to_model(i, check_current_menu(item))
			end
			return model
		end
		
		def check_current_menu(item)
			item.current_menu?(@context.request)
		end
		
		
		def site_title
			config.site_title
		end
		
		def br
			" "
		end
		
		def pan
			items = []
			
			
			data = pan_data
			data.last[0] = nil
			data.each do |path, caption|
				items << (path ? HTML::Tag.new('a', {'href' => @context.route_to(path)}, caption) : caption)
				items << "\n"
			end
			
			if config.home_url then
				items.unshift("\n")
				items.unshift(HTML::Tag.new('a', {'href' => config.home_url}, 'home'))
			end

			
			
			return {:items => items.map{|x| Amrita::SanitizedString.new(x.to_s)}}
		end
		
		def pan_data
			data = []
			data << ['/', config.menu_captions['blog_top']]
			
			return data
		end
		
		
		def body_class
			(master_mode? ? 'for-master' : 'for-guest')
		end
		
		def up_href
			data = pan_data
			if data.size >= 2 then
				return @context.route_to(data[-2][0])
			end
		
		end
		
		def base_css_link
			if snapshot_mode? then
				path = Pathname.new('files/res/pinkyblog/base.css')
				Amrita.a({:href => path.relative_path_from(@context.snapshot_path.dirname).to_s})
			else
			

				uri = request.file_route_to(config.res_dir_path + 'pinkyblog/base.css')
				Amrita.a({:href => uri})
			end
		end

		
		def cdp_link
			if snapshot_mode? then
				path = Pathname.new('files/csstemplate/style.css')
				Amrita.a({:href => path.relative_path_from(@context.snapshot_path.dirname).to_s})

			else
				if request.get_cookie('cdp_name') then
					file_path = config.cdp_dir_path + request.get_cookie('cdp_name').slice(/[a-zA-Z0-9_-]+/) + 'csstemplate/style.css'
				else
	 				file_path = config.cdp_dir_path + 'style.css'
				end
				uri = request.file_route_to(file_path)
				Amrita.a({:href => uri})
				
			end
		end
		
		def print_cdp_link
			if snapshot_mode? then
				path = Pathname.new('files/res/pinkyblog/print.css')
				Amrita.a({:href => path.relative_path_from(@context.snapshot_path.dirname).to_s})

			else
				uri = request.file_route_to(config.res_dir_path + 'pinkyblog/print.css')
				Amrita.a({:href => uri})
				
			end
		end

		
		
		
		
		def atom_link
			Amrita.a({:href => @context.get_feed_url('created.xml')}) unless snapshot_mode?
		end

		

		def links
			models = []
			models << {:rel => 'index', :href => @context.route_to('entries')}
			models << {:rel => 'home', :href => config.home_url} if config.home_url
			models << {:rel => 'search', :href => @context.route_to('search')}
			if up_href then
				models << {:rel => 'up', :href => up_href}
			end
			
			return models
		end
		
		def page_footer
			model = {}
			model[:top_href] = '#PAGETOP'
			model[:address_items] = []
			index = 2
			
			if config.writer_address then
				model[:address_items] << {:id => sprintf("FOOTER%02d", index), :caption => 'mail', :title => '執筆者にメールを送る',
				                          :href => PinkyBlog.get_html_entity("mailto:#{config.writer_address}")}
				index += 1
			end
			
			model[:address_items] << {:id => sprintf("FOOTER%02d", index), :caption => 'CSS Designplate',
			                          :title => 'CSS着せ替えテンプレートプロジェクト', :href => 'http://css-designplate.info/'}
			index += 1
			
			model[:address_items] << {:id => sprintf("FOOTER%02d", index), :caption => 'Pinky:blog',
			                          :title => 'このblogツールの配布サイト', :href => 'http://scl.littlestar.jp/pinkyblog/'}

			
			return model
		end
		
		
		def sections
			[]
		end
		
		def kizi
			sects = []
			unless @context.warnings.empty? then
				sects << gen_sect(WarningSection)
			end
			
			if @context.message then
				sects << gen_sect(MessageSection)
			end
			
			sects << self.sections
			
			if @context.master_mode? and request.get('debug') then
				sects << gen_sect(DebugInformationSection)
			end
			
			return sects
		end
		
		def http_status
			HTTP_OK
		end
		
		
		def to_s
			path = config.lib_dir_path + 'pinkyblog/template/main.html'
			text = path.read.untaint
			tmpl = Amrita::TemplateText.new(text)
			tmpl.amrita_id = 'amrita_src'
			buf = ""
			tmpl.expand(buf, self)

			return @context.parse_html(buf)

		end
		
		def cookies
			# メッセージ表示の解除（クッキーを削除する）
			eater = CGI::Cookie.new('message', '')
			eater.expires = Time.now - 30
			return (@cookies || []) + [eater]
		end
		
		def headers
			re = {}
			re['Content-Type'] = 'text/html'
			re['status'] = http_status
			re['cookie'] = self.cookies
			
			re
		end
		
		def out(cgi)
			h = self.headers
			h['status'] = http_status
			cgi.out(h){self.to_s}
			return nil
		end
		
		def snapshot(root_path)
			path = root_path + @context.snapshot_path
			path.untaint
			FileUtils.mkdir_p(path.dirname)
			path.open('w'){|f|
				f.write(self.to_s)
			}
			return nil
		end
		

		
	end
	
=begin
	# 出力キャッシュ。HTML自体を内容に持つ
	class Cache
		def initialize(html)
			@html = html
		end
	
		def out(cgi)
			headers = {}
			headers['type'] = 'text/html'
			headers['status'] = HTTTP_OK
			
			cgi.out(headers){@html}
			return nil
		end
	end
=end
	
	
	class ErrorScreen < Screen
		require_opt :message
		attr_reader :message
		default_opt :http_status, HTTP_BAD_REQUEST
		
		def page_title
			return "#{@http_status} - #{config.site_title}"
		end
		
		def http_status
			@http_status
		end
		
		def sections
			[gen_sect(ErrorSection, {:body => @message, :http_status => @http_status})]
		end
		
	end
	
	class PostBlockScreen < Screen
		def page_title
			return "#{HTTP_BAD_REQUEST} - #{config.site_title}"
		end
		
		def http_status
			HTTP_BAD_REQUEST
		end
		
		def sections
			[gen_sect(PostBlockSection)]
		end
	end

	
	class ForbiddenScreen < Screen
		def page_title
			return "#{@http_status} - " + config.site_title
		end
		
		def sections
			body = "あなたは管理者としてログインした状態にありません。
			        正常にログインできない場合、ブラウザのCookie機能が有効になっているかどうか
			        確認してください。"
			
			return [gen_sect(ErrorSection, {:body => body, :http_status => HTTP_FORBIDDEN})]
			
		end
		
		def http_status
			HTTP_FORBIDDEN
		end
	end
	
	class TopScreen < Screen
		require_opt :recent_entries, :welcome_entry
	
		def page_title
			return config.site_title
		end
		
		def check_current_menu(item)
			item.path == '/'
		end

		
		def sections
			sections = super
			
			if master_mode? || @welcome_entry.visible? then
				sections << gen_sect(EntrySection, {:entry => @welcome_entry})
			end
			
			sections << gen_sect(HeadlineSection, {:entries => @recent_entries})
			sections << gen_sect(MessageFormSection) if !(snapshot_mode?) && config.message_form_visible
			return sections
		end


	end
	
	class AboutScreen < Screen
		require_opt :about_blog_entry, :about_writer_entry
	
		def page_title
			return "#{config.menu_captions['about']} - #{config.site_title}"
		end
		

		
		def sections
			sections = super
			
			if master_mode? || @about_blog_entry.visible? then
				sections << gen_sect(EntrySection, {:entry => @about_blog_entry}) 
			end
			
			if master_mode? || @about_writer_entry.visible? then
				sections << gen_sect(EntrySection, {:entry => @about_writer_entry})
			end
			return sections
		end
		
		def pan_data
			data = super
			data << ['about', config.menu_captions['about']]
			return data
		end
	end
	
	class MessageResponseScreen < Screen
		require_opt :entry
		
		def page_title
			return "ひとことレス - #{config.site_title}"
		end
		
		def sections
			return [gen_sect(EntrySection, {:entry => @entry, :ex_footer_visible => false})]
		end

		def pan_data
			data = super
			data << ['message_response', 'ひとことレス']
			return data
		end

	end

	
	class EntryScreen < Screen
		require_opt :entry, :referer_list
	
		
		def page_title
			return "#{@entry.title} - #{config.site_title}"
		end

		
		def sections
			sections = super
			
			
			sections << gen_sect(EntrySection, {:entry => @entry, :ex_footer_visible => false})
			if @entry.commentable? then
				sections << gen_sect(CommentSection, {:comments => @entry.comments}) unless @entry.comments.empty?
				
				opts = {}
				opts[:entry_id] = @entry.id
				opts[:default_name] = request.get_cookie('default_name')
				opts[:default_address] = request.get_cookie('default_address')
				sections << gen_sect(CommentFormSection, opts) unless snapshot_mode?
			end
			sections << gen_sect(EntryInformationSection, {:entry => @entry, :referer_list => @referer_list})
			
			return sections
		end
		
		def pan_data
			data = super
			data << [@entry.id, @entry.title]
			return data
		end



		
	end
	
	class EntryEditScreen < Screen
		require_opt :entry, :tag_list
		default_opt :parameters, nil
		
		def page_title
			if @entry.title.empty? then
				"記事の編集 - #{config.site_title}"
			else
				"#{@entry.title}の編集 - #{config.site_title}"
			end
		end

		
		def sections
			sections = super
			if @parameters then
				sections <<	gen_sect(PreviewSection, {:content => @parameters[:content], :format => @parameters[:format]})
			end
			opts = {:entry => @entry, :tag_list => @tag_list, :parameters => @parameters}
			sections << gen_sect(EntryEditFormSection, opts)
			return sections
		end
		
		def pan_data
			data = super
			data << ["entries/#{@entry.id}", @entry.title]
			data << ["entries/#{@entry.id}/edit_form", "編集"]
			return data
		end
	end
	
	class EntryAddScreen < Screen
		require_opt :tag_list
		default_opt :parameters, nil

		
		def page_title
			return "新しい記事の作成 - #{config.site_title}"
		end

		
		def sections
			sections = super
			if @parameters then
				sections <<	gen_sect(PreviewSection, {:content => @parameters[:content], :format => @parameters[:format]})
			end
			opts = {:entry => nil, :tag_list => @tag_list, :parameters => @parameters}
			sections << gen_sect(EntryEditFormSection, opts)
			return sections
		end
		
		def pan_data
			data = super
			data << ['master_menu', "管理者メニュー"]
			data << ["master_menu/entry_add_form", "新しい記事の作成"]
			return data
		end
	end

	
	class FormatGuideScreen < Screen
		def page_title
			return "記事の書き方について - #{config.site_title}"
		end

		
		def sections
			sections = super
			sections <<	gen_sect(FormatGuideSection)
			return sections
		end
		
		def pan_data
			data = super
			data << ['format_guide', "記事の書き方について"]
			return data
		end
	end
	
	class FormatDetailScreen < Screen
		def page_title
			
			return "#{module_handler.translator_modules[format_name]::CAPTION} - #{config.site_title}"
		end
		
		def format_name
			request.path_items[1]
		end

		
		def sections
			sections = super
			sections <<	gen_sect(FormatDetailSection)
			#sections <<	gen_sect(FormatDetailSourceSection)
			return sections
		end
		
		def pan_data
			data = super
			data << ['format_guide', "記事の書き方について"]
			data << ["format_guide/#{format_name}", module_handler.translator_modules[format_name]::CAPTION]
			return data
		end
	end

	
	class PagingScreen < Screen
		def add_navigation(sections)
		
			
			opts = {:prev_href => prev_href, :prev_caption => prev_caption,
			        :next_href => nil, :next_caption => nil}
			sections.unshift(gen_sect(NavigationSection, opts)) if prev_href
			opts = {:prev_href => nil, :prev_caption => nil,
			        :next_href => next_href, :next_caption => next_caption}
			sections.push(gen_sect(NavigationSection, opts)) if next_href
			return sections
		end
		
		def start
			@start || request.start
		end
		
		def prev_caption
			number = [page_length, start].min
			"&lt;&lt;新しい#{number}記事"
		end
		
		def next_caption
			number = [page_length, total - start - page_length].min
			"過去の#{number}記事&gt;&gt;"
		end
		
		def extend_query
			nil
		end

		
		def prev_href
			if start > 0 then
				prev_start = start - page_length
				prev_start = 0 if prev_start < 0
				if snapshot_mode? then
					return @context.get_cgi_path(sprintf("%s_st%05d", paging_base_path, prev_start))
				else
					query = "start=#{prev_start}"
					query << '&' << extend_query if extend_query
					return @context.get_cgi_path(paging_base_path, query)
				end
			else
				return nil
			end
			
		end
		
		def next_href
			
			if (start + page_length) < total then
				next_start = start + page_length
				if snapshot_mode? then
					return @context.get_cgi_path(sprintf("%s_st%05d", paging_base_path, next_start))
				else
					query = "start=#{next_start}"
					query << '&' << extend_query if extend_query
					return @context.get_cgi_path(paging_base_path, query)
				end
			else
				return nil
			end
			
		end
		
		
		def page_range
			start...(start + page_length)
		end
		
		def get_page_pan
			last = [page_range.last, total].min
			caption = "#{page_range.first + 1} - #{last} (全 #{total} 記事)"
			return [paging_base_path, caption]
		end
		

		
		def links
			models = super
			models << {:rel => 'prev', :href => prev_href} if prev_href
			models << {:rel => 'next', :href => next_href} if next_href
			
			return models
		end

		
	end
	
	class EntryListScreen < PagingScreen
		require_opt :entries
		default_opt :tag_list, {}
		default_opt :access_counts, {}
	
		def initialize(context, opts = {})
			super
			
			# タグ絞込み
			unless request.tags.empty? then
				@entries.delete_if{|x| (request.tags - x.tags).size >= 1}
			end

		end
		
		def paging_base_path
			'entries'
		end
		
		
		def page_length
			ENTRY_LIST_PAGE_LENGTH
		end
		
		def total
			@entries.size
		end
		

		
		def page_title
			return "#{config.menu_captions['entry_list']} - #{config.site_title}"
		end
		
		def extend_query
			query = "sort=#{request.sort}"
			unless request.tags.empty? then
				query << '&' << request.tags.map{|x| "tags=" + PinkyBlog.encode_tag(x)}.join('&')
			end
			return query
		end
		
		def sections
			sections = super
			
			@entries = @entries.dup
			# タグ絞込み
			unless request.tags.empty? then
				@entries.all?{|x| (request.tags - x.tags).empty?}
			end

			case request.sort
			when BY_CREATED
				@entries.sort!{|a, b| b.created <=> a.created}
			when BY_ACCESS
				@entries.sort!{|a, b| (@access_counts[b.id] || 0) <=> (@access_counts[a.id] || 0)}
			when BY_MODIFIED
				@entries.sort!{|a, b| b.last_modified <=> a.last_modified}
			end

			
			
			sections << gen_sect(TagListSection, {:tag_list => @tag_list}) unless snapshot_mode?
			sections << gen_sect(EntryListSection,
			                     {:entries => @entries[page_range], :access_counts => @access_counts})
			add_navigation(sections)
			
			return sections
		end
		
		def pan_data
			data = super
			data << ['entries', config.menu_captions['entry_list']]
			data << get_page_pan
			return data.compact
		end
		
		def up_href
			@context.route_to('/')
		end
		


	end
	
	class RecentScreen < PagingScreen
		require_opt :entries
		
		def paging_base_path
			'recent'
		end
		
		
		
		def page_length
			RECENT_ENTRY_PAGE_LENGTH
		end
		
		def total
			@entries.size
		end
		
		def page_title
			return "#{config.menu_captions['recent_entries']} - #{config.site_title}"
		end
		
		def sections
			sections = super
			

			@entries.sort!{|a, b| b.last_modified <=> a.last_modified}
			
			@entries[page_range].each do |entry|
				sections << gen_sect(EntrySection, {:entry => entry})
			end
			add_navigation(sections)
			
			return sections
		end
		
		def pan_data
			data = super
			data << ['recent', config.menu_captions['recent_entries']]
			data << get_page_pan
			return data.compact
		end
		
		def up_href
			@context.route_to('/')
		end

	end


	class SearchScreen < Screen
		require_opt :keywords, :hit_list
		
		def page_title
			return "#{config.menu_captions['search']} - #{config.site_title}"
		end
		
		def sections
			sections = super
			sections << gen_sect(SearchFormSection, {:keywords => @keywords})
			unless @keywords.empty? then
				sections << gen_sect(SearchResultSection, {:keywords => @keywords, :hit_list => @hit_list}) 
			end
			return sections
		end
		
		def pan_data
			data = super
			data << ['search', config.menu_captions['search']]
			return data
		end
		
	end
	
	class NewsFeedScreen < Screen
		def page_title
			return "#{config.menu_captions['news_feed']} - #{config.site_title}"
		end
		
		def sections
			sections = super
			sections << gen_sect(NewsFeedSection)
			return sections
		end
		
		def pan_data
			data = super
			data << ['news_feed', config.menu_captions['news_feed']]
			return data
		end
		
	end


	class SystemInformationScreen < Screen
		def page_title
			return "システム情報 - #{config.site_title}"
		end
		
		def sections
			sects = super
			sects << gen_sect(SystemInformationSection)
			return sects
		end
		
		def pan_data
			data = super
			data << ['/system', 'システム情報']
			return data
		end
	end

	class ENVScreen < Screen
		def page_title
			return "環境変数 - #{config.site_title}"
		end
		
		def sections
			sects = super
			sects << gen_sect(ENVSection)
			return sects
		end
		
		def pan_data
			data = super
			data << ['master_menu', '管理者メニュー']
			data << ['master_menu/env', '環境変数']
			return data
		end
	end



	class LoginFormScreen < Screen
		def page_title
			return "ログインフォーム - #{config.site_title}"
		end
		
		def sections
			
			return super + [gen_sect(LoginFormSection)]
		end
		
		def pan_data
			data = super
			data << ['login_form', 'ログインフォーム']
			return data
		end
	end

	class MasterMenuScreen < Screen
		require_opt :notifications
	
		
		def page_title
			return "#{config.menu_captions['master_menu']} - #{config.site_title}"
		end
		
		def sections
			sects = super
			sects << gen_sect(NotifyingSection, {:notifications => @notifications}) unless @notifications.empty?
			sects << gen_sect(MasterMenuSection)
			sects << gen_sect(SystemInformationSection)
			return sects
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			return data
		end
	end
	
	class BlogConfigScreen < Screen
		
		def page_title
			return "blog設定 - #{config.site_title}"
		end
		
		def sections
			return super + [gen_sect(BlogConfigFormSection)]
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/blog_config', 'blog設定']
			return data
		end
	end
	
	class PostLimitScreen < Screen
		
		def page_title
			return "投稿制限 - #{config.site_title}"
		end
		
		def sections
			return super + [gen_sect(PostLimitFormSection)]
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/post_limit', '投稿制限']
			return data
		end
	end



	class RefererConfigScreen < Screen
		require_opt :table, :table_text
	
		
		def page_title
			return "リファラ設定 - #{config.site_title}"
		end
		
		def sections
			sections = super
			sections << gen_sect(RefererTableSection, {:table => @table})
			sections << gen_sect(RefererTableEditFormSection, {:table_text => @table_text})
			return sections
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/referer_config', 'リファラ設定']
			return data
		end
	end
	
	class MessageListScreen < Screen
		require_opt :messages
		
		def page_title
			return "ひとことメッセージ一覧 - #{config.site_title}"
		end
		
		def sections
			sects = super
			sects << gen_sect(MessageListSection, {:messages => @messages})
			return sects
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/message_list', 'ひとことメッセージ一覧']
			return data
		end
		

	end
	

	
	
	
	class EntryManagerScreen < Screen
		require_opt :entries, :access_record, :file_data
		
		def page_title
			return "記事の管理・一括操作 - #{config.site_title}"
		end
		
		def sections
			sections = super
			opts = {:entries => @entries, :access_record => @access_record, :file_data => @file_data}
			sections << gen_sect(EntryManagerSection, opts)
			return sections
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/entry_manager', '記事の管理・一括操作']
			return data
		end
	end
	

	
	class EntryDeleteConfirmationScreen < Screen
		require_opt :entries, :access_record, :file_data

		def page_title
			return "記事の削除確認 - #{config.site_title}"
		end
		
		def sections
			sections = super
			opts = {:entries => @entries, :access_record => @access_record, :file_data => @file_data}
			sections << gen_sect(EntryDeleteConfirmationSection, opts)
			return sections
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/entry_manager', '記事の管理・一括操作']
			data << ['master_menu/entry_manager', '削除確認']
			return data
		end
	end

	class SnapshotScreen < Screen


		def page_title
			return "スナップショット - #{config.site_title}"
		end
		
		def sections
			[gen_sect(SnapshotSection, {})]
		end
		
		def pan_data
			data = super
			data << ['master_menu', config.menu_captions['master_menu']]
			data << ['master_menu/snapshot', 'スナップショット']
			return data
		end
	end


	# リダイレクションを表す抽象クラス。Screenと同様にoutメソッドで出力
	class Redirector
		self.extend OptionCheck
		
		require_opt :path
		attr_reader :path
		default_opt :message, nil
		
		def initialize(context, option_args = {})
			@context = context
			# OptionCheck
			self.class.assert_fulfill_requirement(option_args)
			self.class.set_defaults(option_args)
			
			option_args.each_pair do |key, value|
				instance_variable_set("@#{key}", value)
			end
		end
		
		def http_status
			HTTP_SEE_OTHER
		end
		
		def headers
			re = {}
			re['Location'] = location

			re
		end
		
		def location
			@context.absolute_uri_to(@path)
		end
		
		def to_s
			%Q|<html><body>go to url: <a href="#{location}">#{location}</a></body></html>|
		end
		
		def cookies
			re = @cookies.dup
			if @message then
				cookie = CGI::Cookie.new("message", @message)
				re << cookie
			end
			
			re
		end
		
		def out(cgi)
			headers = self.headers
			headers['status'] = http_status
			
			query_pieces = []
			
			headers['cookie'] = self.cookies
			cgi.out(headers){self.to_s}
			return nil
		end
	end
	
	
=begin
	class AtomResponse < Screen
		
		def content_type
			'application/atom+xml'
		end

		def headers
			re = super
			re['Content-Type'] = content_type
			
			re
		end
		
		def cookies
			[]
		end
		
		def to_s			
			re = ''
			REXML::Formatters::Default.new.write(self.get_xml, re)
			
			re

		end
		
		private
		def entry_to_atom_entry(entry)
			re = Atom::Entry.new
			re.id = entry.id
			re.title = entry.title
			re.updated = entry.updated
			re.published = entry.published
			re.links << Atom::Link.new
			re.links.last['href'] = @context.absolute_uri_to("/entries/#{entry.id}")
			
			re
			
		end


	end

	class AtomService < AtomResponse
		def content_type
			'application/atomsvc+xml'
		end
		
		def get_xml
			svc = Atom::Service.new
			ws = Atom::Workspace.new(:title => config.site_title)

			cl = Atom::Collection.new(@context.absolute_uri_to('/app/version'))
			cl.title = 'Version Information'
			ws.collections << cl
			
			
			cl = Atom::Collection.new(@context.absolute_uri_to('/app/entries'))
			cl.title = 'All entries'
			ws.collections << cl
			
			svc.workspaces << ws
			
			return svc.to_xml
		end
		
	end
	
	class AtomEntryCollection < AtomResponse
		def initialize(context, entries)
			super(context)
			@entries = entries
		end
	
		def get_xml
			feed = Atom::Feed.new
			feed.id = @context.request.self_uri
			feed.title = config.site_title
			feed.updated = Time.now
			
			feed.entries = @entries.map{|x| entry_to_atom_entry(x)}
			
			return feed.to_xml
		end
	end
	
	class AtomEntry < AtomResponse
		def initialize(context, entry)
			super(context)
			@entry = entry
		end
	
		def get_xml
			return entry_to_atom_entry(@entry).to_xml
		end
	end

=end

	
end


