#->Request 
require 'cgisup'
require 'pathname'
require 'uri'


require 'pinkyblog/function'
require 'pinkyblog/error'


module PinkyBlog

	class Request
		attr_reader :script_uri, :path_info, :query_string, :cookies
		attr_reader :path_items, :parameters, :referer_url, :remote_address
		alias remote_addr remote_address
		alias script_url script_uri
		alias params parameters
	
		def initialize(script_uri, path_info = nil, query = nil, options = {})
			case script_uri
			when URI::HTTP
				@script_uri = script_uri
			else
				@script_uri = URI.parse(script_uri.to_s)
			end
			
			@path_info = path_info || ''
			@path_items = @path_info.split('/')
			@path_items.shift if @path_items.first and @path_items.first.empty?
			
			

			case query
			when Hash
				@query_string = ''
				@parameters = {}
				query.each_pair do |k, v|
					@parameters[k] = (v.is_a?(Array) ? v : [v])
				end
			else
				@query_string = (query || '')
				@parameters = CGI.parse(@query_string)
			end
			
			@cookies = options['cookies'] || options[:cookies] || {}
			@referer_url = options['referer_url'] || options[:referer_url]
			@remote_address = options['remote_address'] || options[:remote_address] || nil
			
		end
		
		def self.build(base)
			case base
			when CGI
				uri = URI::HTTP.build({:host => base.server_name, :port => base.server_port, :path => base.script_name})
				opts = {:cookies => base.cookies, :referer_url => base.referer, :remote_address => base.remote_addr}
				return self.new(uri, base.path_info, base.params, opts)
			end
		end
		
		def has_key?(key)
			@parameters.include?(key)
		end
		alias include? has_key?
		
		def uri
			uri = @script_uri.dup
			uri.path = File.join((uri.path || ''), @path_info) unless @path_info.empty?
			uri.query = @query_string unless @query_string.empty?
			return uri.normalize
		end
		alias url uri
		alias self_uri uri
		alias self_url uri
		
		def route_to(path, query = nil, fragment = nil)
			if path == '/' then
				# ルートへの相対URIは特別扱いする
				abs = self.absolute_uri_to(path, nil, nil)
				abs.path = '/' if abs.path.empty?
				dest = URI.parse(abs.path)
				dest.query = abs.query if query
				dest.fragment = abs.fragment if fragment
			else
				dest = self.uri.route_to(self.absolute_uri_to(path, query, fragment))
			end
			
			return dest
		end
		alias uri_to route_to
		alias url_to route_to
		
		def absolute_uri_to(path, query = nil, fragment = nil)
			unless path.slice(0, 1) == '/' then
				path = "/#{path}"
			end
			dest = @script_uri.dup
			
			unless path == '/' then
				dest.path += path
				dest.path.squeeze!('/')
			end
			dest.query = query if query
			dest.fragment = fragment if fragment
			return dest
		end
		alias absolute_url_to absolute_uri_to
		
		def file_route_to(dest_path)
			dest = @script_uri.dup
			dest += dest_path.to_s
			return self.uri.route_to(dest)
		end
		alias file_uri_to file_route_to
		alias file_url_to file_route_to
		
		# @parameters[key][0]を取得。空文字列ならnil
		def get(key)
			if has_key?(key) then
				value = @parameters[key][0].to_s
				return(value.empty? ? nil : value)
			else
				return nil
			end
		end
		
		# getと同様だが、常にStringを返す（空文字列でもnilを返さない）
		def get_string(key)
			get(key).to_s
		end
		
		# @parameters[key]を取得。返り値は常にArray
		def get_array(key)
			if has_key?(key) then
				return @parameters[key]
			else
				return []
			end
		end
		
		# @cookies[key][0]を取得。空文字列ならnil
		def get_cookie(key)
			if @cookies.include?(key) then
				value = @cookies[key][0].to_s
				return(value.empty? ? nil : value)
			else
				return nil
			end
		end

		# @cookies[key]を配列で取得
		def get_cookie_array(key)
			if @cookies.include?(key) then
				return [@cookies[key]].flatten
			else
				return []
			end
		end

		
		def page
			(get('page') || get('p') || 0).to_i
		end
		
		def start
			(get('start') || 0).to_i
		end

		def length
			(get('length') || 10).to_i
		end
		
		def entry_id
			get('entry_id') || get('id')
		end
		
		def tags
			re = []
			pattern = /\Atags_\d+\z/
			keys = @parameters.keys.find_all{|x, y| x =~ pattern}
			keys.each do |key|
				re << PinkyBlog.decode_tag(self.get(key))
			end
			
			re
		end
		
		def sort
			if has_key?('submit_created') || get('sort') == 'by_created' then
				@sort = BY_CREATED
			elsif has_key?('submit_access') || get('sort') == 'by_access' then
				@sort = BY_ACCESS
			else
				@sort = BY_MODIFIED
			end
		end
		
		def action
			get('action') || get('act')
		end
	end	
end
