#-> Entry, Comment
require 'uuidtools'
require 'md'

require 'pinkyblog/const'
require 'pinkyblog/function'

module PinkyBlog

	# 記事を表すクラス
	class Entry
	
		attr_accessor :id, :content, :last_modified, :created, :title
		attr_accessor :visible, :edited_number, :format
		alias updated last_modified
		alias published created
		alias visible? visible
		
		def initialize(id, args = {})
			@id = id.to_s
			@title = args[:title] || ""
			@content = args[:content] || ""
			@last_modified = args[:last_modified] || Time.now
			@created = args[:created] || Time.now
			@visible = args[:visible] || false
			@edited_number = args[:edited_number] || 0
			@format = args[:format] || nil
		end
		

		
		def invisible?
			!(self.visible?)
		end
		
		def title_caption
			return nil unless self.title
			(@visible ? self.title : "(非公開) " + self.title)
		end
		
		def commentable?
			false
		end
		
		
		def tags
			[]
		end
		
		def normal_tags
			tags - special_tags
		end
		
		def special_tags
			tags.find_all{|x| x =~ SPECIAL_TAG_PATTERN}
		end
		
		def special_tag_names
			re = []
			tags.each do |tag|
				re << $1 if tag =~ SPECIAL_TAG_PATTERN
			end
			
			re
		end


		
		
		
		def build_from_md(md)
			track = md[0]
			@title = track.headers['Title']
			@title.strip! if @title
			@created = Time.at(track.headers['Created'].to_i)
			@visible = (track.headers['Visible'] == 'true')
			@last_modified = Time.at(track.headers['Last-Modified'].to_i)
			@edited_number = track.headers['Edited-Number'].to_i
			@format = track.headers['Format']
			@content = track.body

			return self
		end
		
		
		def to_md
			headers = {}
			headers['Title'] = @title
			headers['Visible'] = @visible.to_s
			headers['Created'] = @created.to_i
			headers['Last-Modified'] = @last_modified.to_i
			headers['Edited-Number'] = @edited_number
			headers['Format'] = @format
			
			return MD.new([MD::Track.new(headers, @content)])
		end
		
		

		
		
		
		def Entry.get_file_path(dir_path, prefix, id)
			file_name = "#{prefix}_#{id}.md"
			path = Pathname.new(dir_path) + file_name
			return path
		end
		
		def Entry.create_new_id
			extend = sprintf("%04x", rand(16 * 16 * 16 * 16))
			"#{Time.now.to_i}-#{extend}"
		end

		def Entry.load(dir_path, prefix, id)
			return Entry.new(id).load(dir_path, prefix)
		end
		
		def Entry.build_from_md(id, md)
			if PinkyBlog.static_entry_id?(id) then
				return StaticEntry.new(id).build_from_md(md)
			else
				return BasicEntry.new(id).build_from_md(md)
			end
		end
	end
	
	class BasicEntry < Entry
		attr_accessor :comments, :tags, :uuid
		def initialize(id = Time.now.to_i, args = {})
			super
			@comments = args[:comments] || []
			@tags = args[:tags] || []
			@uuid = UUID.random_create.to_s
		end

		def build_from_md(md)
			super
			
			@tags = md[0].headers['Tags'].split(' ')
			@uuid = md[0].headers['UUID']
			
		
			@comments.clear
			md.tracks.find_all{|x| x.type == 'PinkyBlog/Comment'}.each do |track|
				@comments << Comment.new(nil, nil, nil, nil).build_from_md_track(track)
			end
			return self
		end
		
		def commentable?
			!(special_tag_names.include?(SPECIAL_TAG_NAMES['uncommentable']))
		end


		
		def to_md
			md = super
			md[0].type = 'PinkyBlog/BasicEntry'
			md[0].headers['Tags'] = @tags.join(' ')
			md[0].headers['UUID'] = @uuid
			
			@comments.each do |comment|
				md.tracks << comment.to_md_track
			end
			return md
		end
	end
	
	class StaticEntry < Entry
		
		def title
			@title || default_title
		end
		
		def default_title
			STATIC_ENTRY_DEFAULT_TITLES[@id]
		end
		
		def comments
			[]
		end
		
		
		def to_md
			md = super
			md[0].type = 'PinkyBlog/StaticEntry'
			return md
		end

		

	end
	

	# 記事につけられたコメントを表すクラス
	class Comment
		attr_reader :writer, :content, :time, :mail_address, :uuid
		def initialize(writer, content, mail_address, time = Time.now)
			@writer = writer
			@content = content
			@mail_address = mail_address
			@time = time
			@uuid = UUID.random_create.to_s

		end
		
		def content_html
			PinkyBlog.escape_html(@content).split(/\r?\n/).join("<br>\n")
		end
		
		
		def to_md_track
			track = MD::Track.new
			track.type = 'PinkyBlog/Comment'
			track.headers['Writer'] = @writer if @writer
			track.headers['Time'] = @time.to_i.to_s
			track.headers['Mail-Address'] = @mail_address if @mail_address
			track.headers['UUID'] = @uuid

			track.body = @content
			return track
		end
		
		
		def build_from_md_track(track)
			track.type = 'PinkyBlog/Comment'
			@writer = track.headers['Writer']
			@time = Time.at(track.headers['Time'].to_i)
			@mail_address = track.headers['Mail-Address']
			@uuid =  track.headers['UUID'] || UUID.random_create.to_s
			@content = track.body
			return self
		end

		
		

	end

end
