#-> Application
require 'cgisup'
require 'time'
require 'digest/md5'
require 'amrita/template'
require 'pathname'
require 'anjson'
require 'jcode'
#require 'atom/service'

require 'pinkyblog/const'
require 'pinkyblog/config'
require 'pinkyblog/function'
require 'pinkyblog/repository'
require 'pinkyblog/view-context'
require 'pinkyblog/module-handler'
require 'pinkyblog/screen'
require 'pinkyblog/request'


module PinkyBlog
	# モジュールやレポジトリ、設定ファイルなどを統括するクラス
	class Application
		attr_reader :config, :repository, :module_handler
		
		# インスタンス生成＋モジュールと設定ファイルのロード
		def self.load(config = Config.new)
			app = self.new(config)
			app.load_config(app.repository.blog_config_file_path)
			app.load_config(app.repository.post_limit_file_path)
			app.load_modules
			return app
		end
		
		def initialize(config = Config.new)
			@config = config
			@repository = Repository.new(@config.data_dir_path)
			@module_handler = ModuleHandler.new(@config)
			@master_session_ids = []
		end
		
		def load_modules
			@module_handler.load(@config)
		end
		
		def load_config(file_path)
			@config.extend_json(file_path)
		end
		
		def generate_session_id
      md5 = Digest::MD5::new
      now = Time::now
      md5.update(now.to_s)
      md5.update(String(now.usec))
      md5.update(String(rand(0)))
      md5.update(String($$))
      md5.update('pinky')
      return md5.hexdigest
		end
		
		
		def set_new_session_id
			new_id = generate_session_id
			@master_session_ids << new_id
			json = AnJSON.dump({'master_session_ids' => @master_session_ids})
			PinkyBlog.write_text('./_session_data', json, File::WRONLY|File::CREAT|File::TRUNC, 0600)
			return new_id
		end
		
		def delete_session_id(id)
			@master_session_ids.delete(id)
			json = AnJSON.dump({'master_session_ids' => @master_session_ids})
			PinkyBlog.write_text('./_session_data', json, File::WRONLY|File::CREAT|File::TRUNC, 0600)
			return nil
		end
		
		def load_session_id
			json = PinkyBlog.read_text('./_session_data')
			if json then
				data = AnJSON.parse(json)
				@master_session_ids = data['master_session_ids']
			else
				@master_session_ids = []
			end
			return nil
		end
		
		def generate_news_feed_models(context)
		end

		def generate_news_feeds(context)
				req = context.request
				dir_path = @config.feed_dir_path
				
				path = @config.lib_dir_path + 'pinkyblog/template/atom.xml'
				tmpl = Amrita::TemplateText.new(path.read.untaint)
				tmpl.amrita_id = 'amrita_src'
				tmpl.xml = true
				
				feed_ids = @repository.get_feed_ids('created', 'modified', 'comment')
				entries = @repository.load_all_entries
				modified_entries = entries.sort{|a, b| b.last_modified <=> a.last_modified}.slice(0, 20)
				created_entries = entries.sort{|a, b| b.created <=> a.created}.slice(0, 20)
				
	
				model = {}
				model[:title] = @config.site_title
				model[:generator] = "PinkyBlog"
				model[:alternate_link] = req.script_url
				model[:author] = {:name => @config.writer_name}
	
				# created
				model[:id] = "urn:uuid:" + feed_ids['created']
				model[:self_link] = context.get_feed_url('created.xml')
				if created_entries.empty? then
					model[:updated] = Time.now.xmlschema
				else
					model[:updated] = created_entries.first.created.xmlschema
					model[:entries] = created_entries.map{|x| entry_to_atom_model(context, x)}
				end
				
				created_xml = ""
				tmpl.expand(created_xml, model)
	
				# modified
				model[:id] = "urn:uuid:" + feed_ids['modified']
				model[:self_link] = context.get_feed_url('modified.xml')
				if modified_entries.empty? then
					model[:updated] = Time.now.xmlschema
				else
					model[:updated] = modified_entries.first.created.xmlschema
					model[:entries] = modified_entries.map{|x| entry_to_atom_model(context, x)}
				end
				modified_xml = ""
				tmpl.expand(modified_xml, model)
				
				comment_data = [] # [[entry, comment], [entry, comment], ...]
				entries.each do |entry|
					entry.comments.each{|x| comment_data << [entry, x]}
				end
				comment_data.sort!{|a,b| b[1].time <=> a[1].time}
				comment_data.slice!(0, 20)
				
				# comment
				model[:id] = "urn:uuid:" + feed_ids['comment']
				model[:self_link] = context.get_feed_url('comment.xml')
				if comment_data.empty? then
				else
					model[:updated] = comment_data.first[1].time.xmlschema	
					model[:entries] = comment_data.map do |entry, comment|
						comment_to_atom_model(context, entry, comment)
					end
				end
				comment_xml = ""
				tmpl.expand(comment_xml, model)


				open(dir_path + 'created.xml', 'w'){|f| f.write(created_xml)}
				open(dir_path + 'modified.xml', 'w'){|f| f.write(modified_xml)}
				open(dir_path + 'comment.xml', 'w'){|f| f.write(comment_xml)}
				
				return true
		end
		
		def entry_to_atom_model(context, entry)
			model = {}
			model[:title] = entry.title
			model[:id] = "urn:uuid:#{entry.uuid}"
			model[:updated] = entry.last_modified.xmlschema
			model[:published] = entry.created.xmlschema
			html = module_handler.translate(context, entry.format, entry.content)
			model[:content] = PinkyBlog.escape_html(html)
			model[:alternate_link] = context.get_cgi_url("entries/#{entry.id}")
			model[:categories] = entry.normal_tags.map{|x| {:term => x}}
			return model
		end
		
		def comment_to_atom_model(context, entry, comment)
			model = {}
			model[:title] = comment.writer || '（無記名）'
			model[:id] = "urn:uuid:#{comment.uuid}"
			model[:updated] = comment.time
			model[:published] = comment.time
			model[:summary] = comment.content
			model[:alternate_link] = context.get_cgi_url("entries/#{entry.id}", nil, 'COMMENT')
			return model
		end

		
		def generate_snapshot(req)
			root = Pathname.new("./snapshot/")
			context = ViewContext.new(@config, @module_handler, req, false, root)
			FileUtils.mkdir_p(root)
			opts = {}
			
			
			context.snapshot_path = Pathname.new('./index.html')
			get_top_screen(context, opts).snapshot(root)
			
			context.snapshot_path = Pathname.new('./files/about.html')
			get_about_screen(context, opts).snapshot(root)

			

			context.snapshot_path = Pathname.new('./files/entries.html')
			opts[:start] = 0
			screen = get_entry_list_screen(context, opts)
			screen.snapshot(root)
			start = 0
			until start > screen.total do
				context.snapshot_path = Pathname.new(sprintf("./files/entries_st%05d.html", start))
				opts[:start] = start
				get_entry_list_screen(context, opts).snapshot(root)
				start += screen.page_length
			end

			entries = @repository.load_all_entries
			entries.each do |entry|
				context.snapshot_path = Pathname.new("./files/entries/#{entry.id}.html")
				get_entry_screen(context, opts, entry.id).snapshot(root)
			end
			
			# CSSテンプレート＆リソースをコピー
			if req.get('include_cdp') then
				copy_dir_for_snapshot(@config.cdp_dir_path, root + 'files/csstemplate/')
			end
			if req.get('include_res') then
				copy_dir_for_snapshot(@config.res_dir_path, root + 'files/res/')
			end
		end
		
		

		def copy_dir_for_snapshot(src_root, dest_root)
			src_root = Pathname.new(src_root)
			Dir.mkdir(dest_root) unless dest_root.exist?
			Pathname.glob(src_root + '**/*').each do |src|
				src.untaint
				dest = dest_root + src.relative_path_from(src_root)
				if src.directory? then
					FileUtils.mkdir(dest) unless dest.exist?
				else
					FileUtils.cp(src, dest)
				end
			end
			
		end
		private :copy_dir_for_snapshot

		
		
		


		# Requestを受けてScreenやRedirectorを返す
		def request(req)

			# 管理者としてログイン済みかどうかをチェック
			master_mode = false
			load_session_id
			
			opts = {:cookies => []}
			ids = req.get_cookie_array('session_id')
			found_id = ids.find{|x| @master_session_ids.include?(x)}
			if found_id then
				master_mode = true
				c = CGI::Cookie.new('session_id', found_id)
				if @config.auto_login? then
					c.expires = Time.now + MASTER_SESSION_TIME_LIMIT
				end
				opts[:cookies] << c
			end
			
			context = ViewContext.new(@config, @module_handler, req, master_mode)
			
			# logout値があればログオフ処理
			if req.get('logout') then
				if master_mode then
					opts[:cookies] << CGI::Cookie.new({'name' => 'session_id', 'expire' => Time.now - 1})
					opts[:message] = "ログアウトしました。"
					opts[:path] = ''
					delete_session_id(req.get_cookie('session_id'))
					return Redirector.new(context, opts)
				else
					opts[:message] = "あなたは管理モードからのログアウトを要求しましたが、ログアウトはすでに完了しています。"
					opts[:http_status] = HTTP_CONFLICT
					return ErrorScreen.new(context, opts)
				end
			end
			
			
			case req.path_items[0]
			#when 'app'
			#	request_app(context, opts)
			
			when 'post'
				request_post(context, opts)
			when 'version', 'system'
				SystemInformationScreen.new(context, opts)
				


			when 'entries'
				entry_id = req.path_items[1]
				if entry_id then
				
					case req.path_items[2]
					when 'edit', 'edit_form'
						# エントリ編集
						if master_mode then
							opts[:entry] = @repository.load_entry(entry_id)
							opts[:entry] ||= (PinkyBlog.static_entry_id? ? StaticEntry.new(entry_id) : BasicEntry.new(entry_id))
							opts[:tag_list] = @repository.get_tag_list([], true)
							
							return EntryEditScreen.new(context, opts)
						else
							return ForbiddenScreen.new(context, opts)
						end
					when nil
						# 個別エントリ表示
						
						ids = @repository.get_entry_ids
						unless ids.include?(entry_id) then
							opts[:http_status] = HTTP_NOT_FOUND
							opts[:message] = "指定されたID #{PinkyBlog.escape_html(entry_id)} の記事は見つかりませんでした。"
							return ErrorScreen.new(context, opts)
						end

						unless master_mode then
							ex_pattern = /#{Regexp.escape(req.script_url.to_s)}/i
							if req.referer_url.to_s.empty? or req.referer_url.to_s =~ ex_pattern then
								# サイト内からのリンクorリファラ不明のリンク
								@repository.lock{
									@repository.record_access(entry_id, nil, req.remote_address)
								}
							else
								@repository.lock{
									@repository.record_access(entry_id, req.referer_url, req.remote_address)
								}
							end
						end
							
						return get_entry_screen(context, opts, entry_id)
					end
					
				else
					return get_entry_list_screen(context, opts)
				end
				
			when 'login', 'login_form'
				return LoginFormScreen.new(context, opts)

			when 'search'

				opts[:keywords] = req.get_string('keyword').split(/\s|　/).map{|x| CGI.unescape(x)}
				opts[:hit_list] = []
				unless opts[:keywords].empty? then
					opts[:hit_list] = @repository.search(opts[:keywords], master_mode)
				end
				
				return SearchScreen.new(context, opts)

			when 'recent'
				return get_recent_screen(context, opts)

			when 'about'
				return get_about_screen(context, opts)

			when 'news_feed'
				return NewsFeedScreen.new(context, opts)

				
			when 'master_menu'
				return ForbiddenScreen.new(context, opts) unless master_mode
				
				case req.path_items[1]
				when 'entry_add_form'
					opts[:tag_list] = @repository.get_tag_list([], true)
					return EntryAddScreen.new(context, opts)
					
				when 'message_list'
					@repository.lock{
						opts[:messages] = @repository.load_messages
						@repository.read_mark_messages
					}
					return MessageListScreen.new(context, opts)
					
				when 'blog_config'
					return BlogConfigScreen.new(context, opts)

				when 'post_limit'
					return PostLimitScreen.new(context, opts)
		
				when 'referer_config'
					opts[:table] = @repository.load_referer_table
					text = PinkyBlog.read_text(@repository.referer_table_file_path)
					if text then
						md = MD.parse(text)
						track = md.find_type('PinkyBlog/RefererTable')
						opts[:table_text] = track.body
					else
						opts[:table_text] = ''
					end
					return RefererConfigScreen.new(context, opts)
					
				when 'entry_manager'
					opts[:entries] = @repository.load_all_entries(true)
					opts[:access_record] = @repository.load_access_data
					opts[:file_data] = {}
					opts[:file_data][:size] = {}
					opts[:entries].each do |entry|
						path = repository.get_entry_file_path(entry.id)
						opts[:file_data][:size][entry.id] = path.size
					end
					return EntryManagerScreen.new(context, opts)
				when 'snapshot'
					
					return SnapshotScreen.new(context, opts)
				else
					opts[:notifications] = []
					
					
					opts[:messages] = @repository.load_messages
					unread = opts[:messages].find_all{|x| !(x.read?)}.size
					if unread > 0 then
						opts[:notifications] << "新しいひとことメッセージが#{unread}件届いています。"
					end
				
					return MasterMenuScreen.new(context, opts)
				end
			when 'format_guide'
				if req.path_items[1] then
					return FormatDetailScreen.new(context, opts)
				else
					return FormatGuideScreen.new(context, opts)
				end
				

			when nil
				get_top_screen(context, opts)
			else
				opts[:message] = "パス #{PinkyBlog.escape_html(req.path_info)} を解釈できませんでした。"
				return ErrorScreen.new(context, opts)
			end
		end
		
		def get_top_screen(context, opts)
			opts[:welcome_entry] = @repository.load_entry('welcome') || StaticEntry.new('welcome')
			opts[:recent_entries] = @repository.load_recent_entries(5, context.master_mode?)
			return TopScreen.new(context, opts)
		end
		
		def get_about_screen(context, opts)
			opts[:about_blog_entry] = @repository.load_entry('about_blog') || StaticEntry.new('about_blog')
			opts[:about_writer_entry] = @repository.load_entry('about_writer') || StaticEntry.new('about_writer')
			return AboutScreen.new(context, opts)
		end
		
		def get_entry_screen(context, opts, entry_id)
			opts[:entry] = @repository.load_entry(entry_id)
			opts[:referer_list] = @repository.get_referer_list(entry_id)
			opts[:ex_footer_visible] = true
			return EntryScreen.new(context, opts)
		end
		
		def get_recent_screen(context, opts)
			opts[:entries] = @repository.load_all_entries(context.master_mode?)
			return RecentScreen.new(context, opts)
		end
		
		def get_entry_list_screen(context, opts)
			opts[:entries] = @repository.load_all_entries(context.master_mode?)
			opts[:tag_list] = @repository.get_tag_list(context.request.tags, context.master_mode?)
			opts[:access_counts] = @repository.load_access_data['counts']
			return EntryListScreen.new(context, opts)
		end



		
		
		

		
		
		private
		
		
		def request_post(context, opts)
			req = context.request
			master_mode = context.master_mode?
			error_proc = Proc.new{|msg|
				ErrorScreen.new(context, {:message => msg})
			}
			
			forbidden_proc = Proc.new{
				ForbiddenScreen.new(context, opts)
			}
			
			redirection = Proc.new{|path, msg|
				opts[:path] = path
				opts[:message] = msg
				Redirector.new(context, opts)
			}
			
			
			case req.action
			when 'change_cdp'
				if req.has_key?('cdp_name') then
					cookie = CGI::Cookie.new('cdp_name', req.get('cdp_name'))
					opts[:cookies] << cookie
					return redirection.call('', "テンプレートを#{req.get('cdp_name')}に変更しました。")
				else
					cookie = CGI::Cookie.new('cdp_name', '')
					cookie.expires = Time.now - 30
					opts[:cookies] << cookie
					return redirection.call('', "テンプレートを変更しました。")
				end
			when 'master_login'
				if req.get("password") == @config.master_password then
	
					auto_login = (req.get('auto_login') ? true : false)
					unless auto_login == @config.auto_login then
						@repository.lock{
							json = PinkyBlog.read_text(@repository.blog_config_file_path)
							if json then
								data = AnJSON.parse(json)
							else
								data = {}
							end
							
							data['auto_login'] = auto_login
							PinkyBlog.write_text(@repository.blog_config_file_path, AnJSON.pretty_build(data))
							
							self.load_config(@repository.blog_config_file_path) # reload for webrick
						}
					end
					
				
					new_id = set_new_session_id
					cookie = CGI::Cookie.new('session_id', new_id)
					if @config.auto_login then
						cookie.expires = Time.now + MASTER_SESSION_TIME_LIMIT
					end
					opts[:cookies] << cookie
					
					return redirection.call('master_menu', "ログインに成功しました。")
				else
					return error_proc.call("パスワードが違うと判定されました。")
				end
				
			when 'snapshot'
				return redirection.call('master_menu', '動作サンプルではこの機能は使えません。') if context.config.demo?
				generate_snapshot(context.request)
				redirection.call('master_menu', 'スナップショットの出力を完了しました。FTPツールなどで確認してください。')
				
			when 'edit_entry'
				if not master_mode then
					return forbidden_proc.call
				elsif req.has_key?('submit_preview') then
					params = {}
					params[:title] = req.get_string('title')
					params[:content] = req.get_string('content').gsub(/\r\n/, "\n")
					params[:invisible] = req.get('invisible')
					params[:tags] = req.tags
					params[:add_tag] = req.get('add_tag')
					params[:format] = req.get('format')
					
					opts = {}
					opts[:tag_list] = @repository.get_tag_list([], true)
					opts[:parameters] = params
	
					if req.get('id') then
						opts[:entry] = @repository.load_entry(req.get('id'))
						screen = EntryEditScreen.new(context, opts)
					else
						screen = EntryAddScreen.new(context, opts)
					end
					return screen
				elsif req.has_key?('submit_complete') && master_mode then
					return redirection.call((req.entry_id ? "entries/#{req.entry_id}" : ""), '動作サンプルでは記事の編集はできません。') if context.config.demo?
					# 入力チェック
					if not PinkyBlog.static_entry_id?(req.entry_id) and req.get_string('title').empty? then
						return error_proc.call("記事タイトルは省略できません。")
					end
					
					edited_id = nil
					to_path = nil
					# 書き込み
					@repository.lock{
						ids = @repository.get_entry_ids
						content = req.get_string('content')
						visible = !(req.get('invisible'))
						format = req.get('format')
						title = req.get_string('title')
					
						if req.entry_id && PinkyBlog.static_entry_id?(req.entry_id) then
							# スタティックエントリ
							if ids.include?(req.entry_id) then
								@repository.edit_static_entry(req.entry_id, visible, title, content, format)
							else
								@repository.add_new_static_entry(req.entry_id, visible, title, content, format)
							end
							
							case req.entry_id
							when 'welcome'
								to_path = ''
							when 'about_blog', 'about_writer'
								to_path = 'about'
							end
						else
							# 通常記事
							tags = req.tags + req.get_string('add_tag').split(/\s|　/)
							tags.uniq!
							
							if req.entry_id then
								@repository.edit_basic_entry(req.entry_id, visible, title, content, format, tags)
							else
								edited_id = Entry.create_new_id
								@repository.add_new_basic_entry(edited_id, visible, title, content, format, tags)

							end
							generate_news_feeds(context)
							
							to_path = "entries/#{edited_id || req.entry_id}"
						end
						
					}
					
					

					return redirection.call(to_path, "エントリの編集を完了しました。")
				end
			when 'comment'
				content = req.get_string('content')

				if content.empty? then
					return error_proc.call("コメント本文に何も入力されていません。")
				end

				if content.jlength > @config.real_comment_length_limit then
					return error_proc.call("コメントが長すぎるため投稿できませんでした。（#{content.jlength} 文字 / #{@config.real_comment_length_limit} 文字）")
				end
				
				if @config.commentator_name_required? and not req.get('name') then
					return error_proc.call("名前を記入してください。")
				end
				
				if @config.commentator_address_required? and not req.get('address') then
					return error_proc.call("メールアドレスを記入してください。")
				end
				
				if @config.check_spam(content) then
					return PostBlockScreen.new(context)
				end
				
				@repository.lock{
					@repository.comment_to_entry(req.entry_id, req.get('name'), content, req.get('address'))
				}
				# コメント欄の名前とアドレスをCookieで記憶
				c1 = CGI::Cookie.new('default_name', req.get_string('name'))
				c1.expires = Time.now + 60*60*24*30
				c2 = CGI::Cookie.new('default_address', req.get_string('address'))
				c2.expires = Time.now + 60*60*24*30
				opts[:cookies] << c1 << c2
				
				generate_news_feeds(context)
				
				return redirection.call("entries/#{req.entry_id}", "コメントの投稿に成功しました。ありがとうございます。")

			when 'message'
				content = req.get_string('content')

				if content.empty? then
					return error_proc.call("メッセージ本文が何も入力されていません。")
				end

				if content.jlength > @config.real_message_length_limit then
					return error_proc.call("メッセージが長すぎるため投稿できませんでした。（#{content.jlength} 文字 / #{@config.real_message_length_limit} 文字）")
				end
				
				if @config.check_spam(content) then
					return PostBlockScreen.new(context)
				end

				
				@repository.lock{
					@repository.add_message(content)
				}
				
				return redirection.call("", "メッセージの送信に成功しました。ありがとうございます。")


			when 'blog_config'
				return redirection.call("master_menu/blog_config", '動作サンプルでは、設定の変更はできません。') if context.config.demo?

				if master_mode then
					return redirection.call('master_menu/blog_config', "記入不備：blogの名前を入力してください。") unless req.get('site_title')
				
					@repository.lock{
						data = {}
						data['site_title'] = req.get_string('site_title')
						data['writer_name'] = req.get_string('writer_name')
						data['writer_address'] = req.get('writer_address')
						data['home_url'] = req.get('home_url')
						data['about_visible'] = (req.get('about_visible') ? true : false)
						data['recent_entries_visible'] = (req.get('recent_entries_visible') ? true : false)
						data['news_feed_visible'] = (req.get('news_feed_visible') ? true : false)
						data['headline_title'] = req.get('headline_title') if req.get('headline_title')
						data['commentator_name_required'] = (req.get('commentator_name_required') ? true : false)
						data['commentator_address_required'] = (req.get('commentator_address_required') ? true : false)
						data['message_form_visible'] = (req.get('message_form_visible') ? true : false)
						data['message_form_title'] = req.get_string('message_form_title')
						data['message_form_guide'] = req.get_string('message_form_guide')
						data['default_translator'] = req.get_string('default_translator')
						data['menu_captions'] = {}
						MENU_KEYS.each do |key|
							cgi_key = "menu_caption_of_#{key}"
							data['menu_captions'][key] = req.get_string(cgi_key) if req.get(cgi_key)
						end
						
						
						PinkyBlog.write_text(@repository.blog_config_file_path, AnJSON.pretty_build(data))
						self.load_config(@repository.blog_config_file_path) # reload for webrick
					}
					return redirection.call('master_menu/blog_config', "blog設定を変更しました。")
				else
					return forbidden_proc.call
				end

			when 'post_limit'
				return redirection.call("master_menu/post_limit", '動作サンプルでは、設定の変更はできません。') if context.config.demo?
				if master_mode then
					@repository.lock{
						data = {}
						data['message_length_limit'] = req.get_string('message_length_limit')
						data['comment_length_limit'] = req.get_string('comment_length_limit')
						data['block_http'] = (req.get('block_http') ? true : false)
						data['block_ascii'] = (req.get('block_ascii') ? true : false)
						data['ng_words'] = req.get_string('ng_word').split(/\r\n|\n/m)
						data['ng_words'].delete_if{|x| x.empty?}
						
						PinkyBlog.write_text(@repository.post_limit_file_path, AnJSON.pretty_build(data))
						self.load_config(@repository.post_limit_file_path) # reload for webrick
					}
					return redirection.call('master_menu/post_limit', "投稿制限を変更しました。")
				else
					return forbidden_proc.call
				end


			when 'referer_config'
				if master_mode then
					return redirection.call("master_menu/referer_config", '動作サンプルでは、設定の変更はできません。') if context.config.demo?
					@repository.lock{
						@repository.save_referer_table(req.get_string('table'))
					}
					return redirection.call('master_menu/referer_config', "リファラ設定を変更しました。")
				else
					return forbidden_proc.call
				end
			when 'delete_message'
				return redirection.call("master_menu/message_list", '動作サンプルでは、この機能は使えません。') if context.config.demo?
				if master_mode then
					@repository.lock{
						target_ids = req.get_array('message_ids')
						unless @repository.delete_messages(target_ids) then
							error_proc.call("メッセージファイルの読み書きに失敗しました。")
						end
					}
					return redirection.call('master_menu/message_list', "選択したメッセージをすべて削除しました。")
				else
					return forbidden_proc.call
				end
				
			when 'act_entries'
				if master_mode then
					if req.get_array('entry_ids').empty? then
						return error_proc.call("記事が一つも選択されていません。")
					end
					
				
					if req.has_key?('submit_delete') then
						opts[:entries] = req.params['entry_ids'].map{|x| @repository.load_entry(x)}
						opts[:access_record] = @repository.load_access_data
						opts[:file_data] = {}
						opts[:file_data][:size] = {}

						opts[:entries].each do |entry|
							path = repository.get_entry_file_path(entry.id)
							opts[:file_data][:size][entry.id] = path.size
						end

						return EntryDeleteConfirmationScreen.new(context, opts)
					elsif req.has_key?('submit_delete_ok') then
						return redirection.call("master_menu/entry_manager", '動作サンプルでは、この操作は行えません。') if context.config.demo?
						@repository.lock{
							req.params['entry_ids'].each do |id|
								@repository.delete_entry(id)
							end
						}
						return redirection.call('master_menu/entry_manager', "選択した記事をtrashディレクトリに移動しました。ファイルの削除は手動で行ってください。")
					elsif req.has_key?('submit_show') then
						return redirection.call("master_menu/entry_manager", '動作サンプルでは、この操作は行えません。') if context.config.demo?
						@repository.lock{
							req.params['entry_ids'].each do |id|
								entry = @repository.load_entry(id)
								entry.visible = true
								@repository.save_entry(entry)
							end
						}
						return redirection.call('master_menu/entry_manager', "選択した記事を「公開」状態にしました。")
					elsif req.has_key?('submit_hide') then
						return redirection.call("master_menu/entry_manager", '動作サンプルでは、この操作は行えません。') if context.config.demo?
						@repository.lock{
							req.params['entry_ids'].each do |id|
								entry = @repository.load_entry(id)
								entry.visible = false
								@repository.save_entry(entry)
							end
						}
						return redirection.call('master_menu/entry_manager', "選択した記事を「非公開」状態にしました。")
						
					elsif req.has_key?('submit_delete_all_tag') then
						@repository.lock{
							req.params['entry_ids'].each do |id|
								entry = @repository.load_entry(id)
								entry.tags.clear
								@repository.save_entry(entry)
							end
						}
						return redirection.call('master_menu/entry_manager', "選択した記事のタグをすべて削除しました。")
					
					elsif req.has_key?('submit_add_tag') || req.has_key?('submit_delete_tag') then
						tags = req.get_string('target_tag').split(/\s|　/)
						tags.uniq!
						@repository.lock{
							req.params['entry_ids'].each do |id|
								entry = @repository.load_entry(id)
								if req.has_key?('submit_add_tag') then
									entry.tags += tags
									entry.tags.uniq!
								elsif req.has_key?('submit_delete_tag') then
									entry.tags -= tags
								end
								
								@repository.save_entry(entry)
							end
						}
						msg = "タグ"
						msg << tags.map{|x| "「#{x}」"}.join
						msg << (req.has_key?('submit_delete_tag') ? "を削除しました。" : "を追加しました。")
						return redirection.call('master_menu/entry_manager', msg)
					end
				else
					return forbidden_proc.call

				end
			else
				return error_proc.call("「#{PinkyBlog.escape_html(req.get_string('action'))}」は正しいpostアクションではありません。")
			end
		end

=begin
		def request_app(context, opts)
			
			case context.request.path_items[1]
			when 'entries'
				if context.request.path_items[2] then
					return AtomEntry.new(context, @repository.load_entry(context.request.path_items[2]))
				else
					return AtomEntryCollection.new(context, @repository.load_all_entries)
				end
			else
				return AtomService.new(context)
			end
		end
=end
		
	end
	
	class ApplicationStub < Application
		def initialize(config = Config.new)
			super
			@config = config
			@repository = RepositoryStub.new(@config.data_dir_path)
			@module_handler = ModuleHandler.new(@config)
			@master_session_ids = []
		end
		
		def load_session_id
			return nil
		end

	end
end