/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate.sh;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Collection;
import java.util.Collections;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import net.morilib.nina.Quadro;
import net.morilib.nina.cmd.NinaInfo;
import net.morilib.nina.translate.NinaTranslator;
import net.morilib.sh.AbstractShFileSystem;
import net.morilib.sh.DefaultShVirtualDirectory;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFile;
import net.morilib.sh.ShProcess;
import net.morilib.sh.ShSecurityPolicy;
import net.morilib.sh.ShStat;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/25
 */
public class ShNinatFileSystem extends AbstractShFileSystem {

	private static class StF implements ShFile {

		private byte[] buf;
		private String name;

		StF(String name, String f) {
			this.name = name;
			this.buf  = f.getBytes();
		}

		@Override
		public boolean isDirectory() {
			return false;
		}

		@Override
		public boolean isExecutable() {
			return false;
		}

		@Override
		public boolean isExist() {
			return true;
		}

		@Override
		public boolean isFile() {
			return true;
		}

		@Override
		public boolean isReadable() {
			return true;
		}

		@Override
		public boolean isWritable() {
			return false;
		}

		@Override
		public boolean isHidden() {
			return false;
		}

		@Override
		public boolean isZeroFile() {
			return buf.length == 0;
		}

		@Override
		public long getLastModified() {
			return 0;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public Collection<ShFile> getFiles() {
			return Collections.emptySet();
		}

		@Override
		public InputStream getInputStream() throws IOException {
			return new ByteArrayInputStream(buf);
		}

		@Override
		public PrintStream getPrintStream(
				boolean append) throws IOException {
			return null;
		}

		@Override
		public ShProcess getProcess() {
			return null;
		}

		@Override
		public ShFile toAbsolute() {
			return this;
		}

		@Override
		public ShFile toCanonical() {
			return this;
		}

		@Override
		public boolean mkdir() {
			return false;
		}

		@Override
		public boolean renameTo(ShFile f) {
			return false;
		}

		@Override
		public void setLastModified(long time) {
			// do nothing
		}

		@Override
		public boolean rmdir() {
			return false;
		}

		@Override
		public boolean deleteFile() {
			return false;
		}

		@Override
		public ShStat getStat() {
			return new ShStat(toString(), 0, buf.length,
					ShStat.READABLE);
		}

		@Override
		public String toString() {
			return name;
		}

	}

	private static class StR implements ShFile {

		private String name, resource;

		StR(String name, String r) {
			this.name = name;
			this.resource = r;
		}

		@Override
		public boolean isDirectory() {
			return false;
		}

		@Override
		public boolean isExecutable() {
			return false;
		}

		@Override
		public boolean isExist() {
			return true;
		}

		@Override
		public boolean isFile() {
			return true;
		}

		@Override
		public boolean isReadable() {
			return true;
		}

		@Override
		public boolean isWritable() {
			return false;
		}

		@Override
		public boolean isHidden() {
			return false;
		}

		@Override
		public boolean isZeroFile() {
			return false;
		}

		@Override
		public long getLastModified() {
			return 0;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public Collection<ShFile> getFiles() {
			return Collections.emptySet();
		}

		@Override
		public InputStream getInputStream() throws IOException {
			return StR.class.getResourceAsStream(resource);
		}

		@Override
		public PrintStream getPrintStream(
				boolean append) throws IOException {
			return null;
		}

		@Override
		public ShProcess getProcess() {
			return null;
		}

		@Override
		public ShFile toAbsolute() {
			return this;
		}

		@Override
		public ShFile toCanonical() {
			return this;
		}

		@Override
		public boolean mkdir() {
			return false;
		}

		@Override
		public boolean renameTo(ShFile f) {
			return false;
		}

		@Override
		public void setLastModified(long time) {
			// do nothing
		}

		@Override
		public boolean rmdir() {
			return false;
		}

		@Override
		public boolean deleteFile() {
			return false;
		}

		@Override
		public ShStat getStat() {
			return new ShStat(toString(), 0, 85, ShStat.READABLE);
		}

		@Override
		public String toString() {
			return name;
		}

	}

	private static class StO implements ShFile {

		private NinaInfo info;
		private String name;
		private Object realname;

		StO(NinaInfo f, String name, Object r) {
			this.info = f;
			this.name = name;
			this.realname = r;
		}

		@Override
		public boolean isDirectory() {
			return false;
		}

		@Override
		public boolean isExecutable() {
			return false;
		}

		@Override
		public boolean isExist() {
			return true;
		}

		@Override
		public boolean isFile() {
			return true;
		}

		@Override
		public boolean isReadable() {
			return true;
		}

		@Override
		public boolean isWritable() {
			return false;
		}

		@Override
		public boolean isHidden() {
			return false;
		}

		@Override
		public boolean isZeroFile() {
			return false;
		}

		@Override
		public long getLastModified() {
			return 0;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public Collection<ShFile> getFiles() {
			return Collections.emptySet();
		}

		@Override
		public InputStream getInputStream() throws IOException {
			ByteArrayOutputStream ous;
			byte[] b = new byte[1024];
			InputStream ins = null;
			int l;

			try {
				ous = new ByteArrayOutputStream();
				ins = info.getInputStream(realname);
				while((l = ins.read(b)) >= 0) {
					ous.write(b, 0, l);
				}
				ous.close();
				return new ByteArrayInputStream(ous.toByteArray());
			} finally {
				if(ins != null) {
					ins.close();
				}
			}
		}

		@Override
		public PrintStream getPrintStream(
				boolean append) throws IOException {
			return null;
		}

		@Override
		public ShProcess getProcess() {
			return null;
		}

		@Override
		public ShFile toAbsolute() {
			return this;
		}

		@Override
		public ShFile toCanonical() {
			return this;
		}

		@Override
		public boolean mkdir() {
			return false;
		}

		@Override
		public boolean renameTo(ShFile f) {
			return false;
		}

		@Override
		public void setLastModified(long time) {
			// do nothing
		}

		@Override
		public boolean rmdir() {
			return false;
		}

		@Override
		public boolean deleteFile() {
			return false;
		}

		@Override
		public ShStat getStat() {
			return new ShStat(toString(), 0, 85, ShStat.READABLE);
		}

		@Override
		public String toString() {
			return name;
		}

	}

	private static class StB implements ShFile {

		private String name;
		private ByteArrayOutputStream ous = null;
		private PrintStream pr;

		StB(String name) {
			this.name = name;
		}

		@Override
		public boolean isDirectory() {
			return false;
		}

		@Override
		public boolean isExecutable() {
			return false;
		}

		@Override
		public boolean isExist() {
			return true;
		}

		@Override
		public boolean isFile() {
			return true;
		}

		@Override
		public boolean isReadable() {
			return true;
		}

		@Override
		public boolean isWritable() {
			return true;
		}

		@Override
		public boolean isHidden() {
			return false;
		}

		@Override
		public boolean isZeroFile() {
			return ous.size() == 0;
		}

		@Override
		public long getLastModified() {
			return 0;
		}

		@Override
		public String getName() {
			return name;
		}

		@Override
		public Collection<ShFile> getFiles() {
			return Collections.emptySet();
		}

		@Override
		public InputStream getInputStream() throws IOException {
			if(ous != null) {
				return new ByteArrayInputStream(ous.toByteArray()) {

					public void close() throws IOException {
						// read once
						super.close();
						ous = null;
					}

				};
			} else {
				return null;
			}
		}

		@Override
		public PrintStream getPrintStream(
				boolean append) throws IOException {
			if(append && ous != null) {
				return pr;
			} else {
				ous = new ByteArrayOutputStream();
				return pr = new PrintStream(ous);
			}
		}

		@Override
		public ShProcess getProcess() {
			return null;
		}

		@Override
		public ShFile toAbsolute() {
			return this;
		}

		@Override
		public ShFile toCanonical() {
			return this;
		}

		@Override
		public boolean mkdir() {
			return false;
		}

		@Override
		public boolean renameTo(ShFile f) {
			return false;
		}

		@Override
		public void setLastModified(long time) {
			// do nothing
		}

		@Override
		public boolean rmdir() {
			return false;
		}

		@Override
		public boolean deleteFile() {
			return false;
		}

		@Override
		public ShStat getStat() {
			return new ShStat(toString(), 0, ous.size(),
					ShStat.WRITABLE);
		}

		@Override
		public String toString() {
			return name;
		}

	}

	//
	private static final String ROOT_D =
			"/" + NinaTranslator.class.getPackage().getName().replace('.', '/');
	private static final String LICENSE_D = ROOT_D + "/license";
	private static final Pattern PTN1 = Pattern.compile(
			"/output/([^/]+)");
	private static final Pattern PTN2 = Pattern.compile(
			"/license/([^/]+)");

	private DefaultShVirtualDirectory root, license;
	private ShFile outputDir;
	private ShFile fragmentFile, definitionFile, fieldFile, headerFile;
	private ShFile readFile, bufFile;
	private ShFile script1File, script2File, script3File;
	private ShFile yjavaFile, ycsFile,  yjsFile,  ycFile;
	private ShFile njavaFile, ncsFile,  njsFile,  ncFile;
	private ShFile sjavaFile, scsFile,  sjsFile,  scFile;
	private ShFile descFile,  licFile,  pliFile;
	private ShFile inj1File,  inj2File, inj3File, injFile;
	private ShFile sdjvFile,  sdcsFile, sdjsFile, sdcFile;
	private ShFile snjvFile,  sncsFile, snjsFile, sncFile;
	private ShFile stjvFile,  stcsFile, stjsFile, stcFile;
	private NinaInfo info;

	private StR newr(String s) {
		return new StR(s, LICENSE_D + "/" + s + ".txt");
	}

	private StR newr0(String s) {
		return new StR(s, ROOT_D + "/" + s);
	}

	/**
	 * 
	 * @param f
	 * @param d
	 * @param opts
	 */
	public ShNinatFileSystem(String f, Quadro q, NinaInfo info) {
		this.info = info;

		root = new DefaultShVirtualDirectory(null, null);
		fragmentFile   = new StF("fragment", f);
		definitionFile = new StF("definition", q.getDefinition());
		fieldFile      = new StF("field", q.getFragmentField());
		headerFile     = new StF("header", q.getFragmentByName("header"));
		readFile       = new StF("read", q.getFragmentByName("read"));
		descFile       = new StF("description", q.getDescription());
		licFile        = new StF("userlicense", q.getLicense().trim());
		bufFile        = new StB("tmp1");
		script1File    = newr0("nina_template.dfa.js.sub1.sh");
		script2File    = newr0("nina_template.nfa.js.sub1.sh");
		script3File    = newr0("nina_template.tm.js.sub1.sh");
		yjavaFile      = newr0("y.tab.java.sh");
		ycsFile        = newr0("y.tab.cs.sh");
		yjsFile        = newr0("y.tab.js.sh");
		ycFile         = newr0("y.tab.c.sh");
		njavaFile      = newr0("nina_template.nfa.java.sh");
		ncsFile        = newr0("nina_template.nfa.cs.sh");
		njsFile        = newr0("nina_template.nfa.js.sh");
		ncFile         = newr0("nina_template.nfa.c.sh");
		sjavaFile      = newr0("nina_template.nfa.java.sub2.sh");
		scsFile        = newr0("nina_template.nfa.cs.sub2.sh");
		sjsFile        = newr0("nina_template.nfa.js.sub2.sh");
		scFile         = newr0("nina_template.nfa.c.sub2.sh");
		pliFile        = newr0("putlicense_cstyle.sh");
		license        = new DefaultShVirtualDirectory(root, "license");
		inj1File       = new StB("inject_tmp1");
		inj2File       = new StB("inject_tmp2");
		inj3File       = new StB("inject_tmp3");
		sdjvFile       = newr0("nina_template.dfa.java.sh");
		sdcsFile       = newr0("nina_template.dfa.cs.sh");
		sdjsFile       = newr0("nina_template.dfa.js.sh");
		sdcFile        = newr0("nina_template.dfa.c.sh");
		snjvFile       = newr0("nina_template.nfa.java.sh");
		sncsFile       = newr0("nina_template.nfa.cs.sh");
		snjsFile       = newr0("nina_template.nfa.js.sh");
		sncFile        = newr0("nina_template.nfa.c.sh");
		stjvFile       = newr0("nina_template.tm.java.sh");
		stcsFile       = newr0("nina_template.tm.cs.sh");
		stjsFile       = newr0("nina_template.tm.js.sh");
		stcFile        = newr0("nina_template.tm.c.sh");

		root.addFile(fragmentFile);
		root.addFile(definitionFile);
		root.addFile(fieldFile);
		root.addFile(headerFile);
		root.addFile(readFile);
		root.addFile(descFile);
		root.addFile(licFile);
		root.addFile(bufFile);
		root.addFile(script1File);
		root.addFile(script2File);
		root.addFile(script3File);
		root.addFile(yjavaFile);
		root.addFile(ycsFile);
		root.addFile(yjsFile);
		root.addFile(ycFile);
		root.addFile(njavaFile);
		root.addFile(ncsFile);
		root.addFile(njsFile);
		root.addFile(ncFile);
		root.addFile(sjavaFile);
		root.addFile(scsFile);
		root.addFile(sjsFile);
		root.addFile(scFile);
		root.addFile(pliFile);
		root.addFile(license);
		root.addFile(inj1File);
		root.addFile(inj2File);
		root.addFile(inj3File);
		root.addFile(sdjvFile);
		root.addFile(sdcsFile);
		root.addFile(sdjsFile);
		root.addFile(sdcFile);
		root.addFile(snjvFile);
		root.addFile(sncsFile);
		root.addFile(snjsFile);
		root.addFile(sncFile);
		root.addFile(stjvFile);
		root.addFile(stcsFile);
		root.addFile(stjsFile);
		root.addFile(stcFile);
		root.addFile(info.getShOutputDir(this));

		if(info.getInjectFile() != null) {
			injFile = new StO(info, "inject", info.getInjectFile());
			root.addFile(injFile);
		}

		license.addFile(newr("apache"));
		license.addFile(newr("bsd"));
		license.addFile(newr("bsd2"));
		license.addFile(newr("default"));
		license.addFile(newr("gpl"));
		license.addFile(newr("lgpl"));
		license.addFile(newr("mit"));
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getRoot()
	 */
	@Override
	public ShFile getRoot() {
		return root;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getFile(java.lang.String)
	 */
	@Override
	public ShFile getFile(String s) {
		Matcher m;

		if(s.equals("/")) {
			return root;
		} else if(s.equals("fragment")) {
			return fragmentFile;
		} else if(s.equals("definition")) {
			return definitionFile;
		} else if(s.equals("field")) {
			return fieldFile;
		} else if(s.equals("header")) {
			return headerFile;
		} else if(s.equals("read")) {
			return readFile;
		} else if(s.equals("description")) {
			return descFile;
		} else if(s.equals("userlicense")) {
			return licFile;
		} else if(s.equals("tmp1")) {
			return bufFile;
		} else if(s.equals("nina_template.dfa.js.sub1.sh")) {
			return script1File;
		} else if(s.equals("nina_template.nfa.js.sub1.sh")) {
			return script2File;
		} else if(s.equals("nina_template.tm.js.sub1.sh")) {
			return script3File;
		} else if(s.equals("y.tab.java.sh")) {
			return yjavaFile;
		} else if(s.equals("y.tab.cs.sh")) {
			return ycsFile;
		} else if(s.equals("y.tab.js.sh")) {
			return yjsFile;
		} else if(s.equals("y.tab.c.sh")) {
			return ycFile;
		} else if(s.equals("nina_template.nfa.java.sh")) {
			return njavaFile;
		} else if(s.equals("nina_template.nfa.cs.sh")) {
			return ncsFile;
		} else if(s.equals("nina_template.nfa.js.sh")) {
			return njsFile;
		} else if(s.equals("nina_template.nfa.c.sh")) {
			return ncFile;
		} else if(s.equals("nina_template.nfa.java.sub2.sh")) {
			return sjavaFile;
		} else if(s.equals("nina_template.nfa.cs.sub2.sh")) {
			return scsFile;
		} else if(s.equals("nina_template.nfa.js.sub2.sh")) {
			return sjsFile;
		} else if(s.equals("nina_template.nfa.c.sub2.sh")) {
			return scFile;
		} else if(s.equals("putlicense_cstyle.sh")) {
			return pliFile;
		} else if(s.equals("inject_tmp1")) {
			return inj1File;
		} else if(s.equals("inject_tmp2")) {
			return inj2File;
		} else if(s.equals("inject_tmp3")) {
			return inj3File;
		} else if(injFile != null && s.equals("inject")) {
			return injFile;
		} else if(s.equals("nina_template.dfa.java.sh")) {
			return sdjvFile;
		} else if(s.equals("nina_template.dfa.cs.sh")) {
			return sdcsFile;
		} else if(s.equals("nina_template.dfa.js.sh")) {
			return sdjsFile;
		} else if(s.equals("nina_template.dfa.c.sh")) {
			return sdcFile;
		} else if(s.equals("nina_template.nfa.java.sh")) {
			return snjvFile;
		} else if(s.equals("nina_template.nfa.cs.sh")) {
			return sncsFile;
		} else if(s.equals("nina_template.nfa.js.sh")) {
			return snjsFile;
		} else if(s.equals("nina_template.nfa.c.sh")) {
			return sncFile;
		} else if(s.equals("nina_template.tm.java.sh")) {
			return stjvFile;
		} else if(s.equals("nina_template.tm.cs.sh")) {
			return stcsFile;
		} else if(s.equals("nina_template.tm.js.sh")) {
			return stjsFile;
		} else if(s.equals("nina_template.tm.c.sh")) {
			return stcFile;
		} else if(s.equals("/output")) {
			return outputDir;
		} else if(s.equals("/license")) {
			return license;
		} else if((m = PTN1.matcher(s)).matches()) {
			return info.getShOutputFile(this, s, m.group(1));
		} else if((m = PTN2.matcher(s)).matches()) {
			return license.getFile(m.group(1));
		} else {
			return null;
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getFile(java.lang.String, java.lang.String)
	 */
	@Override
	public ShFile getFile(String dir, String name) {
		return getFile(getFile(dir), name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getFile(net.morilib.sh.ShFile, java.lang.String)
	 */
	@Override
	public ShFile getFile(ShFile dir, String name) {
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getNativeFile(java.lang.String)
	 */
	@Override
	public ShFile getNativeFile(String s) {
		return getFile(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getSecurityPolicy()
	 */
	@Override
	public ShSecurityPolicy getSecurityPolicy() {
		return ShSecurityPolicy.ALL_PERMITTED;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getProfile()
	 */
	@Override
	public InputStream getProfile() {
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getRc()
	 */
	@Override
	public InputStream getRc() {
		return null;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#getHome()
	 */
	@Override
	public ShFile getHome() {
		return getRoot();
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShFileSystem#searchPath(net.morilib.sh.ShEnvironment, java.lang.String)
	 */
	@Override
	public ShFile searchPath(ShEnvironment env, String name) {
		return null;
	}

}
