/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.automata.nfa.op;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;

import net.morilib.automata.DFA;
import net.morilib.automata.DFAState;
import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.automata.dfa.DFAs;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.util.IntervalMap;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2014/01/20
 */
public class DFAWrappedNFA<T, A, B> implements NFA<T, A, B> {

	//
	private final NFAEdges<T> eedg = new EpsilonEdge<T>(null);

	//
	private Set<DFAState<T, A, B>> allStates;
	private DFA<T, A, B> dfa;

	/**
	 * 
	 * @param dfa
	 */
	public DFAWrappedNFA(DFA<T, A, B> dfa) {
		this.dfa = dfa;
		this.allStates = new HashSet<DFAState<T, A, B>>(
				DFAs.allStates(dfa));
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isState(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isState(NFAState o) {
		return allStates.contains(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Set<NFAState> getStates(NFAState state, T alphabet) {
		DFAState<T, A, B> s;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			return Collections.singleton((NFAState)s.go(alphabet));
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, net.morilib.range.Range)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Set<NFAState> getStates(NFAState state, Range rng) {
		DFAState<T, A, B> s;
		Set<NFAState> r;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			r = new HashSet<NFAState>();
			for(Interval t : s.getAlphabetRanges()) {
				if(t.independentOf(rng)) {
					// do nothing
				} else if(t.isInfimumClosed()) {
					r.add(s.go((T)t.getInfimumBound()));
				} else if(t.isSupremumClosed()) {
					r.add(s.go((T)t.getSupremumBound()));
				} else {
					throw new RuntimeException();
				}
			}
			return r;
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStates(net.morilib.automata.NFAState, java.util.EnumSet)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Set<NFAState> getStates(NFAState state,
			EnumSet<TextBound> bound) {
		DFAState<T, A, B> s;
		Set<NFAState> r;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			r = new HashSet<NFAState>();
			for(TextBound x : bound) {
				r.add((NFAState)s.goBound(x));
			}
			return r;
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStatesEpsilon(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<NFAState> getStatesEpsilon(NFAState state) {
		return Collections.emptySet();
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getStatesBound(net.morilib.automata.NFAState, java.util.EnumSet)
	 */
	@Override
	public Set<NFAState> getStatesBound(NFAState state,
			EnumSet<TextBound> bound) {
		return getStates(state, bound);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getInitialStates()
	 */
	@Override
	public Set<NFAState> getInitialStates() {
		return Collections.singleton((NFAState)dfa.getInitialState());
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isInitialState(net.morilib.automata.NFAState)
	 */
	@Override
	public boolean isInitialState(NFAState o) {
		return dfa.getInitialState().equals(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isFinal(net.morilib.automata.NFAState)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean isFinal(NFAState state) {
		return (allStates.contains(state) &&
				((DFAState<T, A, B>)state).isAccepted());
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isFinalAny(java.util.Set)
	 */
	@Override
	public boolean isFinalAny(Set<NFAState> states) {
		for(NFAState s : states) {
			if(isFinal(s))  return true;
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getEdges(net.morilib.automata.NFAState)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public NFAEdges<T> getEdges(NFAState state) {
		final DFAState<T, A, B> s;

		if(!isState(state))  return eedg;
		s = (DFAState<T, A, B>)state;
		return new NFAEdges<T>() {

			@Override
			public Set<NFAState> goNext(T alphabet) {
				return Collections.singleton((NFAState)s.go(alphabet));
			}

			@Override
			public Set<NFAState> goNext(int alphabet) {
				return goNext(Integer.valueOf(alphabet));
			}

			@Override
			public Set<NFAState> goNext(char alphabet) {
				return goNext(Integer.valueOf(alphabet));
			}

			@Override
			public Set<NFAState> goNextEpsilon() {
				return Collections.emptySet();
			}

			@Override
			public Set<? extends Range> nextAlphabets() {
				Set<Range> r = new HashSet<Range>();

				for(Interval v : s.getAlphabetRanges()) {
					r.add(v);
				}
				return r;
			}

			@Override
			public boolean isNextEpsilon() {
				return false;
			}

		};
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextAlphabets(net.morilib.automata.NFAState)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Set<Interval> nextAlphabets(NFAState state) {
		final DFAState<T, A, B> s;
		Set<Interval> r;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			r = new HashSet<Interval>();
			for(Interval v : s.getAlphabetRanges())  r.add(v);
			return r;
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextAlphabets(java.util.Set)
	 */
	@Override
	public Iterable<Interval> nextAlphabets(Set<NFAState> states) {
		IntervalMap<Void> m = new IntervalMap<Void>();

		for(NFAState s : states) {
			for(Interval v : nextAlphabets(s)) {
				m.put(v, null);
			}
		}
		return m.keySet();
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(net.morilib.automata.NFAState)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Set<T> nextDiscreteAlphabets(NFAState state) {
		final DFAState<T, A, B> s;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			return new HashSet<T>(s.getAlphabets());
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#nextDiscreteAlphabets(java.util.Set)
	 */
	@Override
	public Iterable<T> nextDiscreteAlphabets(Set<NFAState> states) {
		Set<T> m = new HashSet<T>();

		for(NFAState s : states) {
			m.addAll(nextDiscreteAlphabets(s));
		}
		return m;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getAcceptedStates()
	 */
	@Override
	public Set<NFAState> getAcceptedStates() {
		Set<NFAState> r = new HashSet<NFAState>();

		for(DFAState<T, A, B> d : allStates) {
			if(d.isAccepted())  r.add((NFAState)d);
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getMatchTag(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<B> getMatchTag(NFAState state) {
		return Collections.emptySet();
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getMatchTagEnd(net.morilib.automata.NFAState)
	 */
	@Override
	public Set<B> getMatchTagEnd(NFAState state) {
		return Collections.emptySet();
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#getAccept(net.morilib.automata.NFAState)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Set<A> getAccept(NFAState state) {
		DFAState<T, A, B> s;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			return new HashSet<A>(s.getAccepted());
		} else {
			return Collections.emptySet();
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.automata.NFA#isAccepted(net.morilib.automata.NFAState)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public boolean isAccepted(NFAState state) {
		DFAState<T, A, B> s;

		if(isState(state)) {
			s = (DFAState<T, A, B>)state;
			return s.isAccepted();
		} else {
			return false;
		}
	}

}
