/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.utils;

import java.util.Date;

import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.system.IcCardDtoInterface;
import jp.mosp.time.constant.TimeMessageConst;

/**
 * 勤怠打刻機能における有用なメソッドを提供する。<br><br>
 */
public class TimeRecordUtility {
	
	/**
	 * カード読込コマンド。<br>
	 * カード内容に応じて、システム日付で打刻する。<br>
	 */
	public static final String		CMD_READ_CARD			= "ADTR11";
	
	/**
	 * ファイルパス(MosPプラットフォーム基本設定編集領域ヘッダJSP)。
	 */
	public static final String		PATH_IC_EDIT_HEADER_JSP	= "/jsp/addon/timerecorder/icEditHeader.jsp";
	
	/**
	 * JSPファイルパス。<br>
	 */
	public static final String		PATH_TIME_RECODR_JSP	= "/jsp/addon/timerecorder/cardRead.jsp";
	
	/**
	 * カードリーダステータス(始業)。
	 */
	public static final String		CARD_READER_START_WORK	= "01";
	
	/**
	 * カードリーダステータス(終業)。
	 */
	public static final String		CARD_READER_END_WORK	= "02";
	
	/**
	 * カードリーダステータス(休憩入)。
	 */
	public static final String		CARD_READER_START_REST	= "03";
	
	/**
	 * カードリーダステータス(休憩戻)。
	 */
	public static final String		CARD_READER_END_REST	= "04";
	
	/**
	 * カードリーダステータス(カードID登録)。
	 */
	public static final String		CARD_REGIST_IC_CARD		= "05";
	
	/**
	 * パラメータID(対象カードID)。
	 */
	public static final String		PRM_CARD_IC_CARD_ID		= "cid";
	
	/**
	 * パラメータID(カードリーダ打刻時間)。
	 */
	public static final String		PRM_CARD_READER_TIME	= "tim";
	
	/**
	 * パラメータID(カードリーダステータス)。
	 */
	public static final String		PRM_CARD_READER_STATUS	= "sts";
	
	/**
	 * 受付時(ICカード新規登録時)のメッセージコード。<br>
	 */
	public static final String		MSG_REGIST_IC_CARD		= "ADTR001";
	
	/**
	 * カード読み取り失敗字のメッセージコード。<br>
	 */
	public static final String		MSG_FAILED_READ			= "ADTR002";
	
	/**
	 * 既に登録されているため処理できなかった場合のエラーメッセージ。<br>
	 */
	protected static final String	MSG_ALREADY_RECORDED	= "TMW0301";
	
	/**
	 * 開始時刻が登録されていないため処理できなかった場合のエラーメッセージ。<br>
	 */
	protected static final String	MSG_START_NOT_RECORDED	= "TMW0302";
	
	/**
	 * 限度を超えているため処理できなかった場合のエラーメッセージ。<br>
	 */
	protected static final String	MSG_OVER_LIMIT			= "TMW0303";
	
	
	/**
	 * 他クラスからのインスタンス化を防止する。<br>
	 */
	private TimeRecordUtility() {
		// 処理無し
	}
	
	/**
	 * 成功メッセージ(出勤しました)。
	 * @param mospParams MosP処理情報
	 */
	public static void getMessageGoingWork(MospParams mospParams) {
		mospParams.addMessage(TimeMessageConst.MSG_RECORD_TIME, getNameStartWork(mospParams),
				getNameRecordTime(mospParams));
	}
	
	/**
	 * 成功メッセージ(退勤しました)。
	 * @param mospParams MosP処理情報
	 */
	public static void getMessageRetireOffice(MospParams mospParams) {
		mospParams.addMessage(TimeMessageConst.MSG_RECORD_TIME, getNameEndWork(mospParams),
				getNameRecordTime(mospParams));
	}
	
	/**
	 * 成功メッセージ(休憩入りました)。
	 * @param mospParams MosP処理情報
	 */
	public static void getMessageRestTimeInto(MospParams mospParams) {
		mospParams.addMessage(TimeMessageConst.MSG_RECORD_TIME, getNameStartRest(mospParams),
				getNameRecordTime(mospParams));
	}
	
	/**
	 * 成功メッセージ(休憩戻りました)。
	 * @param mospParams MosP処理情報
	 */
	public static void getMessageRestTimeReturn(MospParams mospParams) {
		mospParams.addMessage(TimeMessageConst.MSG_RECORD_TIME, getNameEndRest(mospParams),
				getNameRecordTime(mospParams));
	}
	
	/**
	 * カードIDマスタ情報が無効である場合のエラーメッセージを設定する。
	 * @param dto ICカードマスタ情報
	 * @param mospParams MosP処理情報
	 */
	public static void addIcCardIdInactiveMessage(IcCardDtoInterface dto, MospParams mospParams) {
		String targetDate = DateUtility.getStringDate(dto.getActivateDate());
		String[] aryRep = { mospParams.getName("CardId"), dto.getIcCardId(), targetDate };
		mospParams.addErrorMessage(PlatformMessageConst.MSG_INACTIVE_DAY_CHECK, aryRep);
	}
	
	/**
	 * 打刻失敗メッセージを追加する。
	 * @param mospParams MosP処理情報
	 */
	public static void addRecordTimeFailedMessage(MospParams mospParams) {
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_FAILED, getNameRecordTime(mospParams));
	}
	
	/**
	 * 既に登録されているため打刻できなかった場合のエラーメッセージを追加する。<br>
	 * @param process    打刻対象処理
	 * @param targetDate 打刻対象日
	 * @param mospParams MosP処理情報
	 */
	public static void addAlreadyRecordedErrorMessage(String process, Date targetDate, MospParams mospParams) {
		String[] rep = { DateUtility.getStringDate(targetDate), process, getNameRecordTime(mospParams) };
		mospParams.addErrorMessage(MSG_ALREADY_RECORDED, rep);
	}
	
	/**
	 * 開始時刻が登録されていないため打刻できなかった場合のエラーメッセージを追加する。<br>
	 * @param process    打刻対象処理
	 * @param targetDate 打刻対象日
	 * @param startTime  開始時刻名称
	 * @param mospParams MosP処理情報
	 */
	public static void addStartNotRecordedErrorMessage(String process, Date targetDate, String startTime,
			MospParams mospParams) {
		String[] rep = { DateUtility.getStringDate(targetDate), process, startTime, getNameRecordTime(mospParams) };
		mospParams.addErrorMessage(MSG_START_NOT_RECORDED, rep);
	}
	
	/**
	 * 休憩回数が限度を超えた場合のエラーメッセージを追加する。<br>
	 * @param targetDate 打刻対象日
	 * @param mospParams MosP処理情報
	 */
	public static void addOverLimitErrorMessage(Date targetDate, MospParams mospParams) {
		String[] rep = { DateUtility.getStringDate(targetDate), getNameStartRest(mospParams),
			getNameRecordTime(mospParams) };
		mospParams.addErrorMessage(MSG_OVER_LIMIT, rep);
	}
	
	/**
	 * 打刻名称を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return 打刻名称
	 */
	public static String getNameRecordTime(MospParams mospParams) {
		return mospParams.getName("RecordTime");
	}
	
	/**
	 * 始業名称を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return 始業名称
	 */
	public static String getNameStartWork(MospParams mospParams) {
		return mospParams.getName("StartWork");
	}
	
	/**
	 * 終業名称を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return 終業名称
	 */
	public static String getNameEndWork(MospParams mospParams) {
		return mospParams.getName("EndWork");
	}
	
	/**
	 * 休憩入名称を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return 休憩入名称
	 */
	public static String getNameStartRest(MospParams mospParams) {
		return mospParams.getName("RestTime") + mospParams.getName("Into");
	}
	
	/**
	 * 休憩戻名称を取得する。<br>
	 * @param mospParams MosP処理情報
	 * @return 休憩戻名称
	 */
	public static String getNameEndRest(MospParams mospParams) {
		return mospParams.getName("RestTime") + mospParams.getName("Return");
	}
}
