/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.constant.MospConst;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.workflow.WorkflowIntegrateBeanInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.HolidayInfoReferenceBeanInterface;
import jp.mosp.time.bean.HolidayRequestReferenceBeanInterface;
import jp.mosp.time.bean.RequestUtilBeanInterface;
import jp.mosp.time.bean.ScheduleUtilBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dao.settings.HolidayDaoInterface;
import jp.mosp.time.dao.settings.HolidayDataDaoInterface;
import jp.mosp.time.dao.settings.HolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.HolidayDataDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.HolidayRequestDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;
import jp.mosp.time.utils.TimeUtility;

/**
 * 休暇データ参照クラス。
 */
public class HolidayInfoReferenceBean extends TimeBean implements HolidayInfoReferenceBeanInterface {
	
	/**
	 * 休暇データDAO。
	 */
	private HolidayDataDaoInterface					dao;
	
	/**
	 * 休暇種別マスタDAO。
	 */
	private HolidayDaoInterface						holidayDao;
	
	/**
	 * 休暇申請DAO。
	 */
	private HolidayRequestDaoInterface				holidayRequestDao;
	
	/**
	 * 休暇申請参照。
	 */
	private HolidayRequestReferenceBeanInterface	holidayRequest;
	
	/**
	 * ワークフロー統括
	 */
	private WorkflowIntegrateBeanInterface			workflowIntegrate;
	
	/**
	 * カレンダユーティリティ。
	 */
	private ScheduleUtilBeanInterface				scheduleUtil;
	
	
	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public HolidayInfoReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	public HolidayInfoReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (HolidayDataDaoInterface)createDao(HolidayDataDaoInterface.class);
		holidayDao = (HolidayDaoInterface)createDao(HolidayDaoInterface.class);
		holidayRequestDao = (HolidayRequestDaoInterface)createDao(HolidayRequestDaoInterface.class);
		holidayRequest = (HolidayRequestReferenceBeanInterface)createBean(HolidayRequestReferenceBeanInterface.class);
		workflowIntegrate = (WorkflowIntegrateBeanInterface)createBean(WorkflowIntegrateBeanInterface.class);
		scheduleUtil = (ScheduleUtilBeanInterface)createBean(ScheduleUtilBeanInterface.class);
	}
	
	@Override
	public List<HolidayDataDtoInterface> getHolidayPossibleRequestListForRequest(String personalId, Date targetDate,
			int holidayType) throws MospException {
		List<HolidayDataDtoInterface> list = new ArrayList<HolidayDataDtoInterface>();
		List<HolidayDataDtoInterface> holidayDataList = dao.findForInfoList(personalId, targetDate,
				String.valueOf(MospConst.DELETE_FLAG_OFF), holidayType);
		for (HolidayDataDtoInterface dto : holidayDataList) {
			HolidayDtoInterface holidayDto = holidayDao.findForInfo(dto.getHolidayCode(), dto.getActivateDate(),
					dto.getHolidayType());
			if (holidayDto == null || holidayDto.getInactivateFlag() == MospConst.INACTIVATE_FLAG_ON) {
				continue;
			}
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				// リストに追加
				list.add(dto);
				continue;
			}
			double requestDay = 0;
//			int requestHour = 0;
			// 申請
			List<HolidayRequestDtoInterface> holidayRequestList = holidayRequestDao.findForRequestList(personalId,
					dto.getActivateDate(), holidayType, dto.getHolidayCode(), dto.getActivateDate(),
					dto.getHolidayLimitDate());
			for (HolidayRequestDtoInterface holidayRequestDto : holidayRequestList) {
				if (workflowIntegrate.isFirstReverted(holidayRequestDto.getWorkflow())) {
					// 1次戻の場合
					continue;
				}
				requestDay += holidayRequestDto.getUseDay();
//				requestHour += holidayRequestDto.getUseHour();
			}
			if (dto.getGivingDay() - dto.getCancelDay() - requestDay > 0) {
				dto.setCancelDay(dto.getCancelDay() + requestDay);
				// リストに追加
				list.add(dto);
			}
		}
		return list;
	}
	
	@Override
	public List<HolidayDataDtoInterface> getHolidayPossibleRequestList(String personalId, Date targetDate,
			int holidayType) throws MospException {
		List<HolidayDataDtoInterface> list = new ArrayList<HolidayDataDtoInterface>();
		List<HolidayDataDtoInterface> holidayDataList = dao.findForInfoList(personalId, targetDate,
				String.valueOf(MospConst.DELETE_FLAG_OFF), holidayType);
		for (HolidayDataDtoInterface dto : holidayDataList) {
			HolidayDtoInterface holidayDto = holidayDao.findForInfo(dto.getHolidayCode(), dto.getActivateDate(),
					dto.getHolidayType());
			if (holidayDto == null || holidayDto.getInactivateFlag() == MospConst.INACTIVATE_FLAG_ON) {
				continue;
			}
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				// リストに追加
				list.add(dto);
				continue;
			}
			// 申請
			Map<String, Object> map = holidayRequest.getRequestDayHour(personalId, dto.getActivateDate(), holidayType,
					dto.getHolidayCode(), dto.getActivateDate(), dto.getHolidayLimitDate());
			double requestDay = ((Double)map.get(TimeConst.CODE_REQUEST_DAY)).doubleValue();
			if (dto.getGivingDay() - dto.getCancelDay() - requestDay > 0) {
				dto.setCancelDay(dto.getCancelDay() + requestDay);
				// リストに追加
				list.add(dto);
			}
		}
		return list;
	}
	
	@Override
	public HolidayDataDtoInterface getHolidayPossibleRequestForRequest(String personalId, Date targetDate,
			String holidayCode, int holidayType) throws MospException {
		List<HolidayDataDtoInterface> list = dao.findForEarliestList(personalId, targetDate, holidayCode, holidayType);
		for (HolidayDataDtoInterface dto : list) {
			HolidayDtoInterface holidayDto = holidayDao.findForInfo(dto.getHolidayCode(), dto.getActivateDate(),
					dto.getHolidayType());
			if (holidayDto == null || holidayDto.getInactivateFlag() == MospConst.INACTIVATE_FLAG_ON) {
				return null;
			}
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				return dto;
			}
			double requestDay = 0;
//			int requestHour = 0;
			// 申請
			List<HolidayRequestDtoInterface> holidayRequestList = holidayRequestDao.findForRequestList(personalId,
					dto.getActivateDate(), holidayType, holidayCode, dto.getActivateDate(), dto.getHolidayLimitDate());
			for (HolidayRequestDtoInterface holidayRequestDto : holidayRequestList) {
				if (workflowIntegrate.isFirstReverted(holidayRequestDto.getWorkflow())) {
					// 1次戻の場合
					continue;
				}
				requestDay += holidayRequestDto.getUseDay();
//				requestHour += holidayRequestDto.getUseHour();
			}
			if (dto.getGivingDay() - dto.getCancelDay() - requestDay > 0) {
				dto.setCancelDay(dto.getCancelDay() + requestDay);
				return dto;
			}
		}
		return null;
	}
	
	@Override
	public HolidayDataDtoInterface getHolidayPossibleRequest(String personalId, Date targetDate, String holidayCode,
			int holidayType) throws MospException {
		List<HolidayDataDtoInterface> list = dao.findForEarliestList(personalId, targetDate, holidayCode, holidayType);
		for (HolidayDataDtoInterface dto : list) {
			HolidayDtoInterface holidayDto = holidayDao.findForInfo(dto.getHolidayCode(), dto.getActivateDate(),
					dto.getHolidayType());
			if (holidayDto == null || holidayDto.getInactivateFlag() == MospConst.INACTIVATE_FLAG_ON) {
				return null;
			}
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				return dto;
			}
			// 申請
			Map<String, Object> map = holidayRequest.getRequestDayHour(personalId, dto.getActivateDate(), holidayType,
					holidayCode, dto.getActivateDate(), dto.getHolidayLimitDate());
			double requestDay = ((Double)map.get(TimeConst.CODE_REQUEST_DAY)).doubleValue();
			if (dto.getGivingDay() - dto.getCancelDay() - requestDay > 0) {
				dto.setCancelDay(dto.getCancelDay() + requestDay);
				return dto;
			}
		}
		return null;
	}
	
	@Override
	public double getHolidayRequestDays(String personalId, Date startDate, Date endDate, int holidayRange,
			HolidayDtoInterface holidayDto, HolidayDataDtoInterface holidayDataDto) throws MospException {
		int count = 0;
		List<Date> dateList = TimeUtility.getDateList(startDate, endDate);
		for (Date targetDate : dateList) {
			boolean isDayOff = isDayOff(personalId, targetDate);
			if (mospParams.hasErrorMessage()) {
				return 0;
			}
			if (isDayOff) {
				// 休日の場合
				if (targetDate.equals(startDate) || targetDate.equals(endDate)) {
					// 休暇開始日又は休暇終了日の場合
					addHolidayTargetWorkDateHolidayErrorMessage(targetDate);
					return 0;
				}
				continue;
			}
			count++;
		}
		double holidayRequestDays = 0;
		if (holidayRange == TimeConst.CODE_HOLIDAY_RANGE_ALL) {
			// 全休の場合
			holidayRequestDays = count;
		} else if (holidayRange == TimeConst.CODE_HOLIDAY_RANGE_AM || holidayRange == TimeConst.CODE_HOLIDAY_RANGE_PM) {
			// 午前休又は午後休の場合
			holidayRequestDays = count * TimeConst.HOLIDAY_TIMES_HALF;
		} else {
			mospParams.addErrorMessage(TimeMessageConst.MSG_RANGE_SELECT);
			return 0;
		}
		if (holidayDto.getHolidayType() == TimeConst.CODE_HOLIDAYTYPE_ABSENCE) {
			// 欠勤の場合
			return holidayRequestDays;
		}
		if (holidayDto.getContinuousAcquisition() == 0) {
			// 連続取得が必須の場合
			if (holidayRequestDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay();
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName(), mospParams.getName("Day"));
			return 0;
		} else if (holidayDto.getContinuousAcquisition() == 1) {
			// 連続取得が警告の場合
			if (holidayRequestDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayRequestDays;
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName(), mospParams.getName("Day"));
			return 0;
		} else if (holidayDto.getContinuousAcquisition() == 2) {
			// 連続取得が不要の場合
			if (holidayDto.getNoLimit() == 1) {
				// 付与日数が無制限の場合
				return holidayRequestDays;
			}
			// 付与日数が無制限でない場合
			if (holidayRequestDays <= holidayDataDto.getGivingDay() - holidayDataDto.getCancelDay()) {
				return holidayRequestDays;
			}
			addHolidayNumDaysExcessErrorMessage(holidayDto.getHolidayName(), mospParams.getName("Day"));
			return 0;
		}
		return 0;
	}
	
	@Override
	public boolean hasPersonalApplication(String personalId, Date startDate, Date endDate, int holidayType)
			throws MospException {
		// 個人IDが設定されている、有効日の範囲内で情報を取得
		List<HolidayDataDtoInterface> list = dao.findPersonTerm(personalId, startDate, endDate, holidayType);
		// リスト確認
		if (list.isEmpty()) {
			return false;
		}
		// 期間内全て適用されていたら
		return true;
	}
	
	/**
	 * 休日かどうか確認する。<br>
	 * @param personalId 個人ID
	 * @param date 対象年月日
	 * @return 休日の場合true、そうでない場合false
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	protected boolean isDayOff(String personalId, Date date) throws MospException {
		RequestUtilBeanInterface requestUtil = (RequestUtilBeanInterface)createBean(RequestUtilBeanInterface.class);
		requestUtil.setRequests(personalId, date);
		// 勤務形態コードを取得
		String workTypeCode = scheduleUtil.getScheduledWorkTypeCode(personalId, date);
		if (mospParams.hasErrorMessage()) {
			return false;
		}
		if (TimeConst.CODE_HOLIDAY_LEGAL_HOLIDAY.equals(workTypeCode)
				|| TimeConst.CODE_HOLIDAY_PRESCRIBED_HOLIDAY.equals(workTypeCode)) {
			// 法定休日又は所定休日の場合
			WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = requestUtil.getWorkOnHolidayDto(true);
			if (workOnHolidayRequestDto == null) {
				return true;
			}
			int substitute = workOnHolidayRequestDto.getSubstitute();
			if (substitute == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_ON) {
				// 振替申請する場合
				return false;
			} else if (substitute == TimeConst.CODE_WORK_ON_HOLIDAY_SUBSTITUTE_OFF) {
				// 振替申請しない場合
				addOthersRequestErrorMessage(date, mospParams.getName("WorkingHoliday"));
				return false;
			}
		}
		return false;
	}
	
}
