/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.dao.workflow.WorkflowDaoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.time.base.TimeBean;
import jp.mosp.time.bean.SubHolidayReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.dao.settings.AttendanceDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayDaoInterface;
import jp.mosp.time.dao.settings.SubHolidayRequestDaoInterface;
import jp.mosp.time.dto.settings.AttendanceDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayDtoInterface;
import jp.mosp.time.dto.settings.SubHolidayRequestDtoInterface;

/**
 * 代休データ参照クラス。
 */
public class SubHolidayReferenceBean extends TimeBean implements SubHolidayReferenceBeanInterface {
	
	/**
	 * 代休データDAO。
	 */
	protected SubHolidayDaoInterface		dao;
	
	/**
	 * 代休申請データDAO。
	 */
	protected SubHolidayRequestDaoInterface	subHolidayRequestDao;
	
	/**
	 * 勤怠データDAO。
	 */
	protected AttendanceDaoInterface		attendanceDao;
	
	/**
	 * ワークフローDAO。
	 */
	protected WorkflowDaoInterface			workflowDao;
	
	/**
	 * 深夜代休を利用する際、勤務日から数えた利用可能日数。<br>
	 */
	public static final int					SUB_HOLIDAY_AVAILABLE_DAY	= -10;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public SubHolidayReferenceBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosP処理情報
	 * @param connection DBコネクション
	 */
	public SubHolidayReferenceBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (SubHolidayDaoInterface)createDao(SubHolidayDaoInterface.class);
		subHolidayRequestDao = (SubHolidayRequestDaoInterface)createDao(SubHolidayRequestDaoInterface.class);
		attendanceDao = (AttendanceDaoInterface)createDao(AttendanceDaoInterface.class);
		workflowDao = (WorkflowDaoInterface)createDao(WorkflowDaoInterface.class);
	}
	
	@Override
	public String[][] getSelectArray(String personalId, Date startDate, Date endDate, String holidayRange,
			SubHolidayRequestDtoInterface dto) throws MospException {
		List<SubHolidayDtoInterface> list = new ArrayList<SubHolidayDtoInterface>();
		List<String> holidayRangeList = new ArrayList<String>();
		double subHolidayDays = 0;
		if ("1".equals(holidayRange)) {
			// 全休の場合
			subHolidayDays = 1;
		} else if ("2".equals(holidayRange) || "3".equals(holidayRange)) {
			// 午前休又は午後休の場合
			subHolidayDays = TimeConst.HOLIDAY_TIMES_HALF;
		} else {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		// 一覧取得
		List<SubHolidayDtoInterface> subHolidayList = dao.getSubHolidayList(personalId, startDate, endDate,
				subHolidayDays);
		for (SubHolidayDtoInterface subHolidayDto : subHolidayList) {
			AttendanceDtoInterface attendanceDto = attendanceDao.findForKey(subHolidayDto.getPersonalId(),
					subHolidayDto.getWorkDate(), subHolidayDto.getTimesWork());
			if (attendanceDto == null) {
				continue;
			}
			WorkflowDtoInterface workflowDto = workflowDao.findForKey(attendanceDto.getWorkflow());
			if (workflowDto == null) {
				continue;
			}
			if (!PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				// 勤怠が承認完了でない場合
				continue;
			}
			if (subHolidayDto.getSubHolidayType() == 3) {
				// 深夜代休
				// 深夜代休は勤務日から10日以内であれば利用可能
				if (subHolidayDto.getWorkDate().before(DateUtility.addDay(endDate, SUB_HOLIDAY_AVAILABLE_DAY))) {
					continue;
				}
			}
			double count = 0;
			List<SubHolidayRequestDtoInterface> subHolidayRequestList = subHolidayRequestDao.findForList(personalId,
					subHolidayDto.getWorkDate(), subHolidayDto.getTimesWork(), subHolidayDto.getSubHolidayType());
			for (SubHolidayRequestDtoInterface subHolidayRequestDto : subHolidayRequestList) {
				WorkflowDtoInterface subHolidayRequestWorkflowDto = workflowDao.findForKey(subHolidayRequestDto
					.getWorkflow());
				if (subHolidayRequestWorkflowDto == null) {
					continue;
				}
				if (PlatformConst.CODE_STATUS_DRAFT.equals(subHolidayRequestWorkflowDto.getWorkflowStatus())
						|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(subHolidayRequestWorkflowDto.getWorkflowStatus())) {
					// 代休申請が下書又は取下の場合
					continue;
				}
				if (dto != null) {
					if (subHolidayRequestDto.getTmdSubHolidayRequestId() == dto.getTmdSubHolidayRequestId()) {
						continue;
					}
				}
				int subHolidayRange = subHolidayRequestDto.getHolidayRange();
				if (subHolidayRange == 1) {
					// 全休の場合
					count++;
				} else if (subHolidayRange == 2 || subHolidayRange == 3) {
					// 午前休又は午後休の場合
					count += TimeConst.HOLIDAY_TIMES_HALF;
				}
			}
			double notUseDay = subHolidayDto.getSubHolidayDays() - count;
			if (notUseDay >= subHolidayDays) {
				list.add(subHolidayDto);
				if (notUseDay >= 1) {
					// 残りが全休の場合
					holidayRangeList.add(mospParams.getName("FrontWithCornerParentheses")
							+ mospParams.getName("AllTime") + mospParams.getName("BackWithCornerParentheses"));
				} else if (notUseDay >= TimeConst.HOLIDAY_TIMES_HALF) {
					// 残りが半休の場合
					holidayRangeList.add(mospParams.getName("FrontWithCornerParentheses")
							+ mospParams.getName("HalfTime") + mospParams.getName("BackWithCornerParentheses"));
				} else {
					// 対象データ無し
					return getNoObjectDataPulldown();
				}
			}
		}
		if (list.size() != holidayRangeList.size()) {
			// 対象データ無し
			return getNoObjectDataPulldown();
		}
		String[][] array = prepareSelectArray(list.size(), false);
		int i = 0;
		for (SubHolidayDtoInterface subHolidayDto : list) {
			array[i][0] = DateUtility.getStringDate(subHolidayDto.getWorkDate()) + ","
					+ subHolidayDto.getSubHolidayType();
			StringBuffer sb = new StringBuffer();
			sb.append(mospParams.getName("FrontWithCornerParentheses"));
			int subHolidayType = subHolidayDto.getSubHolidayType();
			if (subHolidayType == TimeConst.CODE_PRESCRIBED_SUBHOLIDAY_CODE) {
				// 所定
				sb.append(mospParams.getName("Prescribed"));
			} else if (subHolidayType == TimeConst.CODE_LEGAL_SUBHOLIDAY_CODE) {
				// 法定
				sb.append(mospParams.getName("Legal"));
			} else if (subHolidayType == TimeConst.CODE_MIDNIGHT_SUBHOLIDAY_CODE) {
				// 深夜
				sb.append(mospParams.getName("Midnight"));
			} else {
				continue;
			}
			sb.append(mospParams.getName("BackWithCornerParentheses"));
			sb.append(getStringDate(subHolidayDto.getWorkDate()));
			sb.append(holidayRangeList.get(i));
			array[i][1] = sb.toString();
			i++;
		}
		return array;
	}
	
}
