/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean;

import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;

/**
 * 時差出勤申請参照インターフェース。
 */
public interface DifferenceRequestReferenceBeanInterface {
	
	/**
	 *  時差出勤申請からレコードを取得する。<br>
	 *  個人ID、申請日で合致するレコードが無い場合、nullを返す。<br>
	 * @param personalId 個人ID
	 * @param requestDate 申請日
	 * @return 時差出勤申請
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	DifferenceRequestDtoInterface findForKeyOnWorkflow(String personalId, Date requestDate) throws MospException;
	
	/**
	 * 時差出勤申請データ取得。
	 * <p>
	 * レコード識別IDから時差出勤申請データを取得。
	 * </p>
	 * @param id レコード識別ID
	 * @return 時差出勤申請データ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	DifferenceRequestDtoInterface findForKey(long id) throws MospException;
	
	/**
	 * 時差出勤申請データ取得する。<br>
	 * 対象個人ID、対象日、勤務回数から時差出勤申請情報を取得する。<br>
	 * @param personalId 対象個人ID
	 * @param requestDate 対象日
	 * @param timesWork 勤務回数
	 * @return 時差出勤申請データ
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	DifferenceRequestDtoInterface findForKey(String personalId, Date requestDate, int timesWork) throws MospException;
	
	/**
	 * 申請日の承認状態を取得する。<br>
	 * 申請日の時差出勤申請が承認完了である場合は「残済 」、
	 * 承認完了でない場合は「休出申 」、
	 * 申請していない場合は「""」を返す。
	 * @param personalId 個人ID
	 * @param requestDate 申請日
	 * @return 承認状態
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String getStatusForRequestDate(String personalId, Date requestDate) throws MospException;
	
	/**
	 *  ワークフロー番号からレコードを取得する。<br>
	 *  ワークフロー番号で合致するレコードが無い場合、nullを返す。<br>
	 * @param workflow ワークフロー番号
	 * @return 時差出勤申請
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	DifferenceRequestDtoInterface findForWorkflow(long workflow) throws MospException;
	
	/**
	 * @param id レコード識別ID
	 * @return 時差出勤DTO
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	DifferenceRequestDtoInterface findForId(long id) throws MospException;
	
	/**
	 * 承認段階、承認状況から時差出勤申請一覧を取得する。<br>
	 * @param personalId 個人ID
	 * @param workflowStage 承認段階
	 * @param workflowStatus 承認状況
	 * @param routeCode ルートコード
	 * @return 時差出勤申請データリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<DifferenceRequestDtoInterface> getListForWorkflowStatus(String personalId, int workflowStage,
			String workflowStatus, String routeCode) throws MospException;
	
	/**
	 * 未承認時差出勤申請データリストを取得する。<br>
	 * @param personalId 個人ID
	 * @param approvalStage 承認段階
	 * @return 未承認時差出勤申請データリスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<DifferenceRequestDtoInterface> getListForNotApproved(String personalId, int approvalStage)
			throws MospException;
	
	/**
	 * 時差出勤申請情報リストを取得する。<br>
	 * 対象個人IDの対象期間における申請を取得する。<br>
	 * @param personalId 対象個人ID
	 * @param firstDate  対象期間初日
	 * @param lastDate   対象期間最終日
	 * @return 時差出勤申請情報リスト
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	List<DifferenceRequestDtoInterface> getDifferenceRequestList(String personalId, Date firstDate, Date lastDate)
			throws MospException;
	
	/**
	 * 時差出勤時刻を取得する。<br>
	 * @param dto 時差申請DTO
	 * @return 時差出勤時刻
	 */
	String getDifferenceTime(DifferenceRequestDtoInterface dto);
	
	/**
	 * 時差出勤のプルダウンの取得
	 * @param type 対象コード
	 * @return 時差出勤のプルダウン
	 */
	String[][] getDifferenceSelectArray(String type);
	
	/**
	 * @param personalId 対象個人ID
	 * @param requestDate 対象年月日
	 * @return 時差出勤区分の略称。時差出勤申請のデータが存在しない場合null
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	String getDifferenceAbbr(String personalId, Date requestDate) throws MospException;
	
	/**
	 * @param type 時差出勤区分
	 * @return 時差出勤区分の略称。時差出勤申請のデータが存在しない場合null
	 */
	String getDifferenceAbbr(String type);
	
	/**
	 * 基本情報チェック
	 * @param personalId 個人ID
	 * @param targetDate 対象日
	 * @throws MospException インスタンスの取得或いはSQL実行に失敗した場合
	 */
	void chkBasicInfo(String personalId, Date targetDate) throws MospException;
}
