/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * 
 */
package jp.mosp.time.bean.impl;

import java.sql.Connection;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.time.bean.AllowanceRegistBeanInterface;
import jp.mosp.time.dao.settings.AllowanceDaoInterface;
import jp.mosp.time.dto.settings.AllowanceDtoInterface;
import jp.mosp.time.dto.settings.impl.TmdAllowanceDto;

/**
 * 勤怠データ手当情報登録クラス。
 */
public class AllowanceRegistBean extends PlatformBean implements AllowanceRegistBeanInterface {
	
	/**
	 * 手当マスタDAOクラス。<br>
	 */
	AllowanceDaoInterface	dao;
	

	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 */
	public AllowanceRegistBean() {
		super();
	}
	
	/**
	 * {@link PlatformBean#PlatformBean(MospParams, Connection)}を実行する。<br>
	 * @param mospParams MosPパラメータクラス
	 * @param connection DBコネクション
	 */
	public AllowanceRegistBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	@Override
	public void initBean() throws MospException {
		dao = (AllowanceDaoInterface)createDao(AllowanceDaoInterface.class);
	}
	
	@Override
	public AllowanceDtoInterface getInitDto() {
		return new TmdAllowanceDto();
	}
	
	@Override
	public void insert(AllowanceDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 新規登録情報の検証
		checkInsert(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdAllowanceId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	@Override
	public void update(AllowanceDtoInterface dto) throws MospException {
		// DTOの妥当性確認
		validate(dto);
		if (mospParams.hasErrorMessage()) {
			return;
		}
		// ロック対象テーブルの追加
		addTargetTable(dao.getTable(dao.getClass()), true);
		// ロック開始
		lockTables();
		// 履歴更新情報の検証
		checkUpdate(dto);
		if (mospParams.hasErrorMessage()) {
			// ロック解除
			unLockTable();
			return;
		}
		// 論理削除
		logicalDelete(dao, dto.getTmdAllowanceId());
		// レコード識別ID最大値をインクリメントしてDTOに設定
		dto.setTmdAllowanceId(findForMaxId(dao) + 1);
		// 登録処理
		dao.insert(dto);
		// ロック解除
		unLockTable();
	}
	
	/**
	 * 新規登録時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkInsert(AllowanceDtoInterface dto) throws MospException {
		// 対象レコードの有効日が重複していないかを確認
		checkDuplicateInsert(dao.findForHistory(dto.getPersonalId(), dto.getWorkDate(), dto.getWorks(), dto
			.getAllowanceCode(), dto.getAllowance()));
	}
	
	/**
	 * 履歴更新時の確認処理を行う。<br>
	 * @param dto 対象DTO
	 * @throws MospException SQLの作成に失敗した場合、或いはSQL例外が発生した場合
	 */
	protected void checkUpdate(AllowanceDtoInterface dto) throws MospException {
		// 対象レコード識別IDのデータが削除されていないかを確認
		checkExclusive(dao, dto.getTmdAllowanceId());
	}
	
	/**
	 * 登録情報の妥当性を確認する。
	 * @param dto 対象DTO
	 */
	protected void validate(AllowanceDtoInterface dto) {
		// TODO 保留
	}
	
}
