/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.time.base;

import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.framework.utils.MospUtility;
import jp.mosp.platform.base.PlatformAction;
import jp.mosp.platform.bean.human.HumanReferenceBeanInterface;
import jp.mosp.platform.constant.PlatformConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.human.HumanDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowCommentDtoInterface;
import jp.mosp.platform.dto.workflow.WorkflowDtoInterface;
import jp.mosp.platform.human.constant.PlatformHumanConst;
import jp.mosp.time.bean.ApplicationReferenceBeanInterface;
import jp.mosp.time.bean.ScheduleDateReferenceBeanInterface;
import jp.mosp.time.bean.ScheduleReferenceBeanInterface;
import jp.mosp.time.bean.WorkTypeReferenceBeanInterface;
import jp.mosp.time.constant.TimeConst;
import jp.mosp.time.constant.TimeMessageConst;
import jp.mosp.time.dto.settings.ApplicationDtoInterface;
import jp.mosp.time.dto.settings.DifferenceRequestDtoInterface;
import jp.mosp.time.dto.settings.HolidayDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDateDtoInterface;
import jp.mosp.time.dto.settings.ScheduleDtoInterface;
import jp.mosp.time.dto.settings.SubstituteDtoInterface;
import jp.mosp.time.dto.settings.WorkOnHolidayRequestDtoInterface;

/**
 * MosP勤怠管理におけるActionの基本機能を提供する。<br>
 */
public abstract class TimeAction extends PlatformAction {
	
	/**
	 * ファイルパス(MosP勤怠管理用JavaScript)。
	 */
	public static final String					PATH_TIME_JS	= "/pub/time/base/js/time.js";
	
	/**
	 * ファイルパス(MosP勤怠管理用JavaScript)。
	 */
	public static final String					PATH_TIME_CSS	= "/pub/time/base/css/time.css";
	
	/**
	 * MosP勤怠管理用BeanHandler。
	 */
	protected TimeBeanHandlerInterface			time;
	
	/**
	 * MosP勤怠管理参照用BeanHandler。
	 */
	protected TimeReferenceBeanHandlerInterface	timeReference;
	

	@Override
	protected void addBaseJsCssFiles() {
		super.addBaseJsCssFiles();
		// MosP勤怠管理用JavaScriptファイル追加
		mospParams.addJsFile(PATH_TIME_JS);
		// MosP勤怠管理用CSSファイル追加
		mospParams.addCssFile(PATH_TIME_CSS);
	}
	
	/**
	 * MosP勤怠管理用BeanHandlerを取得する。<br>
	 * @return MosP勤怠管理用BeanHandler
	 * @throws MospException インスタンスの取得に失敗した場合
	 */
	protected TimeBeanHandlerInterface time() throws MospException {
		// MosP勤怠管理用BeanHandler存在確認
		if (time != null) {
			return time;
		}
		// MosP勤怠管理用BeanHandler取得
		time = (TimeBeanHandlerInterface)createHandler(TimeBeanHandlerInterface.class);
		return time;
	}
	
	/**
	 * MosP勤怠管理参照用BeanHandlerを取得する。<br>
	 * @return MosP勤怠管理参照用BeanHandler
	 * @throws MospException インスタンスの取得に失敗した場合
	 */
	protected TimeReferenceBeanHandlerInterface timeReference() throws MospException {
		// MosP勤怠管理参照用BeanHandler存在確認
		if (timeReference != null) {
			return timeReference;
		}
		// MosP勤怠管理参照用BeanHandler取得
		timeReference = (TimeReferenceBeanHandlerInterface)createHandler(TimeReferenceBeanHandlerInterface.class);
		return timeReference;
	}
	
	/**
	 * VOから有効日(編集)を取得する。<br>
	 * @return 有効日(編集)
	 */
	protected Date getEditActivateDate() {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 有効日取得
		return getDate(vo.getTxtEditActivateYear(), vo.getTxtEditActivateMonth(), vo.getTxtEditActivateDay());
	}
	
	/**
	 * VOから有効日(検索)を取得する。<br>
	 * @return 有効日(検索)
	 */
	protected Date getSearchActivateDate() {
		// VO準備
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 有効日取得
		return getDate(vo.getTxtSearchActivateYear(), vo.getTxtSearchActivateMonth(), vo.getTxtSearchActivateDay());
	}
	
	/**
	 * VOから有効日(一括更新)を取得する。<br>
	 * @return 有効日(検索)
	 */
	protected Date getUpdateActivateDate() {
		// VO準備
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 有効日取得
		return getDate(vo.getTxtUpdateActivateYear(), vo.getTxtUpdateActivateMonth(), vo.getTxtUpdateActivateDay());
	}
	
	/**
	 * 検索有効日を設定する。<br>
	 * @param date 設定する日付
	 */
	protected void setSearchActivateDate(Date date) {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 設定日付確認
		if (date == null) {
			// nullの場合
			vo.setTxtSearchActivateYear("");
			vo.setTxtSearchActivateMonth("");
			vo.setTxtSearchActivateDay("");
		}
		// 検索有効日設定
		vo.setTxtSearchActivateYear(getStringYear(date));
		vo.setTxtSearchActivateMonth(getStringMonth(date));
		vo.setTxtSearchActivateDay(getStringDay(date));
	}
	
	/**
	 * VOから有効日を取得する。<br>
	 * @return 有効日
	 */
	protected Date getActivateDate() {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 有効日取得
		return getDate(vo.getTxtActivateYear(), vo.getTxtActivateMonth(), vo.getTxtActivateDay());
	}
	
	/**
	 * 年プルダウン取得
	 * @param editYear 取得したサーバ日付
	 * @return 年プルダウン用文字列配列
	 */
	public String[][] getYearArray(String editYear) {
		final int year = 5;
		final int argYear = (Integer.valueOf(editYear) - 3);
		String[][] aryYear = new String[year][2];
		for (int i = 0; i < year; i++) {
			aryYear[i][0] = String.valueOf(i + argYear);
			aryYear[i][1] = String.valueOf(i + argYear);
		}
		return aryYear;
	}
	
	/**
	 * 年プルダウン取得
	 * @param year 年
	 * @return 年プルダウン用文字列配列
	 */
	protected String[][] getYearArray(int year) {
		int y = year - 3;
		String[][] aryYear = new String[5][2];
		for (int i = 0; i < aryYear.length; i++) {
			String stringYear = Integer.toString(y + i);
			aryYear[i][0] = stringYear;
			aryYear[i][1] = stringYear;
		}
		return aryYear;
	}
	
	/**
	 * 年プルダウン取得
	 * @param year 基準年
	 * @param former 過去
	 * @param further 未来
	 * @return 年プルダウン用文字列配列
	 */
	protected String[][] getYearArray(int year, int former, int further) {
		int y = year - former;
		String[][] aryYear = new String[former + further + 1][2];
		for (int i = 0; i < aryYear.length; i++) {
			String stringYear = Integer.toString(y + i);
			aryYear[i][0] = stringYear;
			aryYear[i][1] = stringYear;
		}
		return aryYear;
	}
	
	/**
	 * 月プルダウン取得
	 * @return 月プルダウン用文字列配列
	 */
	public String[][] getMonthArray() {
		final int month = 12;
		String[][] aryMonth = new String[month][2];
		for (int i = 0; i < month; i++) {
			aryMonth[i][0] = String.valueOf(i + 1);
			aryMonth[i][1] = String.valueOf(i + 1);
		}
		return aryMonth;
	}
	
	/**
	 * 日プルダウン取得
	 * @param day プルダウンの最終日付
	 * @return 日プルダウン用文字列配列
	 */
	public String[][] getDayArray(int day) {
		String[][] aryDay = new String[day][2];
		for (int i = 0; i < day; i++) {
			aryDay[i][0] = String.valueOf(i + 1);
			aryDay[i][1] = String.valueOf(i + 1);
		}
		return aryDay;
	}
	
	/**
	 * 時プルダウン取得
	 * @param maxHour 最大時間
	 * @return 0 から maxHour - 1 までの時プルダウン用文字列配列
	 */
	public String[][] getHourArray(int maxHour) {
		if (maxHour < 1) {
			// 最大時間が1未満の場合は1とする
			return getHourArray(1);
		}
		String[][] aryHour = new String[maxHour][2];
		for (int i = 0; i < aryHour.length; i++) {
			String stringHour = Integer.toString(i);
			aryHour[i][0] = stringHour;
			aryHour[i][1] = stringHour;
		}
		return aryHour;
	}
	
	/**
	 * 時プルダウン取得
	 * @return 時プルダウン用文字列配列
	 */
	public String[][] getHourArray() {
		final int hour = 24;
		String[][] aryHour = new String[hour][2];
		for (int i = 0; i < hour; i++) {
			aryHour[i][0] = String.valueOf(i);
			aryHour[i][1] = String.valueOf(i);
		}
		return aryHour;
	}
	
	/**
	 * 分プルダウン取得
	 * @param interval 間隔
	 * @return 分プルダウン用文字列配列
	 */
	public String[][] getMinuteArray(int interval) {
		if (interval <= 0) {
			// 間隔が0以下の場合は1分間隔とする
			return getMinuteArray(1);
		} else if (interval > TimeConst.CODE_DEFINITION_HOUR) {
			// 間隔が60を超える場合は60分間隔とする(プルダウンに表示されるのは「0」のみ)
			return getMinuteArray(TimeConst.CODE_DEFINITION_HOUR);
		}
		String[][] aryMinute = new String[TimeConst.CODE_DEFINITION_HOUR / interval][2];
		for (int i = 0; i < aryMinute.length; i++) {
			String stringMinute = Integer.toString(i * interval);
			aryMinute[i][0] = stringMinute;
			aryMinute[i][1] = stringMinute;
		}
		return aryMinute;
	}
	
	/**
	 * 最終日取得処理
	 * @param year 検索該当年
	 * @param month 検索該当月
	 * @return 該当年月の最終日付を返す
	 */
	public int getLastDay(int year, int month) {
		Calendar cal = Calendar.getInstance();
		cal.set(year, month, 0);
		return cal.get(Calendar.DATE);
	}
	
	/**
	 * 曜日取得処理
	 * @param year 検索該当年
	 * @param month 検索該当月
	 * @param day 検索該当日
	 * @return 曜日を返す
	 */
	@Deprecated
	public String getWeek(int year, int month, int day) {
		String[] week = { mospParams.getName("SundayAbbr"), mospParams.getName("MondayAbbr"),
			mospParams.getName("TuesdayAbbr"), mospParams.getName("WednesdayAbbr"), mospParams.getName("ThursdayAbbr"),
			mospParams.getName("FridayAbbr"), mospParams.getName("SaturdayAbbr") };
		Calendar cal = Calendar.getInstance();
		cal.set(year, month - 1, day);
		return week[(cal.get(Calendar.DAY_OF_WEEK) - 1)];
	}
	
	/**
	 * int型の時間をint型の時間(時)に変換
	 * @param time 時間
	 * @return 時間(時)
	 */
	protected int convIntegerTimeToIntegerHour(int time) {
		return time / TimeConst.CODE_DEFINITION_HOUR;
	}
	
	/**
	  * Int型の時間をString型の時間（時）に変換
	  * @param time 時刻
	  * @return 時間（時）
	  */
	protected String convIntegerTimeToStringHour(int time) {
		return String.valueOf(convIntegerTimeToIntegerHour(time));
	}
	
	/**
	 * int型の時間をint型の時間(分)に変換
	 * @param time 時間
	 * @return 時間(分)
	 */
	protected int convIntegerTimeToIntegerMinute(int time) {
		return Math.abs(time) % TimeConst.CODE_DEFINITION_HOUR;
	}
	
	/**
	 * Int型の時間をString型の時間（分）に変換
	 * @param time 時刻
	 * @return 時間（分）
	 */
	protected String convIntegerTimeToStringMinutes(int time) {
		int minute = convIntegerTimeToIntegerMinute(time);
		StringBuffer sb = new StringBuffer();
		if (minute < 10) {
			sb.append(0);
		}
		sb.append(minute);
		return sb.toString();
	}
	
	/**
	 * 対象個人IDの対象日における人事情報を取得し、VOに設定する。<br>
	 * @param personalId 対象個人ID
	 * @param targetDate 対象日
	 * @throws MospException 人事情報の取得に失敗した場合
	 */
	protected void setEmployeeInfo(String personalId, Date targetDate) throws MospException {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 個人IDを設定
		vo.setPersonalId(personalId);
		// 対象日を設定
		vo.setTargetDate(targetDate);
		// 人事情報取得及び確認
		HumanDtoInterface humanDto = reference().human().getHumanInfo(personalId, targetDate);
		if (humanDto == null) {
			// エラーメッセージ設定
			addHumanHistoryNotExistErrorMessage(targetDate);
			return;
		}
		// 社員コードを設定
		vo.setLblEmployeeCode(humanDto.getEmployeeCode());
		// 社員名を設定
		vo.setLblEmployeeName(getLastFirstName(humanDto.getLastName(), humanDto.getFirstName()));
		// 所属名を設定
		vo.setLblSectionName(reference().section().getSectionName(humanDto.getSectionCode(), targetDate));
	}
	
	/**
	 * 対象個人IDを取得する。<br>
	 * 画面遷移用のパラメータをMosP処理情報から取得する。<br>
	 * @return 対象個人ID
	 */
	protected String getTargetPersonalId() {
		return (String)mospParams.getGeneralParam(TimeConst.PRM_TARGET_PERSONAL_ID);
	}
	
	/**
	 * 対象日を取得する。<br>
	 * 画面遷移用のパラメータをMosP処理情報から取得する。<br>
	 * @return 対象日
	 */
	protected Date getTargetDate() {
		return (Date)mospParams.getGeneralParam(TimeConst.PRM_TARGET_DATE);
	}
	
	/**
	 * 対象年を取得する。<br>
	 * 画面遷移用のパラメータをMosP処理情報から取得する。<br>
	 * @return 対象年
	 */
	protected int getTargetYear() {
		return (Integer)mospParams.getGeneralParam(TimeConst.PRM_TARGET_YEAR);
	}
	
	/**
	 * 対象月を取得する。<br>
	 * 画面遷移用のパラメータをMosP処理情報から取得する。<br>
	 * @return 対象月
	 */
	protected int getTargetMonth() {
		return (Integer)mospParams.getGeneralParam(TimeConst.PRM_TARGET_MONTH);
	}
	
	/**
	 * 対象ワークフロー番号を取得する。<br>
	 * 画面遷移用のパラメータをMosP処理情報から取得する。<br>
	 * @return 対象ワークフロー番号
	 */
	protected long getTargetWorkflow() {
		return (Long)mospParams.getGeneralParam(TimeConst.PRM_TARGET_WORKFLOW);
	}
	
	/**
	 * 対象個人IDを設定する。<br>
	 * 画面遷移用のパラメータをMosP処理情報に設定する。<br>
	 * @param targetPersonalId 対象個人ID
	 */
	protected void setTargetPersonalId(String targetPersonalId) {
		mospParams.addGeneralParam(TimeConst.PRM_TARGET_PERSONAL_ID, targetPersonalId);
	}
	
	/**
	 * 対象日を設定する。<br>
	 * 画面遷移用のパラメータをMosP処理情報に設定する。<br>
	 * @param targetDate 対象日
	 */
	protected void setTargetDate(Date targetDate) {
		mospParams.addGeneralParam(TimeConst.PRM_TARGET_DATE, targetDate);
	}
	
	/**
	 * 対象年を設定する。<br>
	 * 画面遷移用のパラメータをMosP処理情報に設定する。<br>
	 * @param targetYear 対象年
	 */
	protected void setTargetYear(int targetYear) {
		mospParams.addGeneralParam(TimeConst.PRM_TARGET_YEAR, targetYear);
	}
	
	/**
	 * 対象月を設定する。<br>
	 * 画面遷移用のパラメータをMosP処理情報に設定する。<br>
	 * @param targetMonth 対象月
	 */
	protected void setTargetMonth(int targetMonth) {
		mospParams.addGeneralParam(TimeConst.PRM_TARGET_MONTH, targetMonth);
	}
	
	/**
	 * 対象ワークフロー番号を設定する。<br>
	 * 画面遷移用のパラメータをMosP処理情報に設定する。<br>
	 * @param targetWorkflow 対象ワークフロー番号
	 */
	protected void setTargetWorkflow(long targetWorkflow) {
		mospParams.addGeneralParam(TimeConst.PRM_TARGET_WORKFLOW, targetWorkflow);
	}
	
	/**
	 * リクエストされた譲渡Actionクラス名を取得する。
	 * @return 譲渡Actionクラス名
	 */
	protected String getTransferredAction() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_ACTION);
	}
	
	/**
	 * リクエストされた譲渡インデックスを取得する。
	 * @return 譲渡インデックス
	 */
	protected int getTransferredIndex() {
		return getInt(mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_INDEX));
	}
	
	/**
	 * リクエストされた譲渡社員コードを取得する。
	 * @return 譲渡社員コード
	 */
	protected String getTransferredEmployeeCode() {
		return mospParams.getRequestParam(PlatformHumanConst.PRM_TRANSFERRED_EMPLOYEE_CODE);
	}
	
	/**
	 * リクエストされた譲渡汎用区分を取得する。
	 * @return 譲渡汎用区分
	 */
	@Override
	protected String getTransferredType() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_TYPE);
	}
	
	/**
	 * リクエストされた譲渡汎用月を取得する。
	 * @return 譲渡汎用月
	 */
	protected String getTransferredMonth() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_MONTH);
	}
	
	/**
	 * リクエストされた譲渡汎用区分を取得する。
	 * @return 譲渡汎用コード
	 */
	protected String getTransferredGenericCode() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_GENERIC_CODE);
	}
	
	/**
	 * 数字フォーマット
	 * @param number 数字
	 * @param fraction 小数点以下最大桁数
	 * @return 数字文字列
	 */
	protected String getNumberString(Double number, int fraction) {
		if (number == null) {
			return mospParams.getName("Hyphen");
		}
		return getNumberString(number.doubleValue(), fraction);
	}
	
	/**
	 * 数字フォーマット
	 * @param number 数字
	 * @param fraction 小数点以下最大桁数
	 * @return 時間(文字列)
	 */
	public String getNumberString(double number, int fraction) {
		String IntFcn = "0";
		for (int i = 1; i < fraction; i++) {
			IntFcn = IntFcn + "0";
		}
		DecimalFormat numFormat = new DecimalFormat("0" + mospParams.getName("Dot") + IntFcn);
		return numFormat.format(number);
	}
	
	/**
	  * プルダウンの先頭行に任意の値を追加
	  * @param target 対象プルダウン配列
	  * @param code 対象行コード
	  * @param value 対象行表示値
	  * @return aryNew 生成配列
	  */
	@Deprecated
	public static String[][] addArrayFirstRowValue(String[][] target, String code, String value) {
		String[][] aryNew = new String[target.length + 1][2];
		for (int i = 0; i < aryNew.length; i++) {
			if (i == 0) {
				aryNew[0][0] = code;
				aryNew[0][1] = value;
			} else {
				aryNew[i][0] = target[i - 1][0];
				aryNew[i][1] = target[i - 1][1];
			}
		}
		return aryNew;
	}
	
	/**
	  * プルダウンの先頭行に空白を追加
	  * @param target 対象プルダウン配列
	  * @return 追加後プルダウン
	  */
	@Deprecated
	public static String[][] addArrayFirstRowBlank(String[][] target) {
		return addArrayFirstRowValue(target, "", "");
	}
	
	/**
	 * 時分形式で出力する。
	 * @param time 分
	 * @return HH.MM
	 */
	protected String toTimeDotFormatString(Integer time) {
		if (time == null) {
			return mospParams.getName("Hyphen");
		}
		return getTimeDotFormat(time.intValue());
	}
	
	/**
	 * 一覧時分形式で出力する。
	 * @param time 分
	 * @return HH.MM
	 */
	protected String getTimeDotFormat(int time) {
		StringBuffer sb = new StringBuffer();
		sb.append(convIntegerTimeToIntegerHour(time));
		sb.append(mospParams.getName("Dot"));
		sb.append(convIntegerTimeToStringMinutes(time));
		return sb.toString();
	}
	
	/**
	 * 時分形式で出力する。
	 * @param time 分
	 * @return HH時間MM分
	 */
	protected String getTimeTimeFormat(int time) {
		StringBuffer sb = new StringBuffer();
		sb.append(convIntegerTimeToIntegerHour(time));
		sb.append(mospParams.getName("Time"));
		sb.append(convIntegerTimeToStringMinutes(time));
		sb.append(mospParams.getName("Minutes"));
		return sb.toString();
	}
	
	/**
	 * 時分形式で出力する。
	 * @param date1 前時間
	 * @param date2 後時間
	 * @return hh:mm～hh:mm
	 */
	protected String getTimeWaveFormat(Date date1, Date date2) {
		if (date1 == null || date2 == null) {
			return "";
		}
		if (date1.compareTo(date2) <= 0) {
			return DateUtility.getStringTime(date1) + mospParams.getName("Wave") + DateUtility.getStringTime(date2);
		} else {
			// 終了時間が次の日になっているので時間の調整を行う。
			int afterHour = DateUtility.getHour(date2) + 24;
			String afterMinute = DateUtility.getStringMinute(date2);
			return DateUtility.getStringTime(date1) + mospParams.getName("Wave") + String.valueOf(afterHour)
					+ mospParams.getName("HalfColon") + afterMinute;
		}
	}
	
	/**
	 * 時分形式で出力する。
	 * @param date1 前時間
	 * @param date2 後時間
	 * @return hh:mm～hh:mm
	 */
	protected String getTimeWaveFormat(String date1, String date2) {
		StringBuffer sb = new StringBuffer();
		if (date1 != null) {
			sb.append(date1);
		}
		sb.append(mospParams.getName("Wave"));
		if (date2 != null) {
			sb.append(date2);
		}
		return sb.toString();
	}
	
	/**
	 * 時分形式で出力する。
	 * @param time 分
	 * @return HH:MM
	 */
	protected String getTimeColonFormat(int time) {
		StringBuffer sb = new StringBuffer();
		sb.append(convIntegerTimeToIntegerHour(time));
		sb.append(mospParams.getName("HalfColon"));
		sb.append(convIntegerTimeToStringMinutes(time));
		return sb.toString();
	}
	
	/**
	 * numberの値が0の場合は"-"を設定する。<br>
	 * @param number 数字
	 * @param fraction 小数点以下最大桁数
	 * @return 変換後の文字列
	 */
	protected String setNumberString(double number, int fraction) {
		if (number == 0) {
			// 数字が0の場合は"-"を返す
			return mospParams.getName("Hyphen");
		}
		return getNumberString(number, fraction);
	}
	
	/**
	 * 下書成功メッセージの設定。
	 */
	protected void addDraftMessage() {
		String[] aryMeassage = { mospParams.getName("WorkPaper") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * 申請成功メッセージの設定。
	 */
	protected void addAppliMessage() {
		String[] aryMeassage = { mospParams.getName("Application") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * 申請成功メッセージの設定。
	 */
	protected void addTakeDownMessage() {
		String[] aryMeassage = { mospParams.getName("TakeDown") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * 項目の値のを設定する。<br>
	 * @param time 時間
	 * @return - 又は数値
	 */
	public String getValueHyphenView(int time) {
		if (time == 0) {
			return mospParams.getName("Hyphen");
		}
		return getTimeColonFormat(time);
	}
	
	/**
	 * 残業申請の区分名設定
	 * @param time 区分
	 * @return 区分名
	 */
	protected String getOvertimeTypeName(int time) {
		if (1 == time) {
			return mospParams.getName("Work") + mospParams.getName("Ahead");
		} else {
			return mospParams.getName("Work") + mospParams.getName("Later");
		}
	}
	
	/**
	 * 代休申請の区分名設定
	 * @param time 区分
	 * @return 区分名
	 */
	protected String getSubHolidayTypeName(int time) {
		if (1 == time) {
			return mospParams.getName("Whole");
		} else if (2 == time) {
			return mospParams.getName("FrontTime");
		} else {
			return mospParams.getName("BackTime");
		}
	}
	
	/**
	 * @param type 休暇区分
	 * @return 文字列に変換した休暇区分
	 */
	protected String setWorkDateSubHolidayType(int type) {
		if (type == 1) {
			// 所定
			return mospParams.getName("Prescribed");
		} else if (type == 2) {
			// 法定
			return mospParams.getName("Legal");
		} else if (type == 3) {
			// 深夜
			return mospParams.getName("Midnight");
		}
		return "";
	}
	
	/**
	 * 時差出勤申請の区分名設定
	 * @param type 区分
	 * @return 区分名
	 */
	protected String getDifferenceTypeName(String type) {
		if (TimeConst.CODE_DIFFERENCE_TYPE_A.equals(type)) {
			return mospParams.getName("CharaA");
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_B.equals(type)) {
			return mospParams.getName("CharaB");
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_C.equals(type)) {
			return mospParams.getName("CharaC");
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_D.equals(type)) {
			return mospParams.getName("CharaD");
		} else if (TimeConst.CODE_DIFFERENCE_TYPE_S.equals(type)) {
			return mospParams.getName("CharaS");
		}
		return type;
	}
	
	/**
	 * ワークフローの状態と段階からワークフロー状態(表示用)を取得する。<br>
	 * @param status ワークフロー状態
	 * @param stage  ワークフロー段階
	 * @return ワークフロー状態(表示用)
	 */
	public String getStatusStageValueView(String status, int stage) {
		try {
			return reference().workflowIntegrate().getWorkflowStatus(status, stage);
		} catch (MospException e) {
			return status;
		}
	}
	
	/**
	 * 苗字と名前を受け取りスペースを挿入した名前を返す。<br>
	 * @param lastName 名
	 * @param firstName 姓
	 * @return スペースを挿入したフルネーム
	 */
	public String getLastFirstName(String lastName, String firstName) {
		return MospUtility.getHumansName(firstName, lastName);
	}
	
	/**
	 * 修正情報を取得する。<br>
	 * @param correctionEmployeeCode 修正社員コード
	 * @param employeeCode 社員コード
	 * @return 修正情報（本/他）
	 */
	public String getCorrectionInfo(String correctionEmployeeCode, String employeeCode) {
		if (correctionEmployeeCode == null || correctionEmployeeCode.equals(employeeCode)) {
			return mospParams.getName("CorrectionHistory");
		} else {
			return mospParams.getName("Other");
		}
	}
	
	/**
	 * classで使用する文字列の設定。<br>
	 * @param dateTimeNum 一覧表示する日 or 時間
	 * @return dateTimeNum判定後の文字色設定タグ
	 */
	protected String setHistoryDateTimeColor(int dateTimeNum) {
		// 初期値に文字色黒を設定する
		String claSpan = "style=\"color: black\"";
		if (dateTimeNum < 0) {
			// dateTimeNumが0未満（マイナス）の場合は赤文字を設定する
			claSpan = "style=\"color: red\"";
		}
		return claSpan;
	}
	
	/**
	 * classで使用する文字列の設定。<br>
	 * @param dateTimeNum 一覧表示する日 or 時間
	 * @return dateTimeNum判定後の文字色設定タグ
	 */
	protected String setHistoryDateTimeDoubleColor(double dateTimeNum) {
		// 初期値に文字色黒を設定する
		String claSpan = "style=\"color: black\"";
		if (dateTimeNum < 0) {
			// dateTimeNumが0未満（マイナス）の場合は赤文字を設定する
			claSpan = "style=\"color: red\"";
		}
		return claSpan;
	}
	
	/**
	 * @param number 日にち
	 * @return 時間(文字列) + 日
	 */
	protected String getFormatRestDay(double number) {
		return getNumberString(number, 1) + mospParams.getName("Day");
	}
	
	/**
	 * カテゴリから表示する文字列を返す。<br>
	 * @param requestType カテゴリ
	 * @return 一覧表示用文字列
	 */
	public String getRequestTypeValueView(String requestType) {
		String displayStatus = "";
		if (requestType.equals(TimeConst.CODE_FUNCTION_WORK_MANGE)) {
			displayStatus = mospParams.getName("WorkManage");
		} else if (requestType.equals(TimeConst.CODE_FUNCTION_OVER_WORK)) {
			displayStatus = mospParams.getName("OvertimeWork");
		} else if (requestType.equals(TimeConst.CODE_FUNCTION_VACATION)) {
			displayStatus = mospParams.getName("Vacation");
		} else if (requestType.equals(TimeConst.CODE_FUNCTION_WORK_HOLIDAY)) {
			displayStatus = mospParams.getName("WorkingHoliday");
		} else if (requestType.equals(TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY)) {
			displayStatus = mospParams.getName("CompensatoryHoliday");
		} else if (requestType.equals(TimeConst.CODE_FUNCTION_DIFFERENCE)) {
			displayStatus = mospParams.getName("TimeDifference");
		}
		return displayStatus;
	}
	
	/**
	 * 承認成功メッセージの設定。
	 */
	protected void addApprovalMessage() {
		String[] aryMeassage = { mospParams.getName("Approval") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * 差戻成功メッセージの設定。
	 */
	protected void addSendingBackMessage() {
		String[] aryMeassage = { mospParams.getName("SendingBack") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * Time型で渡された引数から～時～分フォーマットに変換した文字列を返す。
	 * @param date 変換対象時間
	 * @return 変換した文字列
	 */
	protected String getStringTimeMinutes(Date date) {
		if (date == null) {
			return "";
		}
		return getStringHour(date) + mospParams.getName("Hour") + getStringMinute(date) + mospParams.getName("Minutes");
	}
	
	/**
	 * @param type1 休暇区分1
	 * @param type2 休暇区分2
	 * @return 文字列に変換した休暇区分を返す
	 */
	protected String getHolidayType1Name(int type1, String type2) {
		if (type1 == 1) {
			if (Integer.toString(1).equals(type2)) {
				// 有給休暇
				StringBuffer sb = new StringBuffer();
				sb.append(mospParams.getName("Salaried"));
				sb.append(mospParams.getName("Vacation"));
				return sb.toString();
			} else if (Integer.toString(2).equals(type2)) {
				// ストック休暇
				StringBuffer sb = new StringBuffer();
				sb.append(mospParams.getName("Stock"));
				sb.append(mospParams.getName("Vacation"));
				return sb.toString();
			}
		} else if (type1 == 2) {
			// 特別休暇
			StringBuffer sb = new StringBuffer();
			sb.append(mospParams.getName("Specially"));
			sb.append(mospParams.getName("Vacation"));
			return sb.toString();
		} else if (type1 == 3) {
			// その他休暇
			return mospParams.getName("Others");
		} else if (type1 == 4) {
			// 欠勤
			return mospParams.getName("Absence");
		}
		return "";
	}
	
	/**
	 * @param type1 休暇区分1
	 * @param type2 休暇区分2
	 * @param date 対象年月日
	 * @return 休暇略称
	 * @throws MospException 例外発生時
	 */
	protected String getHolidayType2Abbr(int type1, String type2, Date date) throws MospException {
		if (type1 == 1) {
			return "";
		}
		HolidayDtoInterface dto = timeReference().holiday().getHolidayInfo(type2, date, type1);
		if (dto == null) {
			return type2;
		}
		return dto.getHolidayAbbr();
	}
	
	/**
	 * 引数で渡された休暇種別から表示する文字列を返す。<br>
	 * @param holidayType1 休暇種別1
	 * @param holidayType2 休暇種別2
	 * @param date 対象年月日
	 * @return 表示用文字列
	 * @throws MospException 例外発生時
	 */
	public String getHolidayTypeName(int holidayType1, String holidayType2, Date date) throws MospException {
		StringBuffer sb = new StringBuffer();
		sb.append(getHolidayType1Name(holidayType1, holidayType2));
		sb.append(" ");
		sb.append(getHolidayType2Abbr(holidayType1, holidayType2, date));
		return sb.toString();
	}
	
	/**
	* リクエストされた譲渡年を取得する。
	* @return 譲渡年
	*/
	protected String getTransferredYear() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_YEAR);
	}
	
	/**
	 * 仮締成功メッセージの設定。
	 */
	protected void addTighteningMessage() {
		String[] aryMeassage = { mospParams.getName("Provisional") + mospParams.getName("Cutoff") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * 解除成功メッセージの設定。
	 */
	protected void addReleaseMessage() {
		String[] aryMeassage = { mospParams.getName("Release") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * 申請承認状態から表示する文字列を返す。<br>
	 * @param state 申請承認の状態
	 * @return 一覧表示用文字列
	 */
	public String getStateValueView(int state) {
		String displayState = null;
		if (state == 0) {
			displayState = mospParams.getName("Ram") + mospParams.getName("Application");
		} else {
			displayState = mospParams.getName("Ram") + mospParams.getName("Approval");
		}
		return displayState;
	}
	
	/**
	* 数値で渡ってきた休日範囲を文字列に変換する。
	 * @param holidayRange 休日範囲
	* @return "全休/前休/後休/時休"
	*/
	protected String getHolidayRange(int holidayRange) {
		if (holidayRange == 1) {
			return mospParams.getName("AllTime");
		} else if (holidayRange == 2) {
			return mospParams.getName("FrontTime");
		} else if (holidayRange == 3) {
			return mospParams.getName("BackTime");
		} else if (holidayRange == 4) {
			return mospParams.getName("HolidayTime");
		} else {
			return "";
		}
	}
	
	/**
	 * 確定成功メッセージの設定。
	 */
	protected void addDecideMessage() {
		String[] aryMeassage = { mospParams.getName("Definition") };
		mospParams.addMessage(PlatformMessageConst.MSG_PROCESS_SUCCEED, aryMeassage);
	}
	
	/**
	 * @param workflowDto ワークフローDTOインターフェース
	 * @param commentDto ワークフローコメントDTOインターフェース
	 * @return コメントデータ
	 * @throws MospException 例外処理発生時
	 */
	protected String getWorkflowCommentDtoComment(WorkflowDtoInterface workflowDto,
			WorkflowCommentDtoInterface commentDto) throws MospException {
		String status = getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto.getWorkflowStage());
		if (status.equals(mospParams.getName("WorkPaper"))
				|| status.equals(mospParams.getName("Ram") + mospParams.getName("Approval"))) {
			return "";
		} else {
			return commentDto.getWorkflowComment();
		}
	}
	
	/**
	 * @param workflowDto ワークフローDTOインターフェース
	 * @param commentDto ワークフローコメントDTOインターフェース
	 * @return 最終コメント入力者名
	 * @throws MospException 例外処理発生時
	 */
	protected String getWorkflowCommentDtoLastFirstName(WorkflowDtoInterface workflowDto,
			WorkflowCommentDtoInterface commentDto) throws MospException {
		String status = getStatusStageValueView(workflowDto.getWorkflowStatus(), workflowDto.getWorkflowStage());
		if (status.equals(mospParams.getName("WorkPaper"))
				|| status.equals(mospParams.getName("Ram") + mospParams.getName("Approval"))) {
			return "";
		} else {
			HumanDtoInterface dto = reference().human().getHumanInfo(commentDto.getPersonalId(),
					commentDto.getWorkflowDate());
			return getLastFirstName(dto.getLastName(), dto.getFirstName());
		}
	}
	
	/**
	 * 引数で渡された時間（時、分）から分を設定する。<br>
	 * @param hour 時
	 * @param minute 分
	 * @return 時分から取得した分
	 */
	public int getTimeValue(int hour, int minute) {
		return hour * TimeConst.CODE_DEFINITION_HOUR + minute;
	}
	
	/**
	 * 引数で渡された時間（時、分）から分を設定する。<br>
	 * @param hour 時
	 * @param minute 分
	 * @return 時分から取得した分
	 */
	public int getTimeValue(String hour, String minute) {
		if (hour == null || hour.isEmpty() || minute == null || minute.isEmpty()) {
			return 0;
		}
		return getTimeValue(Integer.valueOf(hour), Integer.valueOf(minute));
	}
	
	/**
	 * @param employeeCode 社員コード
	 * @return 個人ID
	 * @throws MospException 例外発生時
	 * 
	 */
	protected String getPersonalId(String employeeCode) throws MospException {
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		HumanReferenceBeanInterface getHumanInfo = reference().human();
		// 取得したユーザIDとシステム日付から個人IDを取得する
		HumanDtoInterface humanDto = getHumanInfo.getHumanInfoForEmployeeCode(employeeCode, date);
		if (humanDto == null) {
			// 取得したユーザIDに該当する人事マスタのデータがNULLなら処理終了
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Employee"), null);
			return null;
		}
		// 個人IDを返す
		return humanDto.getPersonalId();
	}
	
	/**
	 * @param personalId 個人ID
	 * @return 社員コード
	 * @throws MospException 例外発生時
	 * 
	 */
	protected String getEmployeeCode(String personalId) throws MospException {
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		HumanReferenceBeanInterface getHumanInfo = reference().human();
		// 取得したユーザIDとシステム日付から個人IDを取得する
		HumanDtoInterface humanDto = getHumanInfo.getHumanInfo(personalId, date);
		if (humanDto == null) {
			// 取得したユーザIDに該当する人事マスタのデータがNULLなら処理終了
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Employee"), null);
			return null;
		}
		// 社員コードを返す
		return humanDto.getEmployeeCode();
	}
	
	/**
	 * 引数で渡された年月で該当年月の最終日情報を返す。
	 * @param year 対象年
	 * @param month 対象月
	 * @return 該当月の最終日
	 */
	protected int getEndDay(int year, int month) {
		int day = 0;
		int yearFlg = 0;
		int[] monthDay = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };
		if (year % 4 != 0) {
			// 閏年ではない
			yearFlg = 0;
		} else {
			if (year % 100 == 0) {
				if (year % 400 == 0) {
					// 閏年
					yearFlg = 1;
				} else {
					// 閏年ではない
					yearFlg = 0;
				}
			} else {
				// 閏年
				yearFlg = 1;
			}
		}
		if (yearFlg == 1) {
			if (month == 2) {
				day = monthDay[month - 1] + 1;
			} else {
				day = monthDay[month - 1];
			}
		} else {
			day = monthDay[month - 1];
		}
		return day;
	}
	
	/**
	 * Calendarインスタンス取得
	 * @return	Calendarインスタンス
	 */
	@Deprecated
	private static Calendar getCalendar() {
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.HOUR_OF_DAY, 0);
		cal.set(Calendar.MINUTE, 0);
		cal.set(Calendar.SECOND, 0);
		cal.set(Calendar.MILLISECOND, 0);
		return cal;
	}
	
	/**
	 * 指定曜日可否メソッド
	 * @param date		対象年月日
	 * @param dayOfWeek	対象曜日
	 * @return
	 * <p>
	 * 対象の曜日だった場合true、そうでない場合false
	 * </p>
	 */
	@Deprecated
	public boolean isDayOfWeek(Date date, int dayOfWeek) {
		Calendar cal = getCalendar();
		cal.setTime(date);
		if (cal.get(Calendar.DAY_OF_WEEK) == dayOfWeek) {
			return true;
		}
		return false;
	}
	
	/**
	 * 引数で渡された年月で祝日情報を返す。
	 * @param year 対象年
	 * @param month 対象月
	 * @return MM/DD(W),祝日情報
	 */
	@Deprecated
	protected Map<Date, String> getHolidayMap(int year, int month) {
		// 祝日マップ
		Map<Date, String> holidayMap = new HashMap<Date, String>();
		if (year == 0 || month == 0) {
			return holidayMap;
		}
		Calendar cal = getCalendar();
		if (month == 1) {
			// 元日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 1 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 1);
			holidayMap.put(cal.getTime(), "元日");
			// 成人の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 0);
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 2);
			holidayMap.put(cal.getTime(), "成人の日");
		} else if (month == 2) {
			// 建国記念の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 2 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 11);
			holidayMap.put(cal.getTime(), "建国記念の日");
		} else if (month == 3) {
			// 春分の日
			final double param1 = 21.4471d;
			final double param2 = 0.242377d;
			final double param3 = 4.0d;
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 3 - 1);
			cal.set(Calendar.DAY_OF_MONTH, Double.valueOf(
					param1 + (param2 * (year - 1900)) - Math.floor((year - 1900) / param3)).intValue());
			holidayMap.put(cal.getTime(), "春分の日");
		} else if (month == 4) {
			// 昭和の日
			final int day = 29;
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 4 - 1);
			cal.set(Calendar.DAY_OF_MONTH, day);
			holidayMap.put(cal.getTime(), "昭和の日");
		} else if (month == 5) {
			// 憲法記念日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 5 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 3);
			holidayMap.put(cal.getTime(), "憲法記念日");
			// みどりの日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 5 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 4);
			holidayMap.put(cal.getTime(), "みどりの日");
			// こどもの日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 5 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 5);
			holidayMap.put(cal.getTime(), "こどもの日");
		} else if (month == 7) {
			// 海の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 7 - 1);
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 3);
			holidayMap.put(cal.getTime(), "海の日");
		} else if (month == 9) {
			// 敬老の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 9 - 1);
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 3);
			holidayMap.put(cal.getTime(), "敬老の日");
			// 秋分の日
			final double param1 = 23.8896d;
			final double param2 = 0.242032d;
			final double param3 = 4.0d;
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 9 - 1);
			cal.set(Calendar.DAY_OF_MONTH, Double.valueOf(
					param1 + (param2 * (year - 1900)) - Math.floor((year - 1900) / param3)).intValue());
			holidayMap.put(cal.getTime(), "秋分の日");
		} else if (month == 10) {
			// 体育の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 10 - 1);
			cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
			cal.set(Calendar.DAY_OF_WEEK_IN_MONTH, 2);
			holidayMap.put(cal.getTime(), "体育の日");
		} else if (month == 11) {
			// 文化の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 11 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 3);
			holidayMap.put(cal.getTime(), "文化の日");
			// 勤労感謝の日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 11 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 23);
			holidayMap.put(cal.getTime(), "勤労感謝の日");
		} else if (month == 12) {
			// 天皇誕生日
			cal.set(Calendar.YEAR, year);
			cal.set(Calendar.MONTH, 12 - 1);
			cal.set(Calendar.DAY_OF_MONTH, 23);
			holidayMap.put(cal.getTime(), "天皇誕生日");
		} else {
			// 祝日はないので処理終了
		}
		// 振替休暇を検索
		Set<Date> keySet = holidayMap.keySet();
		TreeMap<Date, String> addMap = new TreeMap<Date, String>();
		Date formerHoliday = null;
		for (Date date : keySet) {
			// 第三条第二項
			// 「国民の祝日」が日曜日に当たるときは、その日後においてその日に最も近い「国民の祝日」でない日を休日とする。 
			// 日曜だった場合、最も近い「国民の祝日」でない日に「振替休日」を追加する。
			if (isDayOfWeek(date, Calendar.SUNDAY)) {
				cal.setTime(date);
				cal.add(Calendar.DAY_OF_MONTH, 1);
				// 既に登録されている休日と重複した場合、一日ずらす。
				while (holidayMap.containsKey(cal.getTime())) {
					cal.add(Calendar.DAY_OF_MONTH, 1);
				}
				if (!holidayMap.containsKey(cal.getTime())) {
					addMap.put(cal.getTime(), "振替休日");
				}
			}
			// 第三条第三項
			// その前日及び翌日が「国民の祝日」である日（「国民の祝日」でない日に限る。）は、休日とする。 
			if (formerHoliday != null) {
				cal.setTime(formerHoliday);
				cal.add(Calendar.DAY_OF_MONTH, 2);
				if (cal.getTime().compareTo(date) == 0) {
					cal.add(Calendar.DAY_OF_MONTH, -1);
					addMap.put(cal.getTime(), "国民の祝日");
				}
			}
			formerHoliday = date;
		}
		holidayMap.putAll(addMap);
		return holidayMap;
	}
	
	/**
	 * 対象時間を分単位で返す<br>
	 * @param date 対象時間
	 * @return 分
	 */
	protected int setHourMinute(Date date) {
		if (date == null) {
			return 0;
		}
		return (DateUtility.getHour(date) * TimeConst.CODE_DEFINITION_HOUR) + DateUtility.getMinute(date);
	}
	
	/**
	 * 開始時刻と終了時間で時間を計算し、分単位で返す<br>
	 * @param start 開始時刻
	 * @param end 終了時刻
	 * @return 分
	 */
	protected int setHourMinute(Date start, Date end) {
		if (start == null || end == null) {
			return 0;
		}
		Date date = new Date(end.getTime() - start.getTime());
		return (DateUtility.getHour(date) * TimeConst.CODE_DEFINITION_HOUR) + DateUtility.getMinute(date);
	}
	
	/**
	 * 休暇コードと有効日から休暇種別略称を取得する。<br>
	 * @param holidayCode 休暇コード
	 * @param activateDate 有効日
	 * @param holidayType 休暇区分
	 * @return 文字列変換した休暇種別
	 * @throws MospException 例外発生時
	 */
	protected String getHolidayAbbr(String holidayCode, Date activateDate, int holidayType) throws MospException {
		HolidayDtoInterface dto = timeReference().holiday().getHolidayInfo(holidayCode, activateDate, holidayType);
		if (dto == null) {
			return holidayCode;
		}
		return dto.getHolidayAbbr();
	}
	
	/**
	 * 開始時/分、終了時/分から勤怠時間を取得する。<br>
	 * @param startHour 開始時
	 * @param startMinute 開始分
	 * @param endHour 終了時
	 * @param endMinute 終了分
	 * @return 分に直した時間
	 * @throws MospException 例外発生時
	 */
	protected int getWorkHour(int startHour, int startMinute, int endHour, int endMinute) throws MospException {
		int rtnMinute = 0;
		if (startHour == endHour) {
			if (startMinute > endMinute) {
				// 終了分よりも開始分のほうが大きいので0を返す
				return 0;
			}
			rtnMinute = endMinute - startMinute;
		}
		if (startHour < endHour) {
			int hour = endHour - startHour;
			int minute = 0;
			if (startMinute > endMinute) {
				minute = TimeConst.CODE_DEFINITION_HOUR + endMinute - startMinute;
				hour = hour - 1;
			} else {
				minute = endMinute - startMinute;
			}
			rtnMinute = hour * TimeConst.CODE_DEFINITION_HOUR + minute;
		}
		if (startHour > endHour) {
			// 終了時よりも開始時のほうが大きいので0を返す
			return 0;
		}
		return rtnMinute;
	}
	
	/**
	 * 時分形式で出力する。
	 * @param date 日時
	 * @return HH:MM
	 */
	protected String getTimeColonFormat(Date date) {
		return DateUtility.getStringHour(date) + mospParams.getName("HalfColon") + DateUtility.getStringMinute(date);
	}
	
	/**
	 * ボタンの背景色文字列の設定。<br>
	 * @param flg 数字
	 * @return 変換後の文字列
	 */
	protected String setButtonBackColor(int flg) {
		String buttonBackColor = "";
		if (flg == 1) {
			// flgが1（有り）の場合は背景色を黄文字を設定する
			buttonBackColor = "style=\"background-color: yellow\"";
		}
		return buttonBackColor;
	}
	
	/**
	 * 承認者用プルダウンの作成。<br>
	 * @param personalId 個人ID
	 * @param date 有効日
	 * @return 結果可否
	 * @throws MospException 例外処理発生時
	 */
	protected boolean setApprvalPullDown(String personalId, Date date) throws MospException {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		// 承認者設定用プルダウン取得
		String[][][] aryApproverInfo = timeReference().approvalInfo().getArrayForApproverSetting(personalId, date,
				PlatformConst.WORKFLOW_TYPE_TIME);
		// 承認者設定用プルダウン確認
		if (aryApproverInfo.length == 0) {
			// 承認者不在メッセージ設定
			addApproverNotExistMessage();
		}
		// エラー確認
		if (mospParams.hasErrorMessage()) {
			return false;
		}
		// 承認者設定用プルダウン設定
		vo.setAryApproverInfo(aryApproverInfo);
		// 承認者タイトルラベル設定
		String[] aryPltLblApproverSetting = new String[aryApproverInfo.length];
		for (int i = 0; i < aryPltLblApproverSetting.length; i++) {
			aryPltLblApproverSetting[i] = String.valueOf(i + 1) + mospParams.getName("Following")
					+ mospParams.getName("Approver");
		}
		vo.setAryPltLblApproverSetting(aryPltLblApproverSetting);
		
		String[] aryApproverSetting = { vo.getPltApproverSetting1(), vo.getPltApproverSetting2(),
			vo.getPltApproverSetting3(), vo.getPltApproverSetting4(), vo.getPltApproverSetting5(),
			vo.getPltApproverSetting6(), vo.getPltApproverSetting7(), vo.getPltApproverSetting8(),
			vo.getPltApproverSetting9(), vo.getPltApproverSetting10() };
		vo.setAryPltApproverSetting(aryApproverSetting);
		
		String[] pltApproverSetting = { "pltApproverSetting1", "pltApproverSetting2", "pltApproverSetting3",
			"pltApproverSetting4", "pltApproverSetting5", "pltApproverSetting6", "pltApproverSetting7",
			"pltApproverSetting8", "pltApproverSetting9", "pltApproverSetting10" };
		vo.setPltApproverSetting(pltApproverSetting);
		return true;
	}
	
	/**
	* リクエストされた休暇種別1を取得する。
	* @return 休暇種別1
	*/
	protected String getTransferredHolidayType1() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_HOLIDAY_TYPE1);
	}
	
	/**
	* リクエストされた休暇種別2を取得する。
	* @return 休暇種別2
	*/
	protected String getTransferredHolidayType2() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_HOLIDAY_TYPE2);
	}
	
	/**
	* リクエストされた休暇範囲を取得する。
	* @return 休暇範囲
	*/
	protected String getTransferredHolidayRange() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_HOLIDAY_RANGE);
	}
	
	/**
	* リクエストされた時休開始時刻を取得する。
	* @return 時休開始時刻
	*/
	protected String getTransferredStartTime() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_START_TIME);
	}
	
	/**
	* 配列で渡された日付の配列からシステム日付に近い該当勤務形態日を求める。
	 * @param aryPullDownDay 該当する日付の配列
	 * @param systemDay システム日付
	* @return 表示するデフォルトの日付
	*/
	protected String getDefaultDay(String[][] aryPullDownDay, int systemDay) {
		String defaultDay = "";
		for (String[] element : aryPullDownDay) {
			if (systemDay <= Integer.valueOf(element[0])) {
				defaultDay = element[0];
				break;
			}
		}
		return defaultDay;
	}
	
	/**
	 * 承認者の配列を取得。<br>
	 * @return 承認者一覧（配列）
	 * @throws MospException 例外処理発生時
	 */
	protected String[] getSelectApproverIds() throws MospException {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		int loopCnt = 0;
		List<String> list = new ArrayList<String>();
		String[] aryApproverSetting = { vo.getPltApproverSetting1(), vo.getPltApproverSetting2(),
			vo.getPltApproverSetting3(), vo.getPltApproverSetting4(), vo.getPltApproverSetting5(),
			vo.getPltApproverSetting6(), vo.getPltApproverSetting7(), vo.getPltApproverSetting8(),
			vo.getPltApproverSetting9(), vo.getPltApproverSetting10() };
		for (; loopCnt < aryApproverSetting.length; loopCnt++) {
			if (aryApproverSetting[loopCnt].isEmpty()) {
				break;
			}
			list.add(aryApproverSetting[loopCnt]);
		}
		return list.toArray(new String[list.size()]);
	}
	
	/**
	 * @param personalId 個人ID
	 * @return 社員名
	 * @throws MospException 例外発生時
	 * 
	 */
	protected String getEmployeeName(String personalId) throws MospException {
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		HumanReferenceBeanInterface getHumanInfo = reference().human();
		// 取得したユーザIDとシステム日付から個人IDを取得する
		HumanDtoInterface humanDto = getHumanInfo.getHumanInfo(personalId, date);
		if (humanDto == null) {
			// 取得したユーザIDに該当する人事マスタのデータがNULLなら処理終了
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Employee"), null);
			return null;
		}
		// 社員名を返す
		return MospUtility.getHumansName(humanDto.getFirstName(), humanDto.getLastName());
	}
	
	/**
	 * @param status 状況
	 * @return 文字列変換後の区分
	 */
	protected String getButtonOnOff(String status) {
		String rtnState = "";
		if (PlatformConst.CODE_STATUS_DRAFT.equals(status) || PlatformConst.CODE_STATUS_REVERT.equals(status)) {
			rtnState = "on";
		}
		return rtnState;
	}
	
	/**
	 * @param personalId 個人ID
	 * @return 社員名
	 * @throws MospException 例外発生時
	 * 
	 */
	protected String getEmployeeFirstName(String personalId) throws MospException {
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		HumanReferenceBeanInterface getHumanInfo = reference().human();
		// 取得したユーザIDとシステム日付から個人IDを取得する
		HumanDtoInterface humanDto = getHumanInfo.getHumanInfo(personalId, date);
		if (humanDto == null) {
			// 取得したユーザIDに該当する人事マスタのデータがNULLなら処理終了
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Employee"), null);
			return null;
		}
		// 社員名を返す
		return humanDto.getFirstName();
	}
	
	/**
	 * @param personalId 個人ID
	 * @return 社員名
	 * @throws MospException 例外発生時
	 * 
	 */
	protected String getEmployeeLastName(String personalId) throws MospException {
		// システム日付取得
		Date date = DateUtility.getSystemDate();
		HumanReferenceBeanInterface getHumanInfo = reference().human();
		// 取得したユーザIDとシステム日付から個人IDを取得する
		HumanDtoInterface humanDto = getHumanInfo.getHumanInfo(personalId, date);
		if (humanDto == null) {
			// 取得したユーザIDに該当する人事マスタのデータがNULLなら処理終了
			mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, mospParams.getName("Employee"), null);
			return null;
		}
		// 社員名を返す
		return humanDto.getLastName();
	}
	
	/**
	 * リクエストされた譲渡汎用日を取得する。
	 * @return 譲渡汎用日
	 */
	protected String getTransferredDay() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_DAY);
	}
	
	/**
	 * リクエストされた時間を取得する。
	 * @return 時間
	 */
	protected String getTransferredHour() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_HOUR);
	}
	
	/**
	 * リクエストされた分を取得する。
	 * @return 分
	 */
	protected String getTransferredMinute() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_MINUTE);
	}
	
	/**
	 * リクエストされた開始時間を取得する。
	 * @return 開始時間
	 */
	protected String getTransferredStartHour() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_START_HOUR);
	}
	
	/**
	 * リクエストされた開始分を取得する。
	 * @return 開始分
	 */
	protected String getTransferredStartMinute() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_START_MINUTE);
	}
	
	/**
	 * リクエストされた終了時間を取得する。
	 * @return 終了時間
	 */
	protected String getTransferredEndHour() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_END_HOUR);
	}
	
	/**
	 * リクエストされた終了分を取得する。
	 * @return 終了分
	 */
	protected String getTransferredEndMinute() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_END_MINUTE);
	}
	
	/**
	 * リクエストされた直行を取得する。
	 * @return 直行
	 */
	protected String getTransferredDirectStart() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_DIRECT_START);
	}
	
	/**
	 * リクエストされた直帰を取得する。
	 * @return 直帰
	 */
	protected String getTransferredDirectEnd() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_DIRECT_END);
	}
	
	/**
	 * リクエストされた勤怠コメントを取得する。
	 * @return 勤怠コメント
	 */
	protected String getTransferredTimeComment() {
		return mospParams.getRequestParam(TimeConst.PRM_TRANSFERRED_TIME_COMMENT);
	}
	
	// TODO timeUtilに作って置き換えたい
	/**
	 * 引数で取得した該当日付から年度を求める。
	 * @param date 日付
	 * @return 年度
	 */
	protected String getFiscalYear(Date date) {
		int year = DateUtility.getYear(date);
		int month = DateUtility.getMonth(date);
		if (month < 4) {
			year = year - 1;
		}
		return String.valueOf(year);
	}
	
	/**
	 * 引数で指定された個人IDと申請日に時差出勤が行われているかどうか判断する。
	 * @param personalId 個人ID
	 * @param date 申請日
	 * @throws MospException 例外発生時
	 */
	protected void getDifferenceRequest1(String personalId, Date date) throws MospException {
		// VO取得
		TimeVo vo = (TimeVo)mospParams.getVo();
		vo.setJsModeDifferenceRequest1("");
		vo.setJsModeRequestDate1("");
		// 該当日付に時差出勤が申請されているか確認する。
		DifferenceRequestDtoInterface diffDto = timeReference().differenceRequest().findForKeyOnWorkflow(personalId,
				date);
		if (diffDto != null) {
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(diffDto.getWorkflow());
			if (workflowDto != null) {
				// ワークフローの状態が下書/取下以外の場合にtrueを返す
				if (PlatformConst.CODE_STATUS_DRAFT.equals(workflowDto.getWorkflowStatus())
						|| PlatformConst.CODE_STATUS_WITHDRAWN.equals(workflowDto.getWorkflowStatus())) {
					return;
				} else {
					vo.setJsModeDifferenceRequest1("on");
					vo.setJsModeRequestDate1(getStringDate(date));
					return;
				}
			} else {
				return;
			}
		} else {
			return;
		}
	}
	
	/**
	 * 承認者が存在しない場合のメッセージを設定する。<br>
	 */
	protected void addApproverNotExistMessage() {
		mospParams.addErrorMessage(PlatformMessageConst.MSG_NOT_EXIST_AND_CHANGE_SOMETHING, mospParams
			.getName("Approver"), mospParams.getName("Application") + mospParams.getName("Day"));
	}
	
	/**
	 * @param personalId 個人ID
	 * @param requestDate 対象日
	 * @param tyep 機能コード
	 * @return 背景
	 * @throws MospException 例外発生時
	 */
	protected String setBackColor(String personalId, Date requestDate, String tyep) throws MospException {
		if (TimeConst.CODE_FUNCTION_VACATION.equals(tyep)) {
			// 休暇
			if (timeReference().holidayRequest().chkHalfHoliday(personalId, requestDate)) {
				if (timeReference().differenceRequest().findForKeyOnWorkflow(personalId, requestDate) != null) {
					return TimeConst.STYLE_BACKGROUND_YELLOW;
				}
			}
		}
		if (TimeConst.CODE_FUNCTION_WORK_HOLIDAY.equals(tyep)) {
			// 休出
			if (timeReference().workOnHolidayRequest().chkHalfHoliday(personalId, requestDate)) {
				if (timeReference().differenceRequest().findForKeyOnWorkflow(personalId, requestDate) != null) {
					return TimeConst.STYLE_BACKGROUND_YELLOW;
				}
			}
		}
		if (TimeConst.CODE_FUNCTION_COMPENSATORY_HOLIDAY.equals(tyep)) {
			// 代休
			if (timeReference().subHolidayRequest().chkHalfHoliday(personalId, requestDate)) {
				if (timeReference().differenceRequest().findForKeyOnWorkflow(personalId, requestDate) != null) {
					return TimeConst.STYLE_BACKGROUND_YELLOW;
				}
			}
		}
		if (TimeConst.CODE_FUNCTION_DIFFERENCE.equals(tyep)) {
			// 時差出勤
			if (timeReference().holidayRequest().chkHalfHoliday(personalId, requestDate)) {
				return TimeConst.STYLE_BACKGROUND_YELLOW;
			}
			if (timeReference().workOnHolidayRequest().chkHalfHoliday(personalId, requestDate)) {
				return TimeConst.STYLE_BACKGROUND_YELLOW;
			}
			if (timeReference().subHolidayRequest().chkHalfHoliday(personalId, requestDate)) {
				return TimeConst.STYLE_BACKGROUND_YELLOW;
			}
			DifferenceRequestDtoInterface dto = timeReference().differenceRequest().findForKeyOnWorkflow(personalId,
					requestDate);
			if (dto == null || !"s".equals(dto.getDifferenceType())) {
				return "";
			}
			Date fiveHour = DateUtility.getDateTime(DateUtility.getYear(dto.getRequestDate()), DateUtility.getMonth(dto
				.getRequestDate()), DateUtility.getDay(dto.getRequestDate()), 5, 0);
			Date twentyTwoHour = DateUtility.getDateTime(DateUtility.getYear(dto.getRequestDate()), DateUtility
				.getMonth(dto.getRequestDate()), DateUtility.getDay(dto.getRequestDate()), 22, 0);
			if (!dto.getRequestStart().before(fiveHour) && !dto.getRequestStart().after(twentyTwoHour)
					&& !dto.getRequestEnd().before(fiveHour) && !dto.getRequestEnd().after(twentyTwoHour)) {
				// 時差出勤時刻が深夜時間帯でない場合
				return TimeConst.STYLE_BACKGROUND_YELLOW;
			}
		}
		return "";
	}
	
	/**
	 * 入力されている社員コートが正しくない場合の設定。
	 */
	protected void addNotExistEmployeesErrorMessage() {
		mospParams.addErrorMessage(TimeMessageConst.MSG_NOT_EXIST_EMPLOYEES, null);
	}
	
	/**
	 * 対象日における人事履歴情報が存在しない場合のエラーメッセージを追加する。<br>
	 * @param targetDate 対象日
	 */
	protected void addHumanHistoryNotExistErrorMessage(Date targetDate) {
		String targetName = mospParams.getName("Target") + mospParams.getName("Employee");
		mospParams.addErrorMessage(PlatformMessageConst.MSG_HISTORY_NOT_EXIST, getStringDate(targetDate), targetName);
	}
	
	/**
	 * 勤務形態コードを取得する。<br>
	 * @param date 申請日
	 * @return 勤務形態コード。
	 * @throws MospException 例外発生時
	 */
	protected String getBeforeDifferenceRequestWorkTypeCode(Date date) throws MospException {
		// VO準備
		TimeVo vo = (TimeVo)mospParams.getVo();
		ApplicationReferenceBeanInterface applicationReference = timeReference().application();
		ScheduleReferenceBeanInterface scheduleReference = timeReference().schedule();
		ScheduleDateReferenceBeanInterface scheduleDateReference = timeReference().scheduleDate();
		WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = timeReference().workOnHolidayRequest()
			.findForKeyOnWorkflow(vo.getPersonalId(), date);
		if (workOnHolidayRequestDto != null) {
			long workflow = workOnHolidayRequestDto.getWorkflow();
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
			if (workflowDto != null && PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				int substitute = workOnHolidayRequestDto.getSubstitute();
				if (substitute == 1) {
					// 申請する場合
					List<SubstituteDtoInterface> list = timeReference().substitute().getSubstituteList(workflow);
					for (SubstituteDtoInterface dto : list) {
						return dto.getSubstituteType();
					}
				} else if (substitute == 2) {
					// 申請しない場合
					return workOnHolidayRequestDto.getWorkOnHolidayType();
				}
			}
		}
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(vo.getPersonalId(), date);
		applicationReference.chkExistApplication(applicationDto, date);
		if (mospParams.hasErrorMessage()) {
			return "";
		}
		ScheduleDtoInterface scheduleDto = scheduleReference.getScheduleInfo(applicationDto.getScheduleCode(), date);
		scheduleReference.chkExistSchedule(scheduleDto, date);
		if (mospParams.hasErrorMessage()) {
			return "";
		}
		ScheduleDateDtoInterface scheduleDateDto = scheduleDateReference.getScheduleDateInfo(scheduleDto
			.getScheduleCode(), scheduleDto.getActivateDate(), date);
		scheduleDateReference.chkExistScheduleDate(scheduleDateDto, date);
		if (mospParams.hasErrorMessage()) {
			return "";
		}
		return scheduleDateDto.getWorkTypeCode();
	}
	
	/**
	 * 時差出勤申請前の勤務形態略称取得。
	 * @param date 対象日
	 * @return 時差出勤申請前の勤務形態略称
	 * @throws MospException 例外発生時
	 */
	protected String getBeforeDifferenceRequestWorkTypeAbbr(Date date) throws MospException {
		// VO準備
		TimeVo vo = (TimeVo)mospParams.getVo();
		WorkTypeReferenceBeanInterface workTypeReference = timeReference().workType();
		ApplicationReferenceBeanInterface applicationReference = timeReference().application();
		ScheduleReferenceBeanInterface scheduleReference = timeReference().schedule();
		ScheduleDateReferenceBeanInterface scheduleDateReference = timeReference().scheduleDate();
		WorkOnHolidayRequestDtoInterface workOnHolidayRequestDto = timeReference().workOnHolidayRequest()
			.findForKeyOnWorkflow(vo.getPersonalId(), date);
		if (workOnHolidayRequestDto != null) {
			long workflow = workOnHolidayRequestDto.getWorkflow();
			WorkflowDtoInterface workflowDto = reference().workflow().getLatestWorkflowInfo(workflow);
			if (workflowDto != null && PlatformConst.CODE_STATUS_COMPLETE.equals(workflowDto.getWorkflowStatus())) {
				int substitute = workOnHolidayRequestDto.getSubstitute();
				if (substitute == 1) {
					// 申請する場合
					List<SubstituteDtoInterface> list = timeReference().substitute().getSubstituteList(workflow);
					for (SubstituteDtoInterface dto : list) {
						return workTypeReference.getWorkTypeAbbr(dto.getSubstituteType(), dto.getSubstituteDate());
					}
				} else if (substitute == 2) {
					// 申請しない場合
					return workTypeReference.getWorkTypeAbbr(workOnHolidayRequestDto.getWorkOnHolidayType(),
							workOnHolidayRequestDto.getRequestDate());
				}
			}
		}
		ApplicationDtoInterface applicationDto = applicationReference.findForPerson(vo.getPersonalId(), date);
		applicationReference.chkExistApplication(applicationDto, date);
		if (mospParams.hasErrorMessage()) {
			return "";
		}
		ScheduleDtoInterface scheduleDto = scheduleReference.getScheduleInfo(applicationDto.getScheduleCode(), date);
		scheduleReference.chkExistSchedule(scheduleDto, date);
		if (mospParams.hasErrorMessage()) {
			return "";
		}
		ScheduleDateDtoInterface scheduleDateDto = scheduleDateReference.getScheduleDateInfo(scheduleDto
			.getScheduleCode(), scheduleDto.getActivateDate(), date);
		scheduleDateReference.chkExistScheduleDate(scheduleDateDto, date);
		if (mospParams.hasErrorMessage()) {
			return "";
		}
		return workTypeReference.getWorkTypeAbbr(scheduleDateDto.getWorkTypeCode(), scheduleDateDto.getScheduleDate());
	}
	
	/**
	 * ワークフロー番号から申請モードを取得する。
	 * @param workflow ワークフロー番号
	 * @return 申請モード
	 * @throws MospException ワークフロー情報入手に失敗した場合
	 */
	protected String getApplicationMode(long workflow) throws MospException {
		// ワークフロー情報取得
		WorkflowDtoInterface dto = reference().workflow().getLatestWorkflowInfo(workflow);
		// ワークフロー情報確認
		if (dto == null) {
			return null;
		}
		// ワークフロー状況確認(下書)
		if (dto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_DRAFT)) {
			return TimeConst.MODE_APPLICATION_DRAFT;
		}
		// ワークフロー状況確認(差戻)
		if (dto.getWorkflowStatus().equals(PlatformConst.CODE_STATUS_REVERT)) {
			return TimeConst.MODE_APPLICATION_REVERT;
		}
		return null;
	}
}
