/*
 * MosP - Mind Open Source Project    http://www.mosp.jp/
 * Copyright (C) MIND Co., Ltd.       http://www.e-mind.co.jp/
 * 
 * This program is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package jp.mosp.platform.bean.file;

import java.io.InputStream;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import jp.mosp.framework.base.MospException;
import jp.mosp.framework.base.MospParams;
import jp.mosp.framework.utils.DateUtility;
import jp.mosp.orangesignal.OrangeSignalUtility;
import jp.mosp.platform.base.PlatformBean;
import jp.mosp.platform.bean.human.base.PlatformHumanBean;
import jp.mosp.platform.constant.PlatformFileConst;
import jp.mosp.platform.constant.PlatformMessageConst;
import jp.mosp.platform.dto.file.ImportDtoInterface;
import jp.mosp.platform.dto.file.ImportFieldDtoInterface;

/**
 * プラットフォームにおけるファイルインポートBeanの基本機能を提供する。<br>
 * <br>
 */
public abstract class PlatformFileBean extends PlatformBean {
	
	/**
	 * {@link PlatformHumanBean}を生成する。<br>
	 */
	public PlatformFileBean() {
		// 処理無し
	}
	
	/**
	 * {@link PlatformBean#PlatformBean()}を実行する。<br>
	 * @param mospParams 設定するMosP処理情報
	 * @param connection 設定するデータベースコネクション
	 */
	protected PlatformFileBean(MospParams mospParams, Connection connection) {
		super(mospParams, connection);
	}
	
	/**
	 * 登録情報リストを取得する。<br>
	 * インポートマスタ情報に従い、登録情報リストを取得する。<br>
	 * @param importDto     インポートマスタ情報
	 * @param requestedFile リクエストされたファイル
	 * @return 登録情報リスト
	 * @throws MospException 登録情報リストの取得に失敗した場合
	 */
	protected List<String[]> getDataList(ImportDtoInterface importDto, InputStream requestedFile) throws MospException {
		// 登録情報リスト準備
		List<String[]> dataList = new ArrayList<String[]>();
		// インポートマスタ情報(ファイル区分)確認
		if (importDto.getType().equals(PlatformFileConst.FILE_TYPE_CSV)) {
			// アップロードファイルを登録情報リストに変換(CSVの場合)
			dataList = OrangeSignalUtility.parse(requestedFile);
		} else {
			// エラーメッセージ設定
			addFileTypeNotExistMessage();
			return dataList;
		}
		// インポートマスタ情報(ヘッダ有無)確認
		if (importDto.getHeader() != PlatformFileConst.HEADER_TYPE_NONE) {
			// ヘッダ除去(登録情報リストから1行目を除く)
			if (dataList.size() > 0) {
				dataList.remove(0);
			}
		}
		// 登録情報件数確認
		if (dataList.size() == 0) {
			// エラーメッセージ設定
			addFileDataNotExistMessage();
		}
		return dataList;
	}
	
	/**
	 * 登録情報からフィールドに対応する値を取得する。<br>
	 * 取得できなかった場合、nullを返す。<br>
	 * @param fieldName フィールド名
	 * @param fieldList インポートフィールド情報リスト
	 * @param data      登録情報
	 * @return 有効日
	 */
	protected String getFieldValue(String fieldName, List<ImportFieldDtoInterface> fieldList, String[] data) {
		// フィールドリスト確認
		for (ImportFieldDtoInterface field : fieldList) {
			// フィールド確認
			if (field.getFieldName().equals(fieldName)) {
				return data[field.getFieldOrder() - 1];
			}
		}
		return null;
	}
	
	/**
	 * 登録情報からフィールド(日付)に対応する値を取得する。<br>
	 * 取得できなかった場合、nullを返す。<br>
	 * @param fieldName フィールド名
	 * @param fieldList インポートフィールド情報リスト
	 * @param data       登録情報
	 * @return 有効日
	 */
	protected Date getDateFieldValue(String fieldName, List<ImportFieldDtoInterface> fieldList, String[] data) {
		// 登録情報から日付文字列を取得
		String date = getFieldValue(fieldName, fieldList, data);
		// 日付文字列確認
		if (date == null || date.isEmpty()) {
			return null;
		}
		// 日付取得
		return getDate(date);
	}
	
	/**
	 * 登録情報リスト内の各登録情報長を確認する。<br>
	 * インポートフィールド情報リスト長と異なる場合、エラーメッセージを設定する。<br>
	 * @param fieldList インポートフィールド情報リスト
	 * @param dataList   登録情報リスト
	 */
	protected void checkCsvLength(List<ImportFieldDtoInterface> fieldList, List<String[]> dataList) {
		// 登録情報リスト確認
		for (int i = 0; i < dataList.size(); i++) {
			// 登録情報長確認
			if (dataList.get(i).length == fieldList.size()) {
				continue;
			}
			// エラーメッセージ追加
			addDataLengthErrorMessage(i);
			break;
		}
	}
	
	/**
	 * 日付オブジェクトを取得する(String→Date)。
	 * @param date 日付文字列(yyyyMMdd 又は yyyy/MM/dd)
	 * @return 日付
	 */
	protected Date getDate(String date) {
		if (date.indexOf("/") == -1) {
			return DateUtility.getDate(date, "yyyyMMdd");
		}
		return DateUtility.getDate(date);
	}
	
	/**
	 * ファイル区分が存在しない場合のメッセージを設定する。<br>
	 */
	protected void addFileTypeNotExistMessage() {
		String[] rep = { mospParams.getName("Data") + mospParams.getName("Type") };
		mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, rep);
	}
	
	/**
	 * ファイルに登録情報が存在しない場合のメッセージを設定する。<br>
	 */
	protected void addFileDataNotExistMessage() {
		String[] rep = { mospParams.getName("Insert") + mospParams.getName("Data") };
		mospParams.addErrorMessage(PlatformMessageConst.MSG_NO_ITEM, rep);
	}
	
	/**
	 * 更新元人事マスタ情報が取得できなかった場合のエラーメッセージを追加する。<br>
	 * @param row          対象行インデックス
	 * @param employeeCode 社員コード
	 * @param activateDate 有効日
	 */
	protected void addEmployeeHistoryNotExistMessage(int row, String employeeCode, Date activateDate) {
		String rep = getRowColonName(row) + getStringDate(activateDate);
		mospParams.addErrorMessage(PlatformMessageConst.MSG_EMP_HISTORY_NOT_EXIST, rep, employeeCode);
	}
	
	/**
	 * 登録情報長が不正な場合のエラーメッセージを追加する。<br>
	 * @param row 対象行インデックス
	 */
	protected void addDataLengthErrorMessage(int row) {
		String rep = getRowColonName(row) + mospParams.getName("Insert") + mospParams.getName("Data");
		mospParams.addErrorMessage(PlatformMessageConst.MSG_INPUT_FORM_AMP, rep, null);
	}
	
}
