/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/

package com.micronova.jsp.tag;

import javax.servlet.jsp.el.*;
import java.lang.reflect.*;
import java.util.*;
import com.micronova.util.*;

/** Codec Function Mapper for JSP2.  Prefix "fn" is reserved for JSTL standard functions.  Other codecs can be specified in one of the following ways:

CodecName:CodecMethod[_argSpec]

or

m:CodecName_CodecMethod[_argSpec]

The "argSpec" is either one of the following, if given:

min (default): method with min number of arguments
max: method with max number of arguments
integer: method with given number of arguments

*/

public class CodecFunctionMapper implements FunctionMapper
{
    protected static CodecFunctionMapper defaultCodecFunctionMapper;

    protected static final String STANDARDFUNCTIONS = "org.apache.taglibs.standard.functions.Functions";
    protected static Map standardFunctionsMap;

    public static final String FIND_MIN = "min";
    public static final String FIND_MAX = "max";

    static
    {
        defaultCodecFunctionMapper = new CodecFunctionMapper();

        try
        {
            standardFunctionsMap = new HashMap();

            Method[] methods = Class.forName(STANDARDFUNCTIONS).getMethods();

            for (int i = 0; i < methods.length; i ++)
            {
                Method method = methods[i];

                standardFunctionsMap.put(method.getName(), method);
            }
        }
        catch (Exception e)
        {
        }
    }

    protected static Method findMethodForName(Class classObject, String name, String findType)
    {
        Method[] methods = classObject.getMethods();

        Method methodFound = null;
        int methodFoundParamCount = 0;

        for (int i = 0; i < methods.length; i ++)
        {
            Method method = methods[i];

            if (name.equals(method.getName()))
            {
                int paramCount = method.getParameterTypes().length;

                boolean isMatch = (methodFound == null);

                if (!isMatch)
                {
                    if (FIND_MIN.equals(findType))
                    {
                        isMatch = (paramCount < methodFoundParamCount);
                    }
                    else if (FIND_MAX.equals(findType))
                    {
                        isMatch = (paramCount > methodFoundParamCount);
                    }
                }
                
                if (isMatch)
                {
                    methodFound = method;
                    methodFoundParamCount = paramCount;
                }
            }
        }

        return methodFound;
    }


    protected CodecFunctionMapper()
    {
        super();
    }

    public static FunctionMapper getInstance()
    {
        return defaultCodecFunctionMapper;
    }

    public Method resolveFunction(String prefix, String name)
    {
        try
        {
            // JSTL standard functions have "fn" prefix

            if ("fn".equals(prefix))
            {
                return (Method)standardFunctionsMap.get(name);
            }

            List namePartList = StringUtil.split(name, '_');
            int namePartListSize = namePartList.size();

            String codecName = prefix;
            String codecMethod = name;
            String argSpec = FIND_MIN;

            if ("m".equals(prefix))
            {
                codecName = (String)namePartList.get(0);
                codecMethod = (String)namePartList.get(1);
                
                if (namePartListSize > 2)
                {
                    argSpec = (String)namePartList.get(2);
                }
            }
            else if (namePartListSize > 1)
            {
                codecMethod = (String)namePartList.get(0);
                argSpec = (String)namePartList.get(1);
            }

            if (codecName.indexOf('.') < 0)
            {
                codecName = EL.CODECPATH + codecName;
            }

            Class c = Class.forName(codecName);
            
            if (FIND_MAX.equals(argSpec) || FIND_MIN.equals(argSpec))
            {
                return findMethodForName(c, codecMethod, argSpec);
            }
            else
            {
                int numArgs = Integer.parseInt(argSpec);
                
                Class[] types = new Class[numArgs];
            
                for (int i = numArgs; --i >=0;)
                {
                    types[i] = Object.class;
                }
                
                return c.getDeclaredMethod(codecMethod, types);
            }
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
    }
}
