/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/

package com.micronova.util.codec;

import java.text.*;
import java.util.*;
import java.net.*;
import java.io.*;
import javax.mail.*;
import javax.mail.internet.*;

import com.micronova.util.*;

/** Type conversion/checking codecs. */

public class CodecType extends Codec
{
    /** returns object as Double if possible, null otherwise. */
    
    public static final Object isDouble(Object object)
    {
        return TypeUtil.isDouble(object);
    }

    /** returns object as Float if possible, null otherwise. */
    
    public static final Object isFloat(Object object)
    {
        return TypeUtil.isFloat(object);
    }

    /** returns object as Long if possible, null otherwise. */
    
    public static final Object isLong(Object object)
    {
        return TypeUtil.isLong(object);
    }

    /** returns object as Integer if possible, null otherwise. */
    
    public static final Object isInteger(Object object)
    {
        return TypeUtil.isInteger(object);
    }

    /** returns object as Short if possible, null otherwise. */
    
    public static Object isShort(Object object)
    {
        return TypeUtil.isShort(object);
    }

    /** returns object as string */

    public static final Object isString(Object object)
    {
        return TypeUtil.isString(object);
    }

    /** returns object as nested map */

    public static final Object isNestedMap(Object object)
    {
        return TypeUtil.isNestedMap(object);
    }

    /** returns object as Boolean if possible, null otherwise. */

    public static final Object isBoolean(Object object)
    {
        return TypeUtil.isBoolean(object);
    }

    /** returns object as Character if possible, null otherwise. */

    public static final Object isCharacter(Object object)
    {
        return TypeUtil.isCharacter(object);
    }

    /** returns object as Calendar or null */

    public static final Object isCalendar(Object object, Object patternSpec, Object localeSpec)
    {
        if (object != null)
        {
            try
            {
                Locale locale = TypeUtil.isLocale(localeSpec);
                String pattern = TypeUtil.isString(patternSpec);
            
                Date date = TypeUtil.isDate(object, pattern, locale);

                Calendar calendar = Calendar.getInstance(locale);
            
                calendar.setTime(date);

                object = calendar;
            }
            catch (Exception e)
            {
                object = null;
            }
        }

        return object;
    }

    /** returns object as Calendar */

    public static final Object isCalendar(Object object, Object patternSpec)
    {
        return isCalendar(object, patternSpec, null);
    }

    /** returns object as Calendar */

    public static final Object isCalendar(Object object)
    {
        return isCalendar(object, null, null);
    }

    /** returns object as Date if possible according to the format specification, null otherwise */

    public static final Object isDate(Object object, Object patternSpec, Object localeSpec)
    {
        if (object != null)
        {
            try
            {
                Locale locale = TypeUtil.isLocale(localeSpec);
                String pattern = TypeUtil.isString(patternSpec);
            
                object = TypeUtil.isDate(object, pattern, locale);
            }
            catch (Exception e)
            {
                object = null;
            }
        }

        return object;
    }

    /** returns object as Date if possible according to the format specification, null otherwise */

    public static final Object isDate(Object object, Object patternSpec)
    {
        return isDate(object, patternSpec, null);
    }

    /** returns object as Date if possible according to the format specification, null otherwise */

    public static final Object isDate(Object object)
    {
        return isDate(object, null, null);
    }

    /** returns object as number if possible, according to the spec */

    public static final Object isNumber(Object object, Object patternSpec, Object localeSpec)
    {
        if (object != null)
        {
            try
            {
                Locale locale = TypeUtil.isLocale(localeSpec);
                String pattern = TypeUtil.isString(patternSpec);
            
                object = TypeUtil.isNumber(object, pattern, locale);
            }
            catch (Exception e)
            {
                object = null;
            }
        }

        return object;
    }

    /** returns object as number if possible */

    public static final Object isNumber(Object object, Object patternSpec)
    {
        return isNumber(object, patternSpec, null);
    }

    /** returns object as number if possible */

    public static final Object isNumber(Object object)
    {
        return isNumber(object, null, null);
    }

    /** converts to SQL date */

    public static final Object isSQLDate(Object object)
    {
        if (object instanceof Date)
        {
            object = new java.sql.Date(((Date)object).getTime());
        }
        else
        {
            Long longObject = (Long)isLong(object);

            if (longObject != null)
            {
                object = new java.sql.Date(longObject.longValue());
            }
            else
            {
                object = null;
            }
        }

        return object;
    }

    public static final Object isSQLDate(Object object, Object patternSpec, Object localeSpec)
    {
        return isSQLDate(isDate(object, patternSpec, localeSpec));
    }

    public static final Object isSQLDate(Object object, Object patternSpec)
    {
        return isSQLDate(isDate(object, patternSpec));
    }

    /** converts to SQL time */

    public static final Object isSQLTime(Object object)
    {
        if (object instanceof Date)
        {
            object = new java.sql.Time(((Date)object).getTime());
        }
        else
        {
            Long longObject = (Long)isLong(object);

            if (longObject != null)
            {
                object = new java.sql.Time(longObject.longValue());
            }
            else
            {
                object = null;
            }
        }

        return object;
    }

    public static final Object isSQLTime(Object object, Object patternSpec, Object localeSpec)
    {
        return isSQLTime(isDate(object, patternSpec, localeSpec));
    }

    public static final Object isSQLTime(Object object, Object patternSpec)
    {
        return isSQLTime(isDate(object, patternSpec));
    }

    /** converts to SQL timestampe */

    public static final Object isSQLTimestamp(Object object)
    {
        if (object instanceof Date)
        {
            object = new java.sql.Timestamp(((Date)object).getTime());
        }
        else
        {
            Long longObject = (Long)isLong(object);

            if (longObject != null)
            {
                object = new java.sql.Timestamp(longObject.longValue());
            }
            else
            {
                object = null;
            }
        }

        return object;
    }

    public static final Object isSQLTimestamp(Object object, Object patternSpec, Object localeSpec)
    {
        return isSQLTimestamp(isDate(object, patternSpec, localeSpec));
    }

    public static final Object isSQLTimestamp(Object object, Object patternSpec)
    {
        return isSQLTimestamp(isDate(object, patternSpec));
    }

    /** returns numeric value of a unicode character */

    public static final Object numericValueOf(Object object)
    {
        if (object != null)
        {
            Character c = TypeUtil.isCharacter(object);

            object = new Long(Character.getNumericValue(c.charValue()));
        }

        return object;
    }

    /** converts string to a URI */

    public static final Object isURI(Object object)
    {
        return TypeUtil.isURI(object);
    }

    /** converts string to a URL */

    public static final Object isURL(Object object, Object context)
    {
        return TypeUtil.isURL(object, TypeUtil.isURL(context));
    }
    /** converts string to a URL */

    public static final Object isURL(Object object)
    {
        return TypeUtil.isURL(object);
    }

    /** converts a string or URI to a File */

    public static final Object isFile(Object object)
    {
        return TypeUtil.isFile(object, null);
    }

    /** converts a string or URI to a File */

    public static final Object isFile(Object object, Object parent)
    {
        return TypeUtil.isFile(object, parent);
    }

    /** converts to a Locale */

    public static final Object isLocale(Object object)
    {
        if (object != null)
        {
            object = TypeUtil.isLocale(object);
        }

        return object;
    }

    /** converts to a TimeZone */

    public static final Object isTimeZone(Object object)
    {
        if (object != null)
        {
            object = TypeUtil.isTimeZone(object);
        }

        return object;
    }

    /** returns unicode block */

    public static final Object unicodeBlockOf(Object object)
    {
        if (object != null)
        {
            Character c = TypeUtil.isCharacter(object);
            
            object = Character.UnicodeBlock.of(c.charValue());
        }

        return object;
    }

    /** returns char as number */

    public static final Object charAsNumber(Object object)
    {
        if (object != null)
        {
            Character c = TypeUtil.isCharacter(object);

            object = new Long((long)c.charValue());
        }

        return object;
    }

    /* returns number as char */

    public static final Object numberAsChar(Object object)
    {
        if (object != null)
        {
            Long l = TypeUtil.isLong(object);

            object = new Character((char)(l.longValue()));
        }

        return object;
    }

    /** returns object as Object[] if object is an array, null otherwise */

    public static Object isArray(Object object)
    {
        return TypeUtil.isArray(object);
    }

    /** returns Object as List if possible, null otherwise */

    public static Object isList(Object object)
    {
        return TypeUtil.isList(object);
    }

    /** returns Object as List if possible, null otherwise */

    public static Object isStringArray(Object object)
    {
        return TypeUtil.isStringArray(object);
    }

    /** returns Object as list of Strings */

    public static Object isStringList(Object object, Object separatorSpec, Object escapeSpec)
    {
        return TypeUtil.isStringList(object, TypeUtil.isCharacter(separatorSpec).charValue(), TypeUtil.isCharacter(escapeSpec).charValue());
    }

    public static final Character DEFAULTESCAPE = new Character('\\');
    public static final Character DEFAULTSEPARATOR = new Character(',');

    public static Object isStringList(Object object, Object separatorSpec)
    {
        return isStringList(object, separatorSpec, DEFAULTESCAPE);
    }

    public static Object isStringList(Object object)
    {
        return isStringList(object, DEFAULTSEPARATOR);
    }

    /** returns ifObject if the object is null, otherwise elseObject */

    public static Object ifNull(Object object, Object ifObject, Object elseObject)
    {
        return ((object == null) ? ifObject : elseObject);
    }

    /** returns Boolean.TRUE if object is null */

    public static Object isNull(Object object)
    {
        return ifNull(object, Boolean.TRUE, null);
    }

    /** returns ifObject if object is null, otherwise null */

    public static Object ifNull(Object object, Object ifObject)
    {
        return ifNull(object, ifObject, object);
    }

    /** returns Boolean TRUE if object is not null */

    public static Object isNotNull(Object object)
    {
        return ifNull(object, null, Boolean.TRUE);
    }

    /** returns ifObject if object is not null, otherwise null */

    public static Object ifNotNull(Object object, Object ifObject)
    {
        return ifNull(object, object, ifObject);
    }

    /** convert object to given class if possible, null otherwise */

    public static Object isClass(Object object, Object classSpec)
    {
        if (object != null)
        {
            try
            {
                Class classObject = null;

                if (classSpec instanceof Class)
                {
                    classObject = (Class)classSpec;
                }
                else
                {
                    classObject = TypeUtil.forName(classSpec.toString(), null);
                }

                object = TypeUtil.isClass(object, classObject);
            }
            catch (Exception e)
            {
                object = null;
            }
        }
        
        return object;
    }

    /** obtains class object if possible */

    public static Object forName(Object object)
    {
        if (object != null)
        {
            try
            {
                object = TypeUtil.forName(object.toString(), null);
            }
            catch (Exception e)
            {
                object = null;
            }
        }
        
        return object;
    }

    /** returns length of object if possible, null otherwise */

    public static final Object length(Object object)
    {
        if (object != null)
        {
            int length = TypeUtil.length(object);

            if (length >= 0)
            {
                return new Integer(length);
            }
        }

        return null;
    }

    /** tries to 'clear' the object is a Collection */

    public static final Object clear(Object object)
    {
        if (object instanceof Collection)
        {
            ((Collection)object).clear();
        }

        return object;
    }

    /** reverses list */

    public static final Object reverse(Object object)
    {
        List list = TypeUtil.isList(object);

        if (list != null)
        {
            Collections.reverse(list);
            object = list;
        }

        return object;
    }

    /** rotates list */

    public static final Object rotate(Object object, Object distance)
    {
        List list = TypeUtil.isList(object);

        if (list != null)
        {
            Integer d = TypeUtil.isInteger(distance);

            Collections.rotate(list, d.intValue());
            object = list;
        }

        return object;
    }

    /** suffles list */

    public static final Object shuffle(Object object)
    {
        List list = TypeUtil.isList(object);

        if (list != null)
        {
            Collections.shuffle(list);
            object = list;
        }

        return object;
    }
    
    /** tries to add 'added' to 'object'.  'object' is converted to a Collection if possible. */

    public static final Object add(Object object, Object added)
    {
        Collection collection = TypeUtil.isCollection(object);

        if (collection != null)
        {
            collection.add(added);
            object = collection;
        }

        return object;
    }

    /** tries to add all 'added' Collection to 'object'.  'object' and 'added' are converted to Collection if possible. */

    public static final Object addAll(Object object, Object added)
    {
        Collection collection = TypeUtil.isCollection(object);

        if (collection != null)
        {
            Collection addedCollection = TypeUtil.isCollection(added);

            if (addedCollection != null)
            {
                collection.addAll(addedCollection);
                object = collection;
            }
        }

        return object;
    }

    /** tries to remove 'removed' from 'object'.  'object' is converted to a Collection if possible. */

    public static final Object remove(Object object, Object added)
    {
        Collection collection = TypeUtil.isCollection(object);

        if (collection != null)
        {
            collection.remove(added);
            object = collection;
        }

        return object;
    }

    /** tries to remove all 'removed' Collection from 'object'.  'object' and 'removed' are converted to Collection if possible. */

    public static final Object removeAll(Object object, Object removed)
    {
        Collection collection = TypeUtil.isCollection(object);

        if (collection != null)
        {
            Collection removedCollection = TypeUtil.isCollection(removed);

            if (removedCollection != null)
            {
                collection.removeAll(removedCollection);
                object = collection;
            }
        }

        return object;
    }

    /** tries to retail all 'retained' Collection from 'object'.  'object' and 'removed' are converted to Collection if possible. */

    public static final Object retainAll(Object object, Object removed)
    {
        Collection collection = TypeUtil.isCollection(object);

        if (collection != null)
        {
            Collection removedCollection = TypeUtil.isCollection(removed);

            if (removedCollection != null)
            {
                collection.retainAll(removedCollection);
                object = collection;
            }
        }

        return object;
    }

    /** returns 'ifObject' is 'object' contains 'contained', otherwise 'elseObject'. */

    public static final Object ifContains(Object object, Object contained, Object ifObject, Object elseObject)
    {
        Collection collection = TypeUtil.isCollection(object);

        boolean isContained = false;

        if (collection != null)
        {
            isContained = collection.contains(contained);
        }

        if (isContained)
        {
            return ifObject;
        }
        else
        {
            return elseObject;
        }
    }

    public static final Object ifContains(Object object, Object contained, Object ifObject)
    {
        return ifContains(object, contained, ifObject, null);
    }

    /** returns 'ifObject' is 'object' contains all of 'contained' Collection, otherwise 'elseObject'. */

    public static final Object ifContainsAll(Object object, Object contained, Object ifObject, Object elseObject)
    {
        Collection collection = TypeUtil.isCollection(object);

        boolean isContained = false;

        if (collection != null)
        {
            Collection containedCollection = TypeUtil.isCollection(contained);

            if (containedCollection != null)
            {
                isContained = collection.containsAll(containedCollection);
            }
        }

        if (isContained)
        {
            return ifObject;
        }
        else
        {
            return elseObject;
        }
    }

    public static final Object ifContainsAll(Object object, Object contained, Object ifObject)
    {
        return ifContainsAll(object, contained, ifObject, null);
    }

    /** returns ifObject if the object is equal to target, otherwise elseObject */

    public static Object ifEqual(Object object, Object target, Object ifObject, Object elseObject)
    {
        boolean isEqual = false;

        if (object == null)
        {
            if (target == null)
            {
                isEqual = true;
            }
        }
        else
        {
            isEqual = object.equals(target);
        }

        if (isEqual)
        {
            return ifObject;
        }
        else
        {
            return elseObject;
        }
    }

    /** returns TRUE if object is equal to target, otherwise null */

    public static Object isEqual(Object object, Object target)
    {
        return ifEqual(object, target, Boolean.TRUE, null);
    }

    /** returns TRUE if object is not equal to target, otherwise null */

    public static Object isNotEqual(Object object, Object target)
    {
        return ifEqual(object, target, null, Boolean.TRUE);
    }

    /** returns ifObject if object is equal to target, otherwise object */

    public static Object ifEqual(Object object, Object target, Object ifObject)
    {
        return ifEqual(object, target, ifObject, object);
    }

    /** returns ifObject if object is not equal to target, otherwise object */

    public static Object ifNotEqual(Object object, Object target, Object ifObject)
    {
        return ifEqual(object, target, object, ifObject);
    }

    /** returns 'ifObject' if object is empty, else 'elseObject' */

    public static final Object ifEmpty(Object object, Object ifObject, Object elseObject)
    {
        if (TypeUtil.isEmpty(object))
        {
            return ifObject;
        }
        else
        {
            return elseObject;
        }
    }

    /** returns Boolean.TRUE if object is empty, otherwise null */

    public static final Object isEmpty(Object object)
    {
        return ifEmpty(object, Boolean.TRUE, null);
    }

    /** returns Boolean.TRUE if object is not empty, otherwise null */

    public static final Object isNotEmpty(Object object)
    {
        return ifEmpty(object, null, Boolean.TRUE);
    }

    /** returns ifObject if object is empty, otherwise object */

    public static final Object ifEmpty(Object object, Object ifObject)
    {
        return ifEmpty(object, ifObject, object);
    }

    /** returns ifObject if object is not empty, otherwise object */

    public static final Object ifNotEmpty(Object object, Object ifObject)
    {
        return ifEmpty(object, object, ifObject);
    }

    /** tries to convert to (valid) internet address */

    public static final Object isInternetAddress(Object object)
    {
        InternetAddress address = null;

        try
        {
            if (object instanceof InternetAddress)
            {
                address = (InternetAddress)object;
            }
            else
            {
                address = new InternetAddress(object.toString());
            }

            address.validate();
        }
        catch (Exception e)
        {
            address = null;
        }

        return address;
    }

    public static final Object indexOfSubList(Object source, Object target) throws Exception
    {
        List sourceList = TypeUtil.isList(source);
        List targetList = TypeUtil.isList(target);

        Integer returnValue = null;

        if ((sourceList != null) && (targetList != null))
        {
            returnValue = new Integer(Collections.indexOfSubList(sourceList, targetList));
        }

        return returnValue;
    }

    public static final Object lastIndexOfSubList(Object source, Object target) throws Exception
    {
        List sourceList = TypeUtil.isList(source);
        List targetList = TypeUtil.isList(target);

        Integer returnValue = null;

        if ((sourceList != null) && (targetList != null))
        {
            returnValue = new Integer(Collections.lastIndexOfSubList(sourceList, targetList));
        }

        return returnValue;
    }

    public static final Object subList(Object object, Object startObject, Object endObject) throws Exception
    {
        Object returnValue = null;

        List objectList = TypeUtil.isList(object);

        if (objectList != null)
        {
            int start = 0;
            int end = 0;

            Integer startInteger = TypeUtil.isInteger(startObject);
            Integer endInteger = TypeUtil.isInteger(endObject);

            if (startInteger != null)
            {
                start = startInteger.intValue();
            }

            if (endInteger != null)
            {
                end = endInteger.intValue();
            }

            returnValue = TypeUtil.subList(objectList, start, end);
        }

        return returnValue;
    }

    /** returns synchronized version of given object as a Collection */

    public static final Object makeSynchronized(Object object) throws Exception
    {
        if (object != null)
        {
            if (object instanceof SortedSet)
            {
                object = Collections.synchronizedSortedSet((SortedSet)object);
            }
            else if (object instanceof SortedMap)
            {
                object = Collections.synchronizedSortedMap((SortedMap)object);
            }
            else if (object instanceof Set)
            {
                object = Collections.synchronizedSet((Set)object);
            }
            else if (object instanceof Map)
            {
                object = Collections.synchronizedMap((Map)object);
            }
            else if (object instanceof List)
            {
                object = Collections.synchronizedList((List)object);
            }
            else
            {
                object = Collections.synchronizedCollection((Collection)object);
            }
        }

        return object;
    }

    /** returns unmodifiable version of given object as a Collection */

    public static final Object makeUnmodifiable(Object object) throws Exception
    {
        if (object != null)
        {
            if (object instanceof SortedSet)
            {
                object = Collections.unmodifiableSortedSet((SortedSet)object);
            }
            else if (object instanceof SortedMap)
            {
                object = Collections.unmodifiableSortedMap((SortedMap)object);
            }
            else if (object instanceof Set)
            {
                object = Collections.unmodifiableSet((Set)object);
            }
            else if (object instanceof Map)
            {
                object = Collections.unmodifiableMap((Map)object);
            }
            else if (object instanceof List)
            {
                object = Collections.unmodifiableList((List)object);
            }
            else
            {
                object = Collections.unmodifiableCollection((Collection)object);
            }
        }

        return object;
    }

}
