/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util.codec;

import com.micronova.util.*;
import java.io.*;
import java.util.*;
import java.util.regex.*;
import javax.activation.*;

/** file codecs */

public class CodecFile extends Codec
{
    /** lists directory as list of strings */

    public static Object list(Object object)
    {
        Object result = null;

        File file = TypeUtil.isFile(object);

        if (file != null)
        {
            result = TypeUtil.isList(file.list());
        }

        return result;
    }

    /** lists directory as list of files */

    public static Object listFiles(Object object)
    {
        Object result = null;

        File file = TypeUtil.isFile(object);

        if (file != null)
        {
            result = TypeUtil.isList(file.listFiles());
        }

        return result;
    }

    /** deletes file, returns Boolean.TRUE or Boolean.FALSE */

    public static Object delete(Object object)
    {
        Object result = null;

        File file = TypeUtil.isFile(object);

        if (file != null)
        {
            result = new Boolean(file.delete());
        }

        return result;
    }

    /** renames file, returns renamed file if successful, otherwise null */

    public static Object rename(Object object, Object renameTo)
    {
        Object result = null;

        File file = TypeUtil.isFile(object);
        File to = TypeUtil.isFile(renameTo);

        if ((file != null) && (to != null))
        {
            try
            {
                if (file.renameTo(to))
                {
                    result = to;
                }
            }
            catch (Exception e)
            {
            }
        }

        return result;
    }

    /** reads file content using given encoding */

    public static Object read(Object object, Object encoding)
    {
        Object result = null;

        InputStream inputStream = null;
        Reader reader = null;

        try
        {
            if (object instanceof InputStream)
            {
                inputStream = (InputStream)object;
            }
            else if (object instanceof Reader)
            {
                reader = (Reader)object;
            }
            else
            {
                File file = TypeUtil.isFile(object);
            
                if (file != null)
                {
                    inputStream = new FileInputStream(file);
                }
            }

            if (inputStream != null)
            {
                result = new String(IOUtil.readAll(inputStream), encoding.toString());
            }

            if (reader != null)
            {
                result = new String(IOUtil.readAll(reader));
            }
        }
        catch (Exception e)
        {
            result = null;
        }
        finally
        {
            IOUtil.tryClose(inputStream);
            IOUtil.tryClose(reader);
        }

        return result;
    }

    /** read as binary string */

    public static Object read(Object object)
    {
        return read(object, "iso-8859-1");
    }

    /** writes object to a file, returns object if successful, otherwise null */
    private static Object write(Object object, Object fileObject, String encoding, boolean doesAppend)
    {
        Object result = null;

        if (object != null)
        {
            File file = TypeUtil.isFile(fileObject);
            
            if (file != null)
            {
                OutputStream outputStream = null;
                Writer writer = null;
                
                try
                {
                    if (object instanceof Reader)
                    {
                        writer = new FileWriter(file, doesAppend);

                        IOUtil.copy((Reader)object, writer);

                        writer.close();

                        writer = null;

                        result = object;
                    }
                    else if (object instanceof InputStream)
                    {
                        outputStream = new FileOutputStream(file, doesAppend);
                        
                        IOUtil.copy((InputStream)object, outputStream);

                        outputStream.close();

                        outputStream = null;

                        result = object;
                    }
                    else
                    {
                        outputStream = new FileOutputStream(file, doesAppend);
                    
                        outputStream.write(object.toString().getBytes(encoding));
                        
                        outputStream.close();
                        
                        outputStream = null;

                        result = object;
                    }
                }
                catch (Exception e)
                {
                    result = null;
                }
                finally
                {
                    IOUtil.tryClose(outputStream);
                    IOUtil.tryClose(writer);
                }
            }
        }

        return result;
    }

    /** writes object to a file, returns object if successful, otherwise null */
    public static Object write(Object object, Object fileObject, Object encoding)
    {
        return write(object, fileObject, encoding.toString(), false);
    }

    /** writes object to a file, returns object if successful, otherwise null */
    public static Object write(Object object, Object fileObject)
    {
        return write(object, fileObject, "iso-8859-1", false);
    }

    /** appends object to a file, returns object if successful, otherwise null */
    public static Object append(Object object, Object fileObject, Object encoding)
    {
        return write(object, fileObject, encoding.toString(), true);
    }
 
    /** appends object to a file, returns object if successful, otherwise null */
    public static Object append(Object object, Object fileObject)
    {
        return write(object, fileObject, "iso-8859-1", true);
    }

    /** make a directory creating parent directories if parents is not null */

    public static Object mkdir(Object object, Object parents)
    {
        Object result = null;

        File file = TypeUtil.isFile(object);

        if (file != null)
        {
            try
            {
                if ((parents == null) ? file.mkdir() : file.mkdirs())
                {
                    result = file;
                }
            }
            catch (Exception e)
            {
            }
        }

        return result;
    }

    /** make a directory */

    public static Object mkdir(Object object)
    {
        return mkdir(object, null);
    }

    /** make a directory, creating parents */

    public static Object mkdirs(Object object)
    {
        return mkdir(object, Boolean.TRUE);
    }

    /** return file type */
    
    public static Object type(Object fileObject)
    {
        File file = TypeUtil.isFile(fileObject);

        if (file != null)
        {
            return (new FileDataSource(file)).getContentType();
        }

        return null;
    }

    /** temporary file, using "prefix*suffix" format. */

    public static Object tempfile(Object object)
    {
        if (object == null)
        {
            object = "temp*";
        }

        try
        {
            String prefix = "";
            String suffix = "";
            
            String[] part = object.toString().split("[*]");
            
            int partLength = part.length;
            
            if (partLength >= 1)
            {
                prefix = part[0];
            }
            
            if (partLength >= 2)
            {
                suffix = part[1];
            }
            
            File directory = null;
            
            int slashIndex = prefix.lastIndexOf('/');
            
            if (slashIndex >= 0)
            {
                String directoryPrefix = prefix.substring(0, slashIndex);
                
                directory = new File(directoryPrefix);
                
                prefix = prefix.substring(slashIndex + 1);
            }
            
            if (directory != null)
            {
                object = File.createTempFile(prefix, suffix, directory);
            }
            else
            {
                object = File.createTempFile(prefix, suffix);
            }
        }
        catch (Exception e)
        {
            object = null;
        }

        return object;
    }

    /** closes stream/reader/writer */

    public static Object close(Object object) throws Exception
    {
        if (object instanceof InputStream)
        {
            ((InputStream)object).close();
        }
        else if (object instanceof OutputStream)
        {
            ((OutputStream)object).close();
        }
        else if (object instanceof Writer)
        {
            ((Writer)object).close();
        }
        else if (object instanceof Reader)
        {
            ((Reader)object).close();
        }

        return object;
    }
}
