package com.micronova.util;

import java.util.*;
import java.io.*;
import java.text.*;
import java.util.regex.*;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import javax.xml.transform.sax.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import java.net.*;

/** XML utilities */

public class XMLMap
{
    /** control keys */

    public static final String TYPEATTRIBUTE = "typeAttribute";
    public static final String INDEXATTRIBUTE = "indexAttribute";
    public static final String DEFAULTTYPE = "defaultType";
    public static final String MAPTYPE = "mapType";
    public static final String LISTTYPE = "listType";
    public static final String ITEMNAME = "itemName";
    public static final String LISTNAME = "listName";
    public static final String NAMEPREFIX = "namePrefix";
    public static final String NAMESPACE = "namespace";
    public static final String ROOTNAME = "rootName";
    public static final String NAMEMAP = "nameMap";
    public static final String ROOTATTRIBUTE = "rootAttribute";
    public static final String HASATTRIBUTE = "hasAttribute";

    public static final String ATTRIBUTE = "^";

    /** default control map */

    public static final Map defaultControlMap;

    static
    {
        defaultControlMap = new HashMap();
        
        defaultControlMap.put(TYPEATTRIBUTE, null);
        defaultControlMap.put(INDEXATTRIBUTE, "index");
        defaultControlMap.put(DEFAULTTYPE, "string");
        defaultControlMap.put(MAPTYPE, "map");
        defaultControlMap.put(LISTTYPE, "list");
        defaultControlMap.put(LISTNAME, "_");
        defaultControlMap.put(ITEMNAME, "item");

        defaultControlMap.put(ROOTNAME, "root");
        defaultControlMap.put(NAMESPACE, "*");
    }

    /** get control map */

    protected static final Map getControlMap(Map map, String key)
    {
        Map value = null;

        if (map != null)
        {
            value = (Map)map.get(key);
        }

        if (value == null)
        {
            value = (Map)defaultControlMap.get(key);
        }

        return value;
    }

    /** get control string */

    protected static final String getControlString(Map map, String key)
    {
        String value = null;

        if (map != null)
        {
            value = (String)map.get(key);
        }

        if (value == null)
        {
            value = (String)defaultControlMap.get(key);
        }

        return value;
    }

    /** add map entries as attributes */

    protected static final Element addAttribute(Element element, Map map)
    {
        if (map != null)
        {
            Iterator iterator = map.entrySet().iterator();

            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();

                element.setAttribute(entry.getKey().toString(), entry.getValue().toString());
            }
        }

        return element;
    }

    /** generates XML representation of a map */

    public static final Document encode(Object object) throws Exception
    {
        return encode(object, null);
    }

    /** generates XML representation of a map, according to the control map */

    public static final Document encode(Object object, Map control) throws Exception
    {
        Document document = DocumentBuilderFactory.newInstance().newDocumentBuilder().newDocument();
        
        String name = getControlString(control, ROOTNAME);

        Element root = buildMapNode(document, name, object, getControlMap(control, ATTRIBUTE), control);

        document.appendChild(root);
    
        return document;
    }

protected static final Element buildMapNode(Document document, String name, Object object, Map attributes, Map control)
    {
        String elementName = name;

        String namePrefix = getControlString(control, NAMEPREFIX);

        if (!TypeUtil.isEmpty(namePrefix))
        {
            elementName = namePrefix + elementName;
        }

        Element element = document.createElement(elementName);

        String typeAttribute = getControlString(control, TYPEATTRIBUTE);

        if (object instanceof Map)
        {
            Map elementAttributes = (Map)((Map)object).get(ATTRIBUTE);

            String mapType = getControlString(control, MAPTYPE);

            if ((!TypeUtil.isEmpty(typeAttribute)) && (!TypeUtil.isEmpty(mapType)))
            {
                element.setAttribute(typeAttribute, mapType);
            }

            TreeMap treeMap = new TreeMap((Map)object);

            Iterator iterator = treeMap.entrySet().iterator();

            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();

                Object key = entry.getKey();
                Object value = entry.getValue();

                if (key != null)
                {
                    String keyString = key.toString();

                    if (!ATTRIBUTE.equals(keyString))
                    {
                        if (value instanceof List)
                        {
                            String itemName = getControlString(control, LISTNAME);
                            
                            if ("*".equals(itemName))
                            {
                                keyString = name;
                            }
                            else
                            {
                                keyString = itemName;
                            }
                        }

                        Map elementAttribute = null;

                        if (elementAttributes != null)
                        {
                            elementAttribute = (Map)elementAttributes.get(keyString);
                        }
                        
                        Element child = buildMapNode(document, keyString, value, elementAttribute, control);

                        element.appendChild(child);
                    }
                }
            }
        }
        else if (object instanceof List)
        {
            String listType = getControlString(control, LISTTYPE);
            String indexAttribute = getControlString(control, INDEXATTRIBUTE);

            if ((!TypeUtil.isEmpty(typeAttribute)) && (!TypeUtil.isEmpty(listType)))
            {
                element.setAttribute(typeAttribute, listType);
            }

            int index = 0;

            Iterator iterator = ((List)object).iterator();

            String itemName = getControlString(control, ITEMNAME);

            while (iterator.hasNext())
            {
                Object item = iterator.next();

                Element child = buildMapNode(document, itemName, item, attributes, control);

                if (!TypeUtil.isEmpty(indexAttribute))
                {
                    child.setAttribute(indexAttribute, Integer.toString(index));
                }

                element.appendChild(child);

                index ++;
            }
        }
        else 
        {
            if (object != null)
            {
                String defaultType = getControlString(control, DEFAULTTYPE);
                
                if ((!TypeUtil.isEmpty(typeAttribute)) && (!TypeUtil.isEmpty(defaultType)))
                {
                    element.setAttribute(typeAttribute, defaultType);
                }

                Text text = document.createTextNode(object.toString());
                element.appendChild(text);
            }
        }

        addAttribute(element, attributes);

        return element;
    }

    /** convert XML back to NestedMap, using null namespace */

    public static Object decode(Node node) throws Exception
    {
        return decode(node, null);
    }

    /** get node name, according to "namespace" */

    public static String getNodeName(Node node, String namespace)
    {
        String nodeName = null;

        String namespaceURI = node.getNamespaceURI();

        String localName = node.getLocalName();

        if (localName != null)
        {
            nodeName = localName;
        }
        else
        {
            nodeName = node.getNodeName();
        }

        if (!"_".equals(nodeName))
        {
            if (!"*".equals(namespace))
            {
                if (namespaceURI != null)
                {
                    if (!namespaceURI.equals(namespace))
                    {
                        nodeName = namespaceURI + "|" + nodeName;
                    }
                }
                else
                {
                    if ((namespace != null) && (!namespace.equals("")))
                    {
                        nodeName = "|" + nodeName;
                    }
                }
            }
        }

        return nodeName;
    }

    /** get attribute map */

    public static NestedMap getAttributeMap(Node node, String namespace) throws Exception
    {
        NestedMap attributeMap = null;

        NamedNodeMap attributeNodeMap = node.getAttributes();
        
        if (attributeNodeMap != null)
        {
            int attributeCount = attributeNodeMap.getLength();
                        
            if (attributeCount > 0)
            {
                attributeMap = new NestedMap();
                
                while (--attributeCount >= 0)
                {
                    Attr attr = (Attr)attributeNodeMap.item(attributeCount);
                                
                    if (attr != null)
                    {
                        String attributeName = getNodeName(attr, namespace);
                        
                        
                        attributeMap.put(attributeName, attr.getValue());
                    }
                }
            }
        }

        return attributeMap;
    }

    /** convert XML back to NestedMap */

    public static Object decode(Node node, Map control) throws Exception
    {
        return decode(node, control, "/");
    }

    /** convert XML back to NestedMap */

    public static Object decode(Node node, Map control, String path) throws Exception
    {
        NestedMap map = null;

        String text = null;

        node.normalize();

        NodeList children = node.getChildNodes();

        int childrenLength = children.getLength();

        String namespace = getControlString(control, NAMESPACE);
        String hasAttribute = getControlString(control, HASATTRIBUTE);

        Map nameMap = getControlMap(control, NAMEMAP);

        for (int i = 0; i < childrenLength; i ++)
        {
            Node child = children.item(i);

            int nodeType = child.getNodeType();

            if (nodeType == Node.ELEMENT_NODE)
            {
                if (map == null)
                {
                    map = new NestedMap();
                }

                String nodeName = getNodeName(child, namespace);

                NestedMap attributeMap = null;
                
                if (TypeUtil.isTrue(hasAttribute))
                {
                    NamedNodeMap attributeNodeMap = child.getAttributes();

                    if (attributeNodeMap != null)
                    {
                        attributeMap = getAttributeMap(child, namespace);
                    }
                }

                String nextPath = path + nodeName + "/";

                String target = getControlString(nameMap, nextPath);

                if (target == null)
                {
                    target = getControlString(nameMap, nodeName);
                    
                    if (target == null)
                    {
                        target = nodeName;
                    }
                }

                if ("_".equals(target))
                {
                    NodeList listItems = child.getChildNodes();
                    int listLength = listItems.getLength();

                    for (int j = 0; j < listLength; j ++)
                    {
                        Node listItem = listItems.item(j);

                        NestedMap listItemAttributeMap = getAttributeMap(listItem, namespace);

                        String indexProperty = "*";

                        if (listItemAttributeMap != null)
                        {
                            indexProperty = listItemAttributeMap.getString("index", indexProperty);
                        }

                        Object decoded = decode(listItem, control, nextPath);

                        if (decoded != null)
                        {
                            map.put("@_." + indexProperty, decoded);

                            if (TypeUtil.isTrue(hasAttribute) && (listItemAttributeMap != null))
                            {
                                map.put("@" + ATTRIBUTE + "._." + indexProperty, listItemAttributeMap);
                            }
                        }
                    }
                }
                else 
                {
                    map.put(target, decode(child, control, nextPath));

                    if (attributeMap != null)
                    {
                        map.put("@" + ATTRIBUTE + "." + target, attributeMap);
                    }
                }
            }
            else if (nodeType == Node.TEXT_NODE)
            {
                if (text == null)
                {
                    text = child.getNodeValue();
                }
                else
                {
                    text = text + child.getNodeValue();
                }
            }
        }

        if (map != null)
        {
            return map;
        }
        else
        {
            return text;
        }
    }
}
