/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.util.regex.*;
import java.lang.reflect.*;
import java.text.*;
import java.io.*;
import java.net.*;
import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import org.w3c.dom.*;

/** datatype-related utilities */

public class TypeUtil
{
    /** primitive type map */

    private static final Map PRIMITIVETYPE;

    static
    {
        PRIMITIVETYPE = new HashMap();

        PRIMITIVETYPE.put("boolean", Boolean.TYPE);
        PRIMITIVETYPE.put("char", Character.TYPE);
        PRIMITIVETYPE.put("byte", Byte.TYPE);
        PRIMITIVETYPE.put("short", Short.TYPE);
        PRIMITIVETYPE.put("int", Integer.TYPE);
        PRIMITIVETYPE.put("long", Long.TYPE);
        PRIMITIVETYPE.put("float", Float.TYPE);
        PRIMITIVETYPE.put("double", Double.TYPE);
    }

    /** returns object as Double if possible, null otherwise. */
    
    public static final Double isDouble(Object object)
    {
        if (object != null)
        {
            if (object instanceof Double)
            {
                return (Double)object;
            }
            else
            {
                try
                {
                    return Double.valueOf(object.toString());
                }
                catch (Exception e)
                {
                }
            }
        }

        return null;
    }

    /** returns object as Float if possible, null otherwise. */
    
    public static final Float isFloat(Object object)
    {
        if (object != null)
        {
            if (object instanceof Float)
            {
                return (Float)object;
            }
            else
            {
                try
                {
                    return  Float.valueOf(object.toString());
                }
                catch (Exception e)
                {
                }
            }
        }

        return null;
    }

    /** returns object as Long if possible, null otherwise. */
    
    public static final Long isLong(Object object)
    {
        if (object != null)
        {
            if (object instanceof Long)
            {
                return (Long)object;
            }
            else
            {
                try
                {
                    return new Long(((Double)isDouble(object)).longValue());
                }
                catch (Exception e)
                {
                }
            }
        }
        
        return null;
    }

    /** returns object as Integer if possible, null otherwise. */
    
    public static final Integer isInteger(Object object)
    {
        if (object != null)
        {
            if (object instanceof Integer)
            {
                return (Integer)object;
            }
            else
            {
                try
                {
                    return new Integer(((Long)isLong(object)).intValue());
                }
                catch (Exception e)
                {
                }
            }
        }
        
        return null;
    }

    /** returns object as Short if possible, null otherwise. */
    
    public static final Short isShort(Object object)
    {
        if (object != null)
        {
            if (object instanceof Short)
            {
                return (Short)object;
            }
            else
            {
                try
                {
                    return new Short(((Long)isLong(object)).shortValue());
                }
                catch (Exception e)
                {
                }
            }
        }
        
        return null;
    }

    /** returns object as Byte if possible, null otherwise. */
    
    public static final Byte isByte(Object object)
    {
        if (object != null)
        {
            if (object instanceof Byte)
            {
                return (Byte)object;
            }
            else
            {
                try
                {
                    return new Byte(((Long)isLong(object)).byteValue());
                }
                catch (Exception e)
                {
                }
            }
        }
        
        return null;
    }

    /** returns object as String if possible, null otherwise. */

    public static final String isString(Object object)
    {
        if (object != null)
        {
            try
            {
                if (object instanceof InputStream)
                {
                    return StringUtil.toBinaryString(IOUtil.readAll((InputStream)object));
                }
                else if (object instanceof Reader)
                {
                    return new String(IOUtil.readAll((Reader)object));
                }
                else if (object instanceof Node)
                {
                    Node node = (Node)object;

                    Transformer transformer = TransformerFactory.newInstance().newTransformer();
                    transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
                    
                    StringWriter writer = new StringWriter();
                    
                    Source source = new DOMSource(node);
                    Result result = new StreamResult(writer);
                    
                    transformer.transform(source, result);

                    return writer.toString();
                }
                else if (object instanceof javax.mail.internet.MimeMessage)
                {
                    ByteArrayOutputStream bOut = new ByteArrayOutputStream();

                    ((javax.mail.internet.MimeMessage)object).writeTo(bOut);

                    return bOut.toString();
                }
                else
                {
                    return object.toString();
                }
            }
            catch (Exception e)
            {
            }
        }

        return null;
    }

    /** returns object as Boolean if possible, null otherwise. */

    public static final Boolean isBoolean(Object object)
    {
        if (object != null)
        {
            if (object instanceof Boolean)
            {
                return (Boolean)object;
            }
            else
            {
                try
                {
                    return Boolean.valueOf(object.toString());
                }
                catch (Exception e)
                {
                }
            }
        }

        return null;
    }

    /** returns true if object is boolean "true" */

    public static final boolean isTrue(Object object)
    {
        Boolean b = isBoolean(object);

        return ((b != null) && (b.booleanValue()));
    }

    /** returns true if object is boolean "false" */

    public static final boolean isFalse(Object object)
    {
        Boolean b = isBoolean(object);

        return ((b != null) && (!b.booleanValue()));
    }

    /** returns object as Character if possible, null otherwise. */
    
    public static final Character isCharacter(Object object)
    {
        if (object != null)
        {
            if (object instanceof Character)
            {
                return (Character)object;
            }
            else
            {
                String s = object.toString();
                
                if (s.length() > 0)
                {
                    return new Character(s.charAt(0));
                }
            }
        }
        
        return null;
    }

    /** returns object as Object[] if object is an array, null otherwise */

    public static final Object[] isArray(Object object)
    {
        if (object != null)
        {
            if (object instanceof Object[])
            {
                return (Object[])object;
            }
            else if (object instanceof Collection)
            {
                return ((Collection)object).toArray();
            }
            else if (object.getClass().isArray())
            {
                int i = Array.getLength(object);

                Object[] array = new Object[i];
                
                while (--i >= 0)
                {
                    array[i] = Array.get(object, i);
                }

                return array;
            }
        }

        return null;
    }

    /** returns object as list of Strings if possible, null otherwise */

    public static final String[] isStringArray(Object object)
    {
        String[] s = null;

        if (object != null)
        {
            if (object instanceof String[])
            {
                s = (String[])object;
            }
            else if (object instanceof Object[])
            {
                Object[] array = (Object[])object;
                int arraySize = array.length;

                s = new String[arraySize];

                for (int i = arraySize; --i >=0; )
                {
                    Object a = array[i];

                    s[i] = (a != null) ? a.toString() : null;
                }
            }
            else if (object instanceof List)
            {
                List list = (List)object;
                int listSize = list.size();

                s = new String[listSize];

                for (int i = listSize; --i >=0; )
                {
                    Object a = list.get(i);

                    s[i] = (a != null) ? a.toString() : null;
                }
            }
        }

        return s;
    }

    /** returns object as List if possible, null otherwise. */

    public static final List isList(Object object)
    {
        if (object != null)
        {
            if (object instanceof List)
            {
                return (List)object;
            }
            else
            {
                Object[] array = isArray(object);

                if (array != null)
                {
                    return Arrays.asList(array);
                }
            }
        }

        return null;
    }

    /** returns list of Strings */

    public static final List isStringList(Object object, char separator, char escape)
    {
        List list = null;

        if (object != null)
        {
            if (object instanceof List)
            {
                list = (List)object;
            }
            else
            {
                list = StringUtil.split(object.toString(), separator, escape);
            }
        }

        return list;
    }
    
    /** returns as Collection if possible, null otherwise */

    public static final Collection isCollection(Object object)
    {
        if (object instanceof Collection)
        {
            return (Collection)object;
        }
        else
        {
            return isList(object);
        }
    }

    /** returns as URI if possible, null otherwise */

    public static final URI isURI(Object object)
    {
        URI uri = null;

        if (object != null)
        {
            try
            {
                if (object instanceof URI)
                {
                    uri = (URI)object;
                }
                else
                {
                    uri = new URI(object.toString());
                }
            }
            catch (Exception e)
            {
                uri = null;
            }
        }

        return uri;
    }

    /** returns as URL if possible, null otherwise */

    public static final URL isURL(Object object, URL contextURL)
    {
        URL url = null;

        if (object != null)
        {
            try
            {
                url = new URL(contextURL, object.toString());
            }
            catch (Exception e)
            {
                url = null;
            }
        }

        return url;
    }

    /** returns as URL if possible, null otherwise */

    public static final URL isURL(Object object)
    {
        URL url = null;

        if (object != null)
        {
            try
            {
                url = new URL(object.toString());
            }
            catch (Exception e)
            {
                url = null;
            }
        }

        return url;
    }

    /** returns as File if possible, otherwise null */

    public static final File isFile(Object object, Object parentObject)
    {
        File file = null;

        if (object != null)
        {
            try
            {
                if (object instanceof URI)
                {
                    file = new File((URI)object);
                }
                else if (object instanceof File)
                {
                    file = (File)object;
                }
                else
                {
                    file = new File(object.toString());
                }

                if (parentObject != null)
                {
                    file = new File(isFile(parentObject), file.toString());
                }
            }
            catch (Exception e)
            {
                file = null;
            }
        }

        return file;
    }

    public static final File isFile(Object object)
    {
        return isFile(object, null);
    }

    /** converts to NestedMap */

    public static final NestedMap isNestedMap(Object object)
    {
        if (object instanceof NestedMap)
        {
            return (NestedMap)object;
        }
        else
        {
            try
            {
                return new NestedMap(object);
            }
            catch (Exception e)
            {
                return null;
            }
        }
    }

    /** returns Locale for given spec (map or string).  If object is null or empty, default locale is returned. */

    public static final Locale isLocale(Object object)
    {
        if (object instanceof Locale)
        {
            return (Locale)object;
        }
        else
        {
            NestedMap specMap = isNestedMap(object);

            String language = specMap.getString("language", "");
            String country = specMap.getString("country", "");
            String variant = specMap.getString("variant", "");
                
            return new Locale(language, country, variant);
        }
    }

    /** returns TimezZone for given spec (map or string).  If object is null or empty, default TimeZone is returned. */

    public static final TimeZone isTimeZone(Object object)
    {
        if (object instanceof TimeZone)
        {
            return (TimeZone)object;
        }
        else
        {
            NestedMap specMap = isNestedMap(object);

            String id = specMap.getString("id", "");
            
            if (id == null)
            {
                return TimeZone.getDefault();
            }
            else
            {
                return TimeZone.getTimeZone(id);
            }
        }
    }


    /** returns object as Calendar if possible */

    public static final Calendar isCalendar(Object object, String pattern, Locale locale) throws Exception
    {
        Calendar calendar = null;

        if (object != null)
        {
            if (object instanceof Calendar)
            {
                calendar = (Calendar)object;
            }
            else
            {
                if (locale == null)
                {
                    locale = Locale.getDefault();
                }

                calendar = Calendar.getInstance(locale);

                Date date = isDate(object, pattern, locale);

                calendar.setTime(date);
            }
        }

        return calendar;
    }

    /** returns object as Date if possible.*/

    public static final Date isDate(Object object, String pattern, Locale locale) throws Exception
    {
        Date date = null;

        if (object != null)
        {
            if (object instanceof Date)
            {
                date = (Date)object;
            }
            else if (object instanceof Calendar)
            {
                date = ((Calendar)object).getTime();
            }
            else
            {
                if (object instanceof Number)
                {
                    Long longObject = (Long)isLong(object);

                    date = new Date();
                    date.setTime(longObject.longValue());
                }
                else
                {
                    SimpleDateFormat simpleDateFormat;
                    
                    if (pattern == null)
                    {
                        simpleDateFormat = new SimpleDateFormat();
                    }
                    else
                    {
                        simpleDateFormat = new SimpleDateFormat(pattern, locale);
                    }
                    
                    ParsePosition parsePosition = new ParsePosition(0);
                    
                    String objectString = object.toString();
                    
                    date = simpleDateFormat.parse(objectString, parsePosition);
                    
                    if (parsePosition.getIndex() != objectString.length())
                    {
                        date = null;
                    }
                }
            }
        }

        return date;
    }

    /** returns object as number if possible, according to the spec */

    public static final Number isNumber(Object object, String pattern, Locale locale) throws Exception
    {
        Number number = null;

        if (object != null)
        {
            if (object instanceof Number)
            {
                number = (Number)object;
            }
            else
            {
                DecimalFormat decimalFormat;

                if (pattern == null)
                {
                    decimalFormat = new DecimalFormat();
                }
                else
                {
                    decimalFormat = new DecimalFormat(pattern, new DecimalFormatSymbols(locale));
                }

                ParsePosition parsePosition = new ParsePosition(0);
                
                String objectString = object.toString();

                number = decimalFormat.parse(objectString, parsePosition);

                if (parsePosition.getIndex() != objectString.length())
                {
                    number = null;
                }
            }
        }

        return number;
    }

    /** returns length if possible, -1 otherwise */

    public static final int length(Object object)
    {
        if (object != null)
        {
            if (object instanceof String)
            {
                return object.toString().length();
            }
            else if (object instanceof Collection)
            {
                return ((Collection)object).size();
            }
            else if (object.getClass().isArray())
            {
                return Array.getLength(object);
            }
        }

        return -1;
    }

    /** alias for 'length' */

    public static final int size(Object object)
    {
        return length(object);
    }

    /** actual size if object is sparse, otherwise length */

    public static final int getActualSize(Object object)
    {
        if (object instanceof SparseObject)
        {
            return ((SparseObject)object).getActualSize();
        }
        else
        {
            return size(object);
        }
    }

    /** returns true if given object is either null or "" */

    public static final boolean isEmptyString(Object object)
    {
        return ((object == null) || ("".equals(object)));
    }

    /** loose check for 'emptiness', returns true if the object is null, "", or an empty collection or an array. */

    public static final boolean isEmpty(Object object)
    {
        if (isEmptyString(object))
        {
            return true;
        }
        else if (object instanceof Collection)
        {
            return ((Collection)object).isEmpty();
        }
        else if (object.getClass().isArray())
        {
            return (Array.getLength(object) == 0);
        }
        else
        {
            return false;
        }
    }

    /** convert object to given class if possible, null otherwise.  If classObject is a primitive or its wrapper class (say, "int" or "Integer"), then tries to convert to the wrapper class. */

    public static final Object isClass(Object object, Class classObject)
    {
        if (classObject.isInstance(object))
        {
            return object;
        }
        else
        {
            try
            {
                if ((classObject == Boolean.TYPE) || (classObject == Boolean.class))
                {
                    return Boolean.valueOf(object.toString());
                }
                else if ((classObject == Byte.TYPE) || (classObject == Byte.class))
                {
                    return Byte.valueOf(object.toString());
                }
                else if ((classObject == Short.TYPE) || (classObject == Short.class))
                {
                    return Short.valueOf(object.toString());
                }
                else if ((classObject == Integer.TYPE) || (classObject == Integer.class))
                {
                    return Integer.valueOf(object.toString());
                }
                else if ((classObject == Long.TYPE) || (classObject == Long.class))
                {
                    return Long.valueOf(object.toString());
                }
                else if ((classObject == Float.TYPE) || (classObject == Float.class))
                {
                    return Float.valueOf(object.toString());
                }
                else if ((classObject == Double.TYPE) || (classObject == Double.class))
                {
                    return Double.valueOf(object.toString());
                }
                else if ((classObject == Character.TYPE) || (classObject == Character.class))
                {
                    return new Character(object.toString().charAt(0));
                }
                else if (classObject == String.class)
                {
                    return object.toString();
                }
            }
            catch (Exception e)
            {
            }
        }

        return null;
    }

    /** converts given object to Pattern */

    public static final Pattern isPattern(Object object)
    {
        Pattern pattern = null;

        try
        {
            if (object instanceof Pattern)
            {
                pattern = (Pattern)object;
            }
            else if (!isEmptyString(object))
            {
                pattern = Pattern.compile(object.toString());
            }
        }
        catch (Exception e)
        {
        }

        return pattern;
    }

    /** returns Class with given className.  ClassName can be primitive type (e.g., "int", "double", etc.).  If className is not primitive but doesn't contain a ".", then defaultPrefix is added to the className.  If defaultPrefix = null, then "java.lang." is assumed.*/

    public static final Class forName(String className, String defaultPrefix) throws Exception
    {
        int dimensions = 0;

        for (;;)
        {
            int arrayIndex = className.lastIndexOf("[]");

            if (arrayIndex == -1)
            {
                break;
            }
            else
            {
                dimensions ++;
                className = className.substring(0, arrayIndex);
            }
        }

        Class classObject = null;

        if (className.indexOf('.') >= 0)
        {
            classObject = Class.forName(className);
        }
        else
        {
            Class primitiveClass = (Class)(PRIMITIVETYPE.get(className));

            if (primitiveClass != null)
            {
                classObject = primitiveClass;
            }
            else
            {
                classObject = Class.forName(defaultPrefix + className);
            }
        }
        
        if (dimensions == 0)
        {
            return classObject;
        }
        else
        {
            return (Array.newInstance(classObject, new int[dimensions])).getClass();
        }
    }

    /** forName with default "java.lang." prefix */

    public static final Class forName(String className) throws Exception
    {
        return forName(className, "java.lang.");
    }

    /** dynamic method call */

    public static final Object invoke(Object object, Class classObject, String methodName, Class[] types, Object[] args) throws Exception
    {
        if (classObject == null)
        {
            classObject = object.getClass();
        }

        if ("*".equals(methodName))
        {
            Constructor c = classObject.getConstructor(types);

            return c.newInstance(args);
        }
        else if (methodName.startsWith("."))
        {
            Field field = classObject.getDeclaredField(methodName.substring(1));
            return field.get(object);
        }
        else
        {
            Method m = classObject.getMethod(methodName, types);

            return m.invoke(object, args);
        }
    }

    /** method invoker */

    public final static String OBJECT = "object";
    public final static String CLASS = "class";
    public final static String METHOD = "method";
    public final static String TYPE = "type";
    public final static String VALUE = "value";
    public final static String OBJECTPROPERTY = "objectProperty";

    public static Object invoke(NestedMap map) throws Exception
    {
        Object returnValue = null;

        Object object = map.get(OBJECT);
        List list = map.getSubList(false);

        Class[] argTypes = null;
        Object[] argValues = null;

        if (list != null)
        {
            int listSize = list.size();

            argTypes = new Class[listSize];
            argValues = new Object[listSize];

            for (int i = 0; i < listSize; i ++)
            {
                Object argSpec = list.get(i);

                Object argType = null;
                Object argValue = argSpec;

                if (argSpec instanceof Map)
                {
                    Map argMap = (Map)argSpec;

                    argType = argMap.get(TYPE);
                    argValue = argMap.get(VALUE);
                }

                Class argClass = Object.class;

                if (argType instanceof String)
                {
                    argClass = TypeUtil.forName(argType.toString());
                }
                else if (argType instanceof Class)
                {
                    argClass = (Class)argType;
                }
                else if (argType != null)
                {
                    argClass = argType.getClass();
                }
                else if (argValue != null)
                {
                    argClass = argValue.getClass();
                }

                argTypes[i] = argClass;

                argValues[i] = TypeUtil.isClass(argValue, argClass);
            }
        }

        String methodName = (String)map.get(METHOD);

        Class classObject = null;

        Object classSpec = map.get(CLASS);

        if (classSpec != null)
        {
            if (classSpec instanceof Class)
            {
                classObject = (Class)classSpec;
            }
            else
            {
                classObject = TypeUtil.forName(classSpec.toString());
            }
        }

        return invoke(object, classObject, methodName, argTypes, argValues);
    }

    /** invoke method on an object instance */

    public static Object invoke(Object object, NestedMap map, String objectProperty) throws Exception
    {
        map.put(objectProperty, object);

        return invoke(map);
    }    

    /** invoke method on an object instance */

    public static Object invoke(Object object, NestedMap map) throws Exception
    {
        return invoke(object, map, map.getString(OBJECTPROPERTY, OBJECT));
    }    

    /** sublist, with index handling */

    public static List subList(List list, int start, int end)
    {
        if (list != null)
        {
            int length = list.size();
            
            if (start < 0)
            {
                start = length + start;
            }
            
            if (end < 0)
            {
                end = length + end;
            }
            
            if (start < 0)
            {
                start = 0;
            }
            
            if (end > length)
            {
                end = length;
            }
            
            if (end <= start)
            {
                list = new ArrayList();
            }
            else
            {
                list = list.subList(start, end);
            }
        }

        return list;
    }
}
