/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/

package com.micronova.util;

import java.net.*;
import javax.net.ssl.*;
import java.util.*;
import java.io.*;
import java.security.*;
import java.util.regex.*;

/** network utilities */

public class NetUtil implements Runnable
{
    /** HTML META Content Type Pattern */

    public static final Pattern htmlMetaPattern = Pattern.compile("(?i)<[ ]*META[ ]*HTTP-EQUIV[ ]*=[ ]*\"?CONTENT-TYPE\"?[ ]*CONTENT[ ]*=[ ]*\"([^\"]+)\"[ ]*/?>");

    /** NestedMap for threaded request */

    private NestedMap _map;

    /** protocol handler package property */

    private final static String HANDLERPKGS = "java.protocol.handler.pkgs";

    /** sun's protocol handler package */

    private final static String SUNNETSSL = "sun.net.ssl.internal.www.protocol";
    /** customizable version of url.openConnection().  Given trustManager or hostnameVerifier are used if not null. */

    public static URLConnection openConnection(URL url, TrustManager trustManager, HostnameVerifier hostnameVerifier) throws Exception
    {
        URLConnection c = null;

        System.setProperty(HANDLERPKGS, SUNNETSSL);
            
        c = url.openConnection();
            
        if (c instanceof HttpsURLConnection)
        {
            HttpsURLConnection sc = (HttpsURLConnection)c;
            
            SSLContext sslContext = SSLContext.getInstance("TLS");
            
            if (trustManager != null)
            {
                TrustManager[] tm = new TrustManager[] {trustManager};
                sslContext.init(null, tm, null);
                
                SSLSocketFactory factory = sslContext.getSocketFactory();
                
                sc.setSSLSocketFactory(factory);
            }
            
            if (hostnameVerifier != null)
            {
                sc.setHostnameVerifier(hostnameVerifier);
            }
        }

        return c;
    }

    /** simpler version, selectively disables TrustManager/HostnameVerifier */

    public static URLConnection openConnection(URL url, boolean usesTrustManager, boolean usesHostnameVerifier) throws Exception
    {
        TrustManager trustManager = null;

        if (!usesTrustManager)
        {
            trustManager = new DefaultTrustManager();
        }

        HostnameVerifier hostnameVerifier = null;

        if (!usesHostnameVerifier)
        {
            hostnameVerifier = new DefaultHostnameVerifier();
        }

        return openConnection(url, trustManager, hostnameVerifier);
    }

    /** mime map properties */

    public final static String MIMETYPE = com.micronova.util.cc.mime.Parser.TYPE;
    public final static String MIMESUBTYPE = com.micronova.util.cc.mime.Parser.SUBTYPE;
    public final static String MIMEPARAMETER = com.micronova.util.cc.mime.Parser.PARAMETER;

    /** parses MIME into type/subType/parameters */
    
    public final static Map parseMime(String mime)
    {
        return com.micronova.util.cc.mime.Parser.parse(mime);
    }

    /** converts parsed MIME to string */

    public final static String encodeMime(Map mimeMap)
    {
        StringBuffer buffer = new StringBuffer();

        buffer.append(mimeMap.get(MIMETYPE));

        Object subType = mimeMap.get(MIMESUBTYPE);

        if (subType != null)
        {
            buffer.append("/");
            buffer.append(subType);
        }

        Map map = (Map)mimeMap.get(MIMEPARAMETER);

        if (map != null)
        {
            Iterator iterator = map.entrySet().iterator();

            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();

                buffer.append(";");
                buffer.append(entry.getKey());
                buffer.append("=\"");
                buffer.append(entry.getValue());
                buffer.append("\"");
            }
        }

        return buffer.toString();
    }

    /** Map-based HTTP request */

    public static final String USESTRUSTMANAGER = "usesTrustManager";
    public static final String USESHOSTNAMEVERIFIER = "usesHostnameVerifier";
    public static final String URL = "url";
    public static final String METHOD = "method";
    public static final String THROWSEXCEPTION = "throwsException";
    public static final String FOLLOWSREDIRECTS = "followsRedirects";
    public static final String ALLOWSFILE = "allowsFile";

    public static final String REQUEST = "request";
    public static final String RESPONSE = "response";

    public static final String HEADER = "header";
    public static final String ENCODING = "encoding";
    public static final String DEFAULTENCODING = "defaultEncoding";
    public static final String CONTENT = "content";
    public static final String TYPE = "type";
    public static final String PARAMETER = "parameter";
    public static final String AUTHORIZATION = "authorization";
    public static final String USER = "user";
    public static final String PASSWORD = "password";
    public static final String STATUS = "status";
    public static final String MESSAGE = "message";
    public static final String BINARYENCODING = "iso-8859-1";

    /** makes an HTTP request */

    public static NestedMap request(NestedMap map) throws Exception
    {
        OutputStream out = null;
        InputStream in = null;

        try
        {
            String urlString = map.getString(URL);
            String method = map.getString(METHOD, "post");

            boolean isGet = ("get".equalsIgnoreCase(method));

            boolean usesTrustManager = (TypeUtil.isTrue(map.get(USESTRUSTMANAGER)));
            boolean usesHostnameVerifier = (TypeUtil.isTrue(map.get(USESHOSTNAMEVERIFIER)));
            Map request = (Map)map.get(REQUEST);

            String requestContent = null;
            String encoding = null;

            if (request != null)
            {
                encoding = (String)request.get(ENCODING);

                if (encoding == null)
                {
                    encoding = BINARYENCODING;
                }

                requestContent = (String)request.get(CONTENT);

                if (requestContent == null)
                {
                    Map requestParameters = (Map)request.get(PARAMETER);

                    if ((requestParameters != null) && (!requestParameters.isEmpty()))
                    {
                        StringBuffer buffer = new StringBuffer();

                        boolean isAppended = false;

                        Iterator iterator = requestParameters.entrySet().iterator();

                        while (iterator.hasNext())
                        {
                            Map.Entry entry = (Map.Entry)iterator.next();

                            Object key = entry.getKey();
                            Object value = entry.getValue();
                            
                            String keyString = NestedMap.encodeString((key != null) ? key.toString() : "", encoding);

                            List valueList = TypeUtil.isList(value);

                            if (valueList != null)
                            {
                                Iterator listIterator = valueList.iterator();

                                while (listIterator.hasNext())
                                {
                                    if (isAppended)
                                    {
                                        buffer.append("&");
                                    }
                            
                                    buffer.append(keyString);
                                    buffer.append("=");
                                    buffer.append(NestedMap.encodeString(listIterator.next().toString(), encoding));
                                    isAppended = true;
                                }
                            }
                            else
                            {
                                String valueString = (value != null) ? value.toString() : "";
                            
                                if (isAppended)
                                {
                                    buffer.append("&");
                                }
                                
                                buffer.append(keyString);
                                buffer.append("=");
                                buffer.append(NestedMap.encodeString(valueString, encoding));
                                isAppended = true;
                            }
                        }

                        requestContent = buffer.toString();
                    }
                }

                if (isGet)
                {
                    if (requestContent != null)
                    {
                        urlString += ("?" + requestContent);
                    }
                }
            }

            URL url = new URL(urlString);

            if (!TypeUtil.isTrue(map.get(ALLOWSFILE)))
            {
                if ("file".equalsIgnoreCase(url.getProtocol()))
                {
                    return null;
                }
            }

            URLConnection c = NetUtil.openConnection(url, usesTrustManager, usesHostnameVerifier);

            HttpURLConnection hc = null;

            if (c instanceof HttpURLConnection)
            {
                hc = (HttpURLConnection)c;
            }
            
            c.setAllowUserInteraction(false);
            c.setDoInput(true);
            c.setDoOutput(true);

            if (TypeUtil.isTrue(map.get(FOLLOWSREDIRECTS)))
            {
                if (hc != null)
                {
                    hc.setInstanceFollowRedirects(true);
                }
            }

            if (request != null)
            {
                Map requestHeaders = (Map)request.get(HEADER);

                if (requestHeaders != null)
                {
                    Iterator iterator = requestHeaders.entrySet().iterator();
                
                    while (iterator.hasNext())
                    {
                        Map.Entry entry = (Map.Entry)iterator.next();

                        String headerName = entry.getKey().toString();

                        Object headerValue = entry.getValue();

                        if (headerValue instanceof List)
                        {
                            List list = (List)headerValue;

                            Iterator valueIterator = list.iterator();

                            while (valueIterator.hasNext())
                            {
                                c.addRequestProperty(headerName, valueIterator.next().toString());
                            }
                            
                        }
                        else
                        {
                            c.addRequestProperty(headerName, headerValue.toString());
                        }
                    }
                }

                String authorization = (String)request.get(AUTHORIZATION);
                
                if ("basic".equals(authorization))
                {
                    String user = (String)request.get(USER);
                    String password = (String)request.get(PASSWORD);
                    
                    if ((user != null) && (password != null))
                    {
                        String credential = user + ":" + password;
                    
                        String encoded = (new sun.misc.BASE64Encoder()).encode(credential.getBytes());
                        c.addRequestProperty("Authorization", "Basic " + encoded);
                    }
                }
            }

            c.connect();

            if (!isGet)
            {
                if (requestContent != null)
                {
                    out = c.getOutputStream();

                    byte[] data = requestContent.getBytes(encoding);
            
                    out.write(data, 0, data.length);
            
                    out.close();
                    
                    out = null;
                }
            }

            NestedMap response = map.getSubMap(RESPONSE);
            
            String responseEncoding = response.getString(ENCODING);
            String responseDefaultEncoding = response.getString(DEFAULTENCODING, BINARYENCODING);

            Map responseHeaders = c.getHeaderFields();

            if (responseHeaders != null)
            {
                Map responseHeaderMap = new NestedMap();

                response.put(HEADER, responseHeaderMap);
                
                Iterator iterator = responseHeaders.entrySet().iterator();
                
                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();
                        
                    Object headerKey = entry.getKey();
                    Object headerValue= entry.getValue();
                        
                    if (headerKey != null)
                    {
                        headerKey = headerKey.toString().toLowerCase();
                    }
                        
                    responseHeaderMap.put(headerKey, headerValue);
                }
            }

            ByteArrayOutputStream buffer = new ByteArrayOutputStream();
                
            try
            {
                in = c.getInputStream();
                    
                IOUtil.copy(in, buffer);
            }
            catch (Exception ee)
            {
                IOUtil.tryClose(in);
                    
                buffer.reset();
                    
                if (hc != null)
                {
                    in = hc.getErrorStream();
                        
                    if (in != null)
                    {
                        IOUtil.copy(in, buffer);
                            
                        IOUtil.tryClose(in);
                    }
                }
                    
                if (TypeUtil.isTrue(map.get(THROWSEXCEPTION)))
                {
                    throw ee;
                }
            }
                
            if (hc != null)
            {
                response.put(STATUS, new Integer(hc.getResponseCode()));
                response.put(MESSAGE, hc.getResponseMessage());
            }
                
            byte[] responseData = buffer.toByteArray();
                
            String contentEncoding = c.getContentEncoding();
                
            if (contentEncoding != null)
            {
                contentEncoding = contentEncoding.toLowerCase();
                    
                if (contentEncoding.indexOf("gzip") >= 0)
                {
                    responseData = StringUtil.decompressGZIP(responseData);
                }
                else if (contentEncoding.indexOf("zip") >= 0)
                {
                    responseData = StringUtil.decompressZip(responseData);
                }
            }
                
            boolean isText = false;
                
            Map mimeType = NetUtil.parseMime(c.getContentType());
                
            response.put(TYPE, mimeType);
                
            String contentString = null;
                
            if (responseEncoding != null)
            {
                encoding = responseEncoding;
            }
            else
            {
                encoding = null;

                if (mimeType != null)
                {
                    String type = (String)mimeType.get(NetUtil.MIMETYPE);
                    
                    if ("text".equals(type))
                    {
                        isText = true;
                        
                        String mimeCharset = (String)((Map)mimeType.get(NetUtil.MIMEPARAMETER)).get("charset");
                        
                        if (mimeCharset != null)
                        {
                            encoding = mimeCharset;
                        }
                    }
                }

                if (encoding == null)
                {
                    String subType = (String)mimeType.get(NetUtil.MIMESUBTYPE);

                    if ("html".equals(subType))
                    {
                        contentString = new String(responseData, BINARYENCODING);

                        Matcher metaMatcher = htmlMetaPattern.matcher(contentString);

                        if (metaMatcher.find())
                        {
                            Map metaType = NetUtil.parseMime(metaMatcher.group(1));

                            String metaCharset = (String)((Map)metaType.get(NetUtil.MIMEPARAMETER)).get("charset");

                            if (metaCharset != null)
                            {
                                encoding = metaCharset;
                            }
                        }
                    }
                }


                if (encoding == null)
                {
                    encoding = responseDefaultEncoding;
                }
                
                response.put(ENCODING, encoding);
            }

            if ((!BINARYENCODING.equals(encoding)) || (contentString == null))
            {
                contentString = new String(responseData, encoding);
            }

            response.put(CONTENT, contentString);
        }
        catch (Exception e)
        {
            throw e;
        }
        finally
        {
            IOUtil.tryClose(out);
            IOUtil.tryClose(in);
        }

        return map;
    }

    /** runnable for threaded request */

    public void run()
    {
        try
        {
            request(_map);
        }
        catch (Exception e)
        {
        }
    }

    /** constructor for threaded request */

    public NetUtil(NestedMap map)
    {
        _map = map;
    }
}
