/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.net.*;

/**

Nested map that accepts keys such as "@customer.name" which is
resolved to the value of "name" in the submap which is the value of
"customer"   The underscore character "_" is reserved as the special key
name prefix.  Especially the value of "_" itself is a list, and its
elements are accessible as "@_.0", "@_.1", ....  For example, 
"@customer._.0.name" indicates the 0-th customer's name ("customer._.0" is another instance of NestedMap).

The following special keys are supported:

__encoded: Form-encoded pairs "X=Y&...", where X is all the nested key names.
E.g., it is something like "customer.firstname=john&customer.lastname=doe"

__source: copy from source

__param: a map consisting of (nested-kay-name, value) pairs.

__keyList: returns list of key Strings at this level (not nested)

__entryList: returns list of (key, value) pairs (not nested)

__valueList: returns list of values (not nested)

__keyListSorted: returns list of keys in sorted order

__entryListSorted: returns list of (key, value) pairs, sorted by key (not nested)
__encodedSorted: same as __encoded, but sorted by key

__listSize: size of "_" list

__listActualSize: actual size of "_" list

#[expression]: [expression] is taken as non-nested expression

*/

public class NestedMap extends MapBean
{
    /** special characters */

    public final static char SEPARATOR = '.';
    public final static char ESCAPE = '_';
    public final static char NESTED = '@';
    public final static char NONNESTED = '#';

    /** special keynames following "_" prefix */

    public final static String ENCODED = "_encoded";
    public final static String CLEAR = "_clear";
    public final static String PARAM = "_param";
    public final static String SOURCE= "_source";
    public final static String LIST = "";
    public final static String LISTSIZE = "_listSize";
    public final static String LISTACTUALSIZE = "_listActualSize";
    public final static String KEYLIST = "_keyList";
    public final static String ENTRYLIST = "_entryList";
    public final static String VALUELIST = "_valueList";
    public final static String KEYLISTSORTED = "_keyListSorted";
    public final static String ENTRYLISTSORTED = "_entryListSorted";
    public final static String ENCODEDSORTED = "_encodedSorted";
    public final static String ENCODING = "_encoding";

    /** special keys for creating submaps and sublists */

    public final static String CREATESUBMAP = "CREATESUBMAP";
    public final static String CREATESUBLIST = "CREATESUBLIST";

    /** list key */

    public final static String LISTKEY = ESCAPE + LIST;

    /** default character encoding */

    protected String _encoding = "utf-8";

    /** used to indicate that a special key ("_xxx") is not supported */

    protected final static String UNSUPPORTED = "UNSUPPORTED";

    /** secondary map for special keys */

    protected NestedMap _secondary;

    /** get secondary map, creating a new map if null if doesCreate is true */

    protected NestedMap getSecondaryMap(boolean doesCreate)
    {
        NestedMap secondary = _secondary;

        if (secondary == null)
        {
            if (doesCreate)
            {
                secondary = new NestedMap();
                _secondary = secondary;
            }
        }

        return secondary;
    }

    /** getObject on the secondary */

    protected Object getSecondary(Object key)
    {
        NestedMap secondary = getSecondaryMap(false);

        if (secondary == null)
        {
            return null;
        }
        else
        {
            return secondary.get(key);
        }
    }

    /** putObject on the secondary */

    protected Object putSecondary(Object key, Object value)
    {
        NestedMap secondary = getSecondaryMap(true);
        
        return secondary.put(key, value);
    }

    /** shallow copy the secondary */

    public void copySecondary(NestedMap map)
    {
        NestedMap mapSecondary = map.getSecondaryMap(false);

        if (mapSecondary != null)
        {
            NestedMap secondary = getSecondaryMap(true);

            secondary.copy(mapSecondary);
        }
    }

    /** clears secondary */

    public void clearSecondary()
    {
        _secondary = null;
    }

    /** clears all, including the secondary */

    public void clear()
    {
        clearSecondary();

        super.clear();
    }

    /** returns special key valules or UNSUPPORTED */

    protected Object getSpecial(String keyString)
    {
        Object returnValue = UNSUPPORTED;

        if (LIST.equals(keyString))
        {
            returnValue = getSubList();
        }
        else if (PARAM.equals(keyString))
        {
            try
            {
                returnValue = exportParams();
            }
            catch (Exception e)
            {
                e.printStackTrace();
                returnValue = null;
            }
        }
        else if (ENCODED.equals(keyString))
        {
            try
            {
                returnValue = encodeMap((Map)exportParams(), _encoding, false);
            }
            catch (Exception e)
            {
                e.printStackTrace();
                returnValue = null;
            }
        }
        else if (ENCODEDSORTED.equals(keyString))
        {
            try
            {
                returnValue = encodeMap((Map)exportParams(), _encoding, true);
            }
            catch (Exception e)
            {
                e.printStackTrace();
                returnValue = null;
            }
        }
        else if (LISTSIZE.equals(keyString))
        {
            List list = getSubList();
            
            returnValue = new Integer(list.size());
        }
        else if (LISTACTUALSIZE.equals(keyString))
        {
            List list = getSubList();
            
            returnValue = new Integer(TypeUtil.getActualSize(list));
        }
        else if (KEYLIST.equals(keyString))
        {
            returnValue = makeList(keySet());
        }
        else if (ENTRYLIST.equals(keyString))
        {
            returnValue = makeList(entrySet());
        }
        else if (VALUELIST.equals(keyString))
        {
            returnValue = makeList(values());
        }
        else if (KEYLISTSORTED.equals(keyString))
        {
            returnValue = makeList((new TreeMap(this)).keySet());
        }
        else if (ENTRYLISTSORTED.equals(keyString))
        {
            returnValue = makeList((new TreeMap(this)).entrySet());
        }
        else if (ENCODING.equals(keyString))
        {
            returnValue = _encoding;
        }

        return returnValue;
    }

    /** URL-encode a string using given encoding */

    public static String encodeString(String string, String encoding) throws Exception
    {
        return URLEncoder.encode(string, encoding);
    }

    /** URL-decode a string using given encoding */

    public static String decodeString(String string, String encoding) throws Exception
    {
        return URLDecoder.decode(string, encoding);
    }

    /** copy from another nested map */

    public void copy(NestedMap map)
    {
        Iterator iterator = map.entrySet().iterator();
            
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            
            Object key = entry.getKey();
            
            putMapObject(key, entry.getValue());
        }

        copySecondary(map);
    }

    /** copies from given source object.  The source object can be either a Map, a List, or a URL-style encoded String ("a=x&b=y...").  If source is a Map, then shallow copy is made from the source map.  If source is an URL-style encoded String, then it is decoded into the NestedMap.  If source is a list, then the "_" list of the NestedMap is populated with new NestedMaps generated using each list element (again a Map, a List, or a URL-style encoded String) as the source. */

    public void copyFromSource(Object source) throws Exception
    {
        if (source instanceof Map)
        {
            Map sourceMap = (Map)source;
            
            Iterator iterator = sourceMap.entrySet().iterator();
            
            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                
                Object key = entry.getKey();
                
                putMapObject(key, entry.getValue());
            }

            /**
            if (source instanceof NestedMap)
            {
                copySecondary((NestedMap)source);
            }
            **/
        }
        else if (source instanceof String)
        {
            importParams((String)source);
        }
        else
        {
            List list = TypeUtil.isList(source);

            if (list != null)
            {
                Iterator iterator = list.iterator();
            
                List mapList = getSubList();

                while (iterator.hasNext())
                {
                    Object item = iterator.next();

                    if (item instanceof Map)
                    {
                        NestedMap itemMap = new NestedMap();

                        itemMap.copyFromSource(item);
                        
                        mapList.add(itemMap);
                    }
                    else
                    {
                        mapList.add(item);
                    }
                }
            }
        }
    }

    /** constructs an empty nested map */

    public NestedMap()
    {
        super();
    }

    /** constructs a nested map initialized from the given source object */

    public NestedMap(Object source) throws Exception
    {
        super();

        copyFromSource(source);
    }

    /** create a new List */

    protected List createList()
    {
        return (List)getObject(this, CREATESUBLIST);
    }

    /** create a new NestedMap */

    protected NestedMap createMap()
    {
        return (NestedMap)getObject(this, CREATESUBMAP);
    }

    /** returns the "_" sublist, creating one if it doesn't exist and doesCreate is true */

    public List getSubList(boolean doesCreate)
    {
        List list = (List)getMapObject(LISTKEY);
        
        if (list == null)
        {
            if (doesCreate)
            {
                list = createList();

                putMapObject(LISTKEY, list);
            }
        }

        return list;
    }

    public List getSubList()
    {
        return getSubList(true);
    }

    /** returns subMap for given key, creating one if it doesn't exist when doesCreate is true*/

    public NestedMap getSubMap(Object key, boolean doesCreate)
    {
        NestedMap map = (NestedMap)get(key);
        
        if (map == null)
        {
            if (doesCreate)
            {
                map = createMap();

                put(key, map);
            }
        }

        return map;
    }

    /** returns subMap for given key, automatically creating it (doesCreate = true) */

    public NestedMap getSubMap(Object key)
    {
        return getSubMap(key, true);
    }

    /** returns subMap at given index in the given list, creating one if it doesn't exist when doesCreate is true */

    public NestedMap getSubMap(List list, int index, boolean doesCreate)
    {
        NestedMap map = null;

        if (list != null)
        {
            map = (NestedMap)list.get(index);
        
            if (map == null)
            {
                if (doesCreate)
                {
                    map = createMap();
                
                    list.set(index, map);
                }
            }
        }
            
        return map;
    }


    /** returns subMap at given index in the given list, creating one if it doesn't exist */

    public NestedMap getSubMap(List list, int index)
    {
        return getSubMap(list, index, true);
    }

    /** returns subMap at given index of the sublist, creating one if it doesn't exist when doesCreate is true */

    public NestedMap getSubMap(int index, boolean doesCreate)
    {
        return getSubMap(getSubList(doesCreate), index, doesCreate);
    }

    /** returns subMap at given index of the sublist, creating one if it doesn't exist */

    public NestedMap getSubMap(int index)
    {
        return getSubMap(index, true);
    }

    /** get element using nested expression */

    public Object getElement(String name)
    {
        // name = name.replaceAll("\\[([^]]*)\\]", ".$1");

        return getElement(name, false);
    }

    /** get element using nested expression */

    protected Object getElement(String name, boolean isList)
    {
        String key = name;
        String indexPart = null;
        String subKey = null;

        int breakIndex = name.indexOf(SEPARATOR);

        if (breakIndex >= 0)
        {
            key = name.substring(0, breakIndex);
            subKey = name.substring(breakIndex + 1);
        }

        if (isList)
        {
            List list = (List)get(LISTKEY);

            if (TypeUtil.isEmptyString(key))
            {
                return list;
            }
            else
            {
                int index = Integer.parseInt(key);

                if (index < 0)
                {
                    index = list.size() + index;
                }

                Object listItem = list.get(index);

                if (TypeUtil.isEmptyString(subKey))
                {
                    return listItem;
                }
                else
                {
                    return ((NestedMap)listItem).getElement(subKey, false);
                }
            }
        }
        else
        {
            Object keyObject = get(key);

            if (TypeUtil.isEmptyString(subKey))
            {
                return keyObject;
            }
            else
            {
                if (LISTKEY.equals(key))
                {
                    return getElement(subKey, true);
                }
                else
                {
                    return ((NestedMap)keyObject).getElement(subKey, false);
                }
            }
        }
    }

    /** sets element using expression */

    public void setElement(String name, Object value)
    {
        // name = name.replaceAll("\\[([^]]*)\\]", ".$1");

        setElement(name, value, false);
    }

    /** set element using nested expression */

    protected void setElement(String name, Object value, boolean isList)
    {
        String key = name;
        String indexPart = null;
        String subKey = null;

        int breakIndex = name.indexOf(SEPARATOR);
        
        if (breakIndex >= 0)
        {
            key = name.substring(0, breakIndex);
            subKey = name.substring(breakIndex + 1);
        }

        if (isList)
        {
            List list = getSubList();

            int listSize = list.size();

            int index = -1;

            boolean isInsertMode = false;
            boolean isIgnoreNullMode = false;

            if ("".equals(key))
            {
                key = "*";
            }

            if (key.startsWith("*"))
            {
                isInsertMode = true;
                key = key.substring(1);

                if (key.startsWith("*"))
                {
                    isIgnoreNullMode = true;
                    key = key.substring(1);
                }
            }

            if (key.length() == 0)
            {
                index = listSize;
            }
            else
            {
                index = Integer.parseInt(key);

                if (index < 0)
                {
                    index = listSize + index;
                }
            }

            if (value == REMOVE)
            {
                value = null;
            }

            if (TypeUtil.isEmptyString(subKey))
            {
                if (isInsertMode)
                {
                    if (value != null)
                    {
                        if (index == listSize)
                        {
                            list.add(value);
                        }
                        else
                        {
                            list.add(index, value);
                        }
                    }
                    else if ((listSize > 0) && (!isIgnoreNullMode))
                    {
                        if (index == listSize)
                        {
                            list.remove(listSize - 1);
                        }
                        else
                        {
                            list.remove(index);
                        }
                    }
                }
                else
                {
                    list.set(index, value);
                }
            }
            else
            {
                NestedMap subStruct = getSubMap(list, index);
                subStruct.setElement(subKey, value);
            }
        }
        else
        {
            if (LISTKEY.equals(key))
            {
                if (TypeUtil.isEmptyString(subKey))
                {
                    putMapObject(key, value);
                }
                else
                {
                    setElement(subKey, value, true);
                }
            }
            else
            {
                if (TypeUtil.isEmptyString(subKey))
                {
                    put(key, value);
                }
                else
                {
                    Object mapObject = get(key);

                    if ((mapObject == null) || (mapObject instanceof NestedMap))
                    {
                        NestedMap subStruct = getSubMap(key);
                        subStruct.setElement(subKey, value);
                    }
                }
            }
        }
    }

    /** decode a nested map */

    public void decode(Map parameters) throws Exception
    {
        Iterator iterator = parameters.keySet().iterator();

        while (iterator.hasNext())
        {
            String name = (String)iterator.next();

            if (name != null)
            {
                if (!name.equals(ENCODED))
                {
                    Object value = parameters.get(name);

                    setElement(name, value);
                }
            }
        }
    }

    /** encode a nested map */

    protected void encode(String prefix, NestedMap map)
    {
        Iterator iterator = keySet().iterator();

        while (iterator.hasNext())
        {
            String key = (String)iterator.next();

            if (key != null)
            {
                if (key.equals(LISTKEY))
                {
                    List list = getSubList();

                    for (int i = list.size(); --i >= 0;)
                    {
                        Object value = list.get(i);
                        
                        if (value != null)
                        {
                            if (value instanceof NestedMap)
                            {
                                ((NestedMap)value).encode(prefix + ESCAPE + SEPARATOR + i + SEPARATOR, map);
                            }
                            else
                            {
                                map.putMapObject(prefix + ESCAPE + SEPARATOR + i, value);
                            }
                        }
                    }
                }
                else
                {
                    Object value = getMapObject(key);

                    if (value != null)
                    {
                        if (value instanceof NestedMap)
                        {
                            ((NestedMap)value).encode(prefix + key + SEPARATOR, map);
                        }
                        else
                        {
                            map.putMapObject(prefix + key, value);
                        }
                    }
                }
            }
        }
    }

    /** export into nested key/value pairs */

    public Map exportParams() throws Exception
    {
        NestedMap map = new NestedMap();

        encode("", map);

        return map;
    }

    /** import from nested form-data string */

    public void importParams(String encoded) throws Exception
    {
        if (encoded != null)
        {
            encoded = encoded.replaceAll("[\\s]", "");

            Map allMap = new HashMap(decodeMap(encoded, _encoding));

            decode(allMap);
        }
    }

    /** import from parameter map (nested key/value pairs) */

    public void importParams(Map parameters) throws Exception
    {
        String encoded = (String)parameters.get(ENCODED);

        importParams(encoded);

        decode(parameters);
    }

    /** gets an object */

    public Object getObject(Object mapSource, Object key)
    {
        if (key == CREATESUBMAP)
        {
            return new NestedMap();
        }
        else if (key == CREATESUBLIST)
        {
            return new SparseList();
        }

        Object returnValue = null;

        if (key != null)
        {
            String keyString = key.toString();

            char prefix = 0;

            if (keyString.length() > 0)
            {
                prefix = keyString.charAt(0);
            }

            if (prefix == NESTED)
            {
                returnValue = getElement(keyString.substring(1));
            }
            else if (prefix == ESCAPE)
            {
                String specialKey = keyString.substring(1);

                Object specialValue = getSpecial(specialKey);

                if (specialValue != UNSUPPORTED)
                {
                    returnValue = specialValue;
                }
                else
                {
                    returnValue = getSecondary(specialKey);
                }
            }
            else
            {
                if (prefix == NONNESTED)
                {
                    keyString = keyString.substring(1);
                }

                returnValue = super.getObject(mapSource, keyString);
            }
        }

        return returnValue;
    }

    /** makes a list from a collection */

    protected List makeList(Collection collection)
    {
        List list = new ArrayList();

        Iterator iterator = collection.iterator();

        while (iterator.hasNext())
        {
            list.add(iterator.next());
        }

        return list;
    }

    /** puts an object */

    public Object putObject(Object mapSource, Object key, Object value)
    {
        if (key != null)
        {
            String keyString = key.toString();

            char prefix = 0;

            if (keyString.length() > 0)
            {
                prefix = keyString.charAt(0);
            }
            
            if (prefix == NESTED)
            {
                setElement(keyString.substring(1), value);
                return null;
            }
            else if (prefix == ESCAPE)
            {
                keyString = keyString.substring(1);

                if (ENCODED.equals(keyString))
                {
                    try
                    {
                        importParams(value.toString());
                    }
                    catch (Exception e)
                    {
                        throw new RuntimeException(e);
                    }
                    
                    return null;
                }
                else if (SOURCE.equals(keyString))
                {
                    try
                    {
                        copyFromSource(value);
                    }
                    catch (Exception e)
                    {
                        throw new RuntimeException(e);
                    }

                    return null;
                }
                else if (PARAM.equals(keyString))
                {
                    try
                    {
                        importParams((Map)value);
                    }
                    catch (Exception e)
                    {
                        throw new RuntimeException(e);
                    }
                
                    return null;
                }
                else if (CLEAR.equals(keyString))
                {
                    clear();
                    return null;
                }
                else if (ENCODING.equals(keyString))
                {
                    _encoding = (String)value;
                    return null;
                }
                else if (!LIST.equals(keyString))
                {
                    return putSecondary(keyString, value);
                }
            }
            else if (prefix == NONNESTED)
            {
                key = keyString.substring(1);
            }
        }

        if (value == null)
        {
            value = REMOVE;
        }

        return super.putObject(mapSource, key, value);
    }

    /** encodes a NestedMap into FormData-style string */

    public static String encodeMap(Map map, String encoding, boolean doesSort) throws Exception
    {
        StringBuffer buffer = new StringBuffer();

        Map cloneMap = (doesSort) ? ((Map)new TreeMap(map)) : ((Map)new HashMap(map));

        boolean isAppended = false;

        Iterator iterator = cloneMap.entrySet().iterator();

        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();

            Object key = entry.getKey();
            Object value = entry.getValue();

            String keyString = (key != null) ? key.toString() : "";
            String valueString = (value != null) ? value.toString() : "";

            if (isAppended)
            {
                buffer.append("&");
            }

            buffer.append(encodeString(keyString, encoding));
            buffer.append("=");
            buffer.append(encodeString(valueString, encoding));

            isAppended = true;
        }

        return buffer.toString();
    }

    /** decodes a FormData-style string into nested map */

    public static NestedMap decodeMap(String string, String encoding) throws Exception
    {
        NestedMap map = new NestedMap();

        String[] lines = string.split("&");

        for (int i = 0; i < lines.length; i ++)
        {
            String[] parts = lines[i].split("=", 2);

            if (parts.length == 2)
            {
                String key = decodeString(parts[0], encoding);
                String value = decodeString(parts[1], encoding);

                if ("".equals(key))
                {
                    key = null;
                }

                if ((key != null) && (value != null))
                {
                    map.setElement(key, value);
                }
            }
        }

        return map;
    }
}
