/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.io.*;
import java.net.*;

import javax.mail.*;
import javax.mail.internet.*;
import javax.activation.*;

/** Simple Map-based mail */

public class MailTransport
{
    public static final String URL = "url";

    public static final String TO = "to";
    public static final String CC = "cc";
    public static final String BCC= "bcc";
    public static final String FROM = "from";
    public static final String REPLYTO = "replyTo";

    public static final String SUBJECT = "subject";
    public static final String TYPE = "type";
    public static final String SENTDATE = "sentDate";

    public static final String HEADER = "header";
    public static final String FILENAME = "fileName";
    public static final String DATASOURCE = "dataSource";
    public static final String CONTENT = "content";

    /** make internet address from an object */

    protected static InternetAddress makeAddress(Object object) throws Exception
    {
        if (object instanceof InternetAddress)
        {
            return (InternetAddress)object;
        }
        else if (object instanceof Map)
        {
            Map map = (Map)object;

            String address = (String)map.get("address");
            String personal = (String)map.get("personal");
            String charset = (String)map.get("charset");

            return new InternetAddress(address, personal, charset);
        }
        else
        {
            return new InternetAddress((String)object);
        }
    }

    /** get address list */

    protected static List getAddressList(Object object)
    {
        if (object == null)
        {
            return null;
        }
        else if (object instanceof NestedMap)
        {
            return ((NestedMap)object).getSubList(false);
        }
        else
        {
            List list = TypeUtil.isList(object);

            if (list == null)
            {
                list = StringUtil.split(object.toString(), ',');
            }

            return list;
        }
    }

    /** sets addresses of a message.  object can be either comma-separated list of addresses, List or array of String or Address */

    protected static void setAddress(Message message, String type, Object object) throws Exception
    {
        List list = getAddressList(object);
                
        if (list != null)
        {
            int listSize = list.size();

            Address[] addresses = null;

            if (listSize > 0)
            {
                addresses = new Address[listSize];
            
                for (int i = listSize; --i >= 0;)
                {
                    addresses[i] = makeAddress(list.get(i));
                }
            }

            if (type == REPLYTO)
            {
                message.setReplyTo(addresses);
            }
            else if (type == FROM)
            {
                message.setFrom(null);
                message.addFrom(addresses);
            }
            else if (type == TO)
            {
                message.setRecipients(Message.RecipientType.TO, addresses);
            }
            else if (type == CC)
            {
                message.setRecipients(Message.RecipientType.CC, addresses);
            }
            else if (type == BCC)
            {
                message.setRecipients(Message.RecipientType.BCC, addresses);
            }
        }
    }

    /** set replyTo or from */

    public static void setReplyTo(MimeMessage message, String type, Object object) throws Exception
    {
        List list = getAddressList(object);
            
        if (list != null)
        {
            int listSize = list.size();
            
            Address[] addresses = new Address[listSize];
            
            for (int i = listSize; --i >= 0;)
            {
                Object listItem = list.get(i);

                InternetAddress address = makeAddress(listItem);

                addresses[i] = address;
            }

            
            message.setReplyTo(addresses);
        }
    }

    /** set sentDate */

    public static void setSentDate(MimeMessage message, Object date) throws Exception
    {
        if (date != null)
        {
            if (date instanceof Date)
            {
                message.setSentDate((Date)date);
            }
            else
            {
                Long longDate = TypeUtil.isLong(date);

                if (longDate != null)
                {
                    message.setSentDate(new Date(longDate.longValue()));
                }
            }
        }
    }

    /** build message parts */

    public static void buildPart(MimePart part, NestedMap map) throws Exception
    {
        NestedMap headerMap = (NestedMap)map.get(HEADER);

        if (headerMap != null)
        {
            List list = headerMap.getSubList(false);

            if (list != null)
            {
                Iterator iterator = list.iterator();

                while (iterator.hasNext())
                {
                    Object header = iterator.next();

                    if (header instanceof Map)
                    {
                        String name = (String)(((Map)header).get("name"));
                        String value = (String)(((Map)header).get("value"));

                        if ((name != null) && (value != null))
                        {
                            part.addHeader(name, value);
                        }
                    }
                    else if (header instanceof String)
                    {
                        part.addHeaderLine(header.toString());
                    }
                }
            }

            Iterator iterator = headerMap.entrySet().iterator();

            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();
                
                Object key = entry.getKey();
                Object value = entry.getValue();
                
                if (value != list)
                {
                    part.setHeader(key.toString(), value.toString());
                }
            }
        }
        
        Object typeObject = map.get(TYPE);
        
        List list = map.getSubList(false);

        if ((list == null) || (list.isEmpty()))
        {
            String content = TypeUtil.isString(map.get(CONTENT));

            if (content == null)
            {
                content = "";
            }

            String type = null;
       
            if (typeObject != null)
            {
                if (typeObject instanceof Map)
                {
                    type = NetUtil.encodeMime((Map)typeObject);
                }
                else if (typeObject instanceof String)
                {
                    type = (String)typeObject;
                }
            }

            if (content != null)
            {
                String typeString = (type != null) ? type : "text/plain; charset=iso-8859-1";
                String fileName = map.getString(FILENAME);

                ByteArrayDataSource byteArrayDataSource = new ByteArrayDataSource(typeString, fileName, content.getBytes("iso-8859-1"));

                part.setDataHandler(new DataHandler(byteArrayDataSource));
            }

            Object dataSource = map.get(DATASOURCE);

            if (dataSource != null)
            {
                if (dataSource instanceof DataSource)
                {
                    part.setDataHandler(new DataHandler((DataSource)dataSource));
                }
                else
                {
                    URL url = new URL(dataSource.toString());

                    part.setDataHandler(new DataHandler(new URLDataSource(url)));
                    part.setFileName(map.getString(FILENAME, url.getFile().replaceAll("/?[^/]*/", "")));
                }

                if (type != null)
                {
                    part.setHeader("Content-Type", type);
                }
            }
        }
        else
        {
            MimeMultipart mp = new MimeMultipart();

            String type = "related";

            if (typeObject instanceof Map)
            {
                type = (String)((Map)typeObject).get(NetUtil.MIMESUBTYPE);
            }
            else
            {
                type = (String)typeObject;
            }
            
            mp.setSubType(type);

            Iterator iterator = list.iterator();

            while (iterator.hasNext())
            {
                NestedMap subMap = (NestedMap)iterator.next();

                MimeBodyPart bodyPart = new MimeBodyPart();

                buildPart(bodyPart, subMap);

                mp.addBodyPart(bodyPart);
            }

            part.setContent(mp);
        }
    }

    /** send map-based email */

    public static void send(Session session, Transport transport, NestedMap map) throws Exception
    {
        MimeMessage message = new MimeMessage(session);

        buildPart(message, map);

        String subject = map.getString(SUBJECT, "");

        message.setSubject(subject);

        setAddress(message, TO, map.get(TO));
        setAddress(message, CC, map.get(CC));
        setAddress(message, BCC, map.get(BCC));
        setAddress(message, REPLYTO, map.get(REPLYTO));
        setAddress(message, FROM, map.get(FROM));
        
        setSentDate(message, map.get(SENTDATE));
        
        transport.send(message);
    }
}
