/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.io.*;

import javax.mail.*;
import javax.mail.internet.*;
import javax.activation.*;

import com.micronova.util.*;

/** MailFolder using NestedMap. */

public class MailFolder
{
    // control keys

    public static final String HEADER = "header";
    public static final String HEADERMAP = "headerMap";
    public static final String CONTENT = "content";
    public static final String MAXCONTENTSIZE = "maxContentSize";
    public static final String MAXPARTSIZE = "maxPartSize";
    public static final String CONTENTASSTRING = "contentAsString";

    // field names

    public static final String RECEIVEDDATE = "receivedDate";
    public static final String MESSAGENUMBER = "messageNumber";
    public static final String SUBJECT = "subject";
    public static final String EXCEPTION = "exception";
    public static final String SENTDATE = "sentDate";
    public static final String TO = "to";
    public static final String CC = "cc";
    public static final String BCC = "bcc";
    public static final String REPLYTO = "replyTo";
    public static final String FROM = "from";
    public static final String FLAGS = "flags";
    public static final String NAME = "name";
    public static final String VALUE = "value";
    public static final String FULLNAME = "fullName";
    public static final String URLNAME = "urlName";
    public static final String MESSAGECOUNT = "messageCount";
    public static final String ADDRESS = "address";
    public static final String PERSONAL = "personal";
    public static final String DESCRIPTION = "description";
    public static final String DISPOSITION = "disposition";
    public static final String FILENAME = "fileName";
    public static final String LINECOUNT = "lineCount";
    public static final String SIZE = "size";
    public static final String CONTENTTYPE = "contentType";
    public static final String TYPE = "type";
    public static final String PARTNAME = "partName";

    public static final String PART = "_part";
    public static final String FOLDER = "_folder";

    public static final int[] ALLMESSAGES = new int[0];

    /** set InternetAddress */

    private static void setAddress(NestedMap map, InternetAddress address)
    {
        map.put(ADDRESS, address.getAddress());
        map.put(PERSONAL, address.getPersonal());
    }

    /** set address */

    private static void setAddresses(NestedMap map, String name, Address[] addresses) throws Exception
    {
        if (addresses != null)
        {
            NestedMap addressListMap = new NestedMap();

            map.put(name, addressListMap);

            List list = addressListMap.getSubList();

            for (int i = 0; i < addresses.length; i ++)
            {
                InternetAddress address = (InternetAddress)addresses[i];

                if (address != null)
                {
                    list.add(address);
                }
            }
        }
    }

    public static String makePartMap(NestedMap partMap, Part part, Map controlMap) throws Exception
    {
        String name = null;

        partMap.put(PART, part);

        partMap.put(DESCRIPTION, part.getDescription());
        partMap.put(FILENAME, part.getFileName());
        partMap.put(LINECOUNT, new Long(part.getLineCount()));
        partMap.put(SIZE, new Long(part.getSize()));
        partMap.put(CONTENTTYPE, part.getContentType());
        partMap.put(TYPE, NetUtil.parseMime(part.getContentType()));

        String[] dispositions = part.getHeader("Content-Disposition");

        if (dispositions != null)
        {
            for (int i = 0; i < dispositions.length; i ++)
            {
                NestedMap dispositionMap = (NestedMap)NetUtil.parseMime(dispositions[i]);

                partMap.put(DISPOSITION, dispositionMap);

                name = (String)dispositionMap.get("@parameter.name");
            }
        }

        if (controlMap != null)
        {
            if (!TypeUtil.isFalse(controlMap.get(HEADER)))
            {
                NestedMap headerMap = new NestedMap();
                
                partMap.put(HEADER, headerMap);
                
                boolean isHeaderMap = (!TypeUtil.isFalse(controlMap.get(HEADERMAP)));
                
                List headerList = headerMap.getSubList();
                
                Enumeration enumHeaders = part.getAllHeaders();
                
                while (enumHeaders.hasMoreElements())
                {
                    Header header = (Header)enumHeaders.nextElement();
                    
                    NestedMap headerElementMap = new NestedMap();
                    
                    String headerName = header.getName();
                    String headerValue = header.getValue();

                    headerElementMap.put(NAME, headerName);
                    headerElementMap.put(VALUE, headerValue);
                    
                    headerList.add(headerElementMap);
                    
                    if (isHeaderMap)
                    {
                        if ((headerName != null) && (headerValue != null))
                        {
                            headerMap.put(headerName, headerValue);
                        }
                    }
                }
            }

            if (!TypeUtil.isFalse(controlMap.get(CONTENT)))
            {
                Integer iMaxSize = TypeUtil.isInteger(controlMap.get(MAXPARTSIZE));

                int maxSize = 0;
                
                if (iMaxSize != null)
                {
                    maxSize = iMaxSize.intValue();
                }

                if ((maxSize <= 0) || (part.getSize() < maxSize))
                {
                    Object content = part.getContent();

                    if (content instanceof Multipart)
                    {
                        Multipart multipart = (Multipart)content;
                        List partMapList = partMap.getSubList();
                        
                        for (int i = 0; i < multipart.getCount(); i ++)
                        {
                            NestedMap subPartMap = new NestedMap();
                            
                            String partName = makePartMap(subPartMap, multipart.getBodyPart(i), controlMap);
                            
                            partMapList.add(subPartMap);
                            
                            if (partName != null)
                            {
                                NestedMap partNameMap = partMap.getSubMap(PARTNAME);
                                partNameMap.put(partName, subPartMap);
                            }
                        }
                    }
                    else
                    {
                        int partSize = part.getSize();

                        Integer iMaxContentSize = TypeUtil.isInteger(controlMap.get(MAXCONTENTSIZE));

                        int maxContentSize = 0;
                        
                        if (iMaxContentSize != null)
                        {
                            maxContentSize = iMaxContentSize.intValue();
                        }

                        if ((maxContentSize <= 0) || (partSize < maxContentSize))
                        {
                            if (partSize > 0)
                            {
                                if (TypeUtil.isTrue(controlMap.get(CONTENTASSTRING)))
                                {
                                    content = TypeUtil.isString(content);
                                }

                                partMap.put(CONTENT, content);
                            }
                        }
                    }
                }
            }
        }

        return name;
    }

    /** reads given messages.  If messageNumbers is null, then no messages are retrieved.  If messageNumbers equal to ALLMESSAGES, then all messages are retrieved. */

    public static Map readMessages(Folder folder, int[] messageNumbers, Map controlMap) throws Exception
    {
        Message[] messages = null;

        if (messageNumbers == ALLMESSAGES)
        {
            messages = folder.getMessages();
        }
        else if (messageNumbers != null)
        {
            messages = folder.getMessages(messageNumbers);
        }

        return makeFolderMap(folder, messages, controlMap);
    }
    
    /** creates a message map */
        
    private static Map makeMessageMap(Message message, Map controlMap)
    {
        NestedMap map = new NestedMap();

        int messageNumber = message.getMessageNumber();

        map.put(MESSAGENUMBER, new Long(messageNumber));

        try
        {
            if (controlMap != null)
            {
                String subject = message.getSubject();

                if (subject != null)
                {
                    map.put(SUBJECT, subject);
                }
                
                Date sentDate = message.getSentDate();
                
                if (sentDate != null)
                {
                    map.put(SENTDATE, sentDate);
                }
                
                Date receivedDate = message.getReceivedDate();
                
                if (receivedDate != null)
                {
                    map.put(RECEIVEDDATE, receivedDate);
                }
                
                setAddresses(map, TO, message.getRecipients(Message.RecipientType.TO));
                setAddresses(map, CC, message.getRecipients(Message.RecipientType.CC));
                setAddresses(map, BCC, message.getRecipients(Message.RecipientType.BCC));
            
                setAddresses(map, REPLYTO,  message.getReplyTo());
                
                setAddresses(map, FROM, message.getFrom());

                makePartMap(map, message, controlMap);
            }
        }
        catch (Exception e)
        {
            map.put(EXCEPTION, e);
        }

        return map;
    }

    private static Map makeFolderMap(Folder folder, Message[] messages, Map controlMap) throws Exception
    {
        NestedMap map = new NestedMap();

        map.put(FOLDER, folder);

        map.put(NAME, folder.getName());
        map.put(FULLNAME, folder.getFullName());
        map.put(URLNAME, folder.getURLName());
        map.put(MESSAGECOUNT, new Integer(folder.getMessageCount()));

        if (messages != null)
        {
            List list = map.getSubList();

            for (int i = 0; i < messages.length; i ++)
            {
                Message message = messages[i];

                Map messageMap = makeMessageMap(message, controlMap);

                list.add(messageMap);
            }
        }

        return map;
    }

    /** delete messages */

    public static void deleteMessages(Folder folder, int[] messageNumbers) throws Exception
    {
        if (messageNumbers != null)
        {
            Message[] messages = folder.getMessages(messageNumbers);

            for (int i = messages.length; --i >= 0;)
            {
                Message message = messages[i];

                message.setFlag(Flags.Flag.DELETED, true);
            }

            // try to expunge if supported

            try
            {
                folder.expunge();
            }
            catch (MethodNotSupportedException e)
            {
            }
        }
    }

    /** decode multipart stream using given header */

    public static NestedMap decodeMultipart(InputStream inputStream, String header, NestedMap controlMap) throws Exception
    {
        StringBuffer buffer = new StringBuffer();

        buffer.append("Content-Type:" + header + "\r\n\r\n");

        ByteArrayInputStream headerInputStream = new ByteArrayInputStream(buffer.toString().getBytes("ISO-8859-1"));

        SequenceInputStream sequenceInputStream = new SequenceInputStream(headerInputStream, inputStream);

        MimeMessage mimeMessage = new MimeMessage(null, sequenceInputStream);

        NestedMap map = new NestedMap();
                
        makePartMap(map, mimeMessage, new NestedMap(controlMap));
                
        return map;
    }
}
