/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.jsp.tag;

import java.lang.reflect.*;
import java.util.*;
import java.util.regex.*;
import java.beans.*;
import java.io.*;
import java.net.*;
import javax.servlet.*;
import javax.servlet.jsp.*;
import javax.servlet.jsp.tagext.*;
import javax.servlet.jsp.jstl.core.Config;
import org.apache.taglibs.standard.tag.common.core.Util;
import com.micronova.util.*;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import org.w3c.dom.*;

/**

Base class for YUZU tag library.

*/

public class YuzuTag extends YuzuRoot implements TryCatchFinally
{
    /** Object indicating 'default value' */

    public static final String DEFAULT = "default";

    /** name of this class (used for parent tag search) */

    public static final String CLASSNAME = "com.micronova.jsp.tag.YuzuTag";

    /** page-scoped variable name for the 'tagvalue' */

    public static final String VALUEVAR = "_";

    /** special target name used for returning a value */

    public static final String RETURNTARGET = "return";

    /** 'value' of this tag (either set by the code or 'value' attribute) */

    protected Object _value;

    /** default value set by 'default' attribute */

    protected Object _defaultValue;

    /** test expression for 'default' attribute */

    protected String _test;

    /** variable name for assignment */

    protected String _var;

    /** target Object for assignment */

    protected Object _target;

    /** attribute name to be set */

    protected String _attribute;

    /** property name to be set */

    protected String _property;

    /** scope of the variable for assignment */

    protected int _scope;
    
    /** class name to be instantiated */

    protected String _className;

    /** codec used on exporting */

    protected String _exportCodec;

    /** codec used on importing */

    protected String _importCodec;

    /** codecs used on processing */

    protected String _processCodec;

    /** codec used on preparation */

    protected String _prepareCodec;

    /** codec used on cleanup */

    protected String _cleanupCodec;

    /** EL expression specifing value to be assigned */

    protected String _assign;

    /** codec used on assignment */

    protected String _assignCodec;

    /** EL expression specifying value to be exported */

    protected String _export;

    /** export mode; 'always' or 'default' */

    protected String _doesExport;

    /** comma-separated list of local variables */

    protected String _local;

    /** local variable map */

    protected Map _localMap;

    /** set to true while inside tag body */

    protected boolean _isInBody;

    /** saves current value of VALUEVAR */

    protected Object _valueVarSaved;

    /** set to true if the tag has body */

    protected boolean _hasBody;

    /** set to true if tagValue is assigned */

    protected boolean _isAssigned;

    /** obtain configuration named configName, or defaultConfiguration if not found */

    protected Object getConfiguration(String configName, Object defaultConfiguration)
    {
        Object configuration = Config.find(pageContext, configName);

        if (configuration == null)
        {
            configuration = defaultConfiguration;
        }

        return configuration;
    }

    /** constructor */

    public YuzuTag()
    {
        super();

        init();
    }
    
    /** initializer.  Since tags may be re-used by the container, this is called both before and after tag processing. */

    protected void init()
    {
        _value = null;
        _defaultValue = null;
        _test = null;
        _var = null;
        _target = null;
        _property = null;
        _scope = PageContext.PAGE_SCOPE; 
        _attribute = null;
        _className = null;
 
        _local = null;
        _localMap = null;
        _valueVarSaved = null;

        _export = DEFAULT;
        _assign = DEFAULT;
        _doesExport = DEFAULT;

        _assignCodec = null;
        _exportCodec = null;
        _importCodec = null;
        _processCodec = null;
        _prepareCodec = null;
        _cleanupCodec = null;

        _isInBody = false;
        _hasBody = false;
        _isAssigned = false;
    }

    /** called after tag processing is done (in doFinally()) */

    protected void cleanup()
    {
        String cleanupCodec = _cleanupCodec;

        if (cleanupCodec != null)
        {
            try
            {
                applyCodec(cleanupCodec, _value);
            }
            catch (Exception e)
            {
                // exception is ignored
            }
        }
    }

    /** applies codecs to an Object */

    protected Object applyCodec(String codecs, Object object) throws Exception
    {
        return EL.applyCodec(pageContext, codecs, object);
    }

    /** sets a page attribute named 'name' to 'value'.  When 'value' is null, then the attribute is removed. */

    protected void setPageAttribute(String name, Object value)
    {
        EL.setPageAttribute(pageContext, name, value);
    }

    /** saves 'local' variables */

    protected void saveLocal() throws Exception
    {
        _valueVarSaved = pageContext.getAttribute(VALUEVAR);

        String local = _local;

        if (local != null)
        {
            Map localMap = new HashMap();
            String[] localArray = local.split("[ ,\t\r\n]+");

            for (int i = 0; i < localArray.length; i ++)
            {
                String key = localArray[i];

                Object value = pageContext.getAttribute(key);

                if (value != null)
                {
                    localMap.put(key, pageContext.getAttribute(key));
                }
            }

            _localMap = localMap;
        }
    }

    /** restores local variables */

    protected void restoreLocal()
    {
        try
        {
            setPageAttribute(VALUEVAR, _valueVarSaved);

            Map localMap = _localMap;
            
            if (localMap != null)
            {
                Iterator iterator = localMap.entrySet().iterator();

                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();

                    setPageAttribute(entry.getKey().toString(), entry.getValue());
                }
            }
        }
        catch (Exception e)
        {
        }
    }

    /** instantiate value with given className */

    protected Object instantiateValue(String className) throws Exception
    {
        Class c = TypeUtil.forName(className);

        return c.newInstance();
    }

    /** prepares tagValue before tag processing starts.  The returned value is assigned to "_value" when tag evaluation starts. */

    protected Object prepareValue(Object tagValue) throws Exception
    {
        if (tagValue == null)
        {
            String className = _className;
        
            if (className != null)
            {
                tagValue = instantiateValue(className);
            }

            String prepareCodec = _prepareCodec;

            if (prepareCodec != null)
            {
                tagValue = applyCodec(prepareCodec, tagValue);
            }
        }

        return tagValue;
    }

    /** called before proecessing the body */

    protected void initBody() throws Exception
    {
        _isInBody = true;
    }

    /** called after processing the body */

    protected void afterBody() throws Exception
    {
        _isInBody = false;
        _hasBody = true;
    }

    public void doInitBody() throws JspException
    {
        try
        {
            super.doInitBody();

            initBody();
        }
        catch (Exception e)
        {
            throw new JspException(e);
        }
    }

    public int doAfterBody() throws JspException
    {
        try
        {
            afterBody();

            return super.doAfterBody();
        }
        catch (Exception e)
        {
            throw new JspException(e);
        }
    }

    public int doStartTag() throws JspException
    {
        try
        {
            doPrepare();

            saveLocal();

            setPageAttribute(VALUEVAR, _value);

            return EVAL_BODY_BUFFERED;
        }
        catch (Exception e)
        {
            throw new JspException(e);
        }
    }

    public int doEndTag() throws JspException
    {
        try
        {
            Object importedValue = doImport(_value);

            /** apply default */

            Object processValue = doDefault(importedValue, _defaultValue, _test);

            /** process value */

            Object processedValue = doProcess(processValue);

            /** assign value */

            boolean isAssigned = doAssign(processedValue);

            _isAssigned = isAssigned;

            /** export value */

            if (doesExport(processedValue, isAssigned))
            {
                doExport(processedValue);
            }

            return EVAL_PAGE;
        }
        catch (Throwable e)
        {
            if (e instanceof JspException)
            {
                throw (JspException)e;
            }
            else
            {
                throw new JspException(e);
            }
        }
    }

    /** returns true if tag body needs to be imported.  Default implementation returns true if value is null. */

    protected boolean doesImport(Object value)
    {
        return (value == null);
    }


    /** imports body into an Object applying importCodec if doesImport() returns true, otherwise returns value itself. */

    protected Object importBody(Object value) throws Exception
    {
        if (doesImport(value))
        {
            String bodyString = bodyContent.getString();

            if (!isEmptyString(bodyString))
            {
                String importCodec = _importCodec;

                if (importCodec != null)
                {
                    return applyCodec(importCodec, bodyString);
                }
                else
                {
                    return bodyString;
                }
            }
        }

        return value;
    }

    /** prepares value, called upon tag opening */

    protected void doPrepare() throws Exception
    {
        _value = prepareValue(_value);
    }

    /** apply default logic */

    protected Object doDefault(Object value, Object defaultValue, String test) throws Exception
    {
        if (test == null)
        {
            if (isEmptyString(value))
            {
                value = defaultValue;
            }
        }
        else
        {
            setPageAttribute(VALUEVAR, value);

            if (!TypeUtil.isTrue(EL.applyCodec(pageContext, test, value)))
            {
                value = defaultValue;
            }
        }

        return value;
    }

    /** does 'import' stage and returns 'importedValue'. */

    protected Object doImport(Object tagValue) throws Exception
    {
        Object importedValue = tagValue;

        if (_hasBody)
        {
            importedValue = importBody(tagValue);
        }

        return importedValue;
    }

    /** If true, then exports the value   Default implementation returns true if isAssigned is false (i.e., "export when not assigned") */
   
    protected boolean doesExport(Object tagValue, boolean isAssigned)
    {
        return (("always".equals(_doesExport)) || (!isAssigned));
    }
    
    /** exports given value */

    protected void exportValue(Object tagValue) throws Exception
    {
        if (tagValue instanceof Node)
        {
            Node node = (Node)tagValue;

            Transformer transformer = TransformerFactory.newInstance().newTransformer();
            transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");

            StringWriter writer = new StringWriter();

            Source source = new DOMSource(node);
            Result result = new StreamResult(writer);
            
            transformer.transform(source, result);
        
            pageContext.getOut().print(writer.toString());
        }
        else if (tagValue instanceof Reader)
        {
            IOUtil.copy((Reader)tagValue, pageContext.getOut());
        }
        else
        {
            pageContext.getOut().print(tagValue);
        }
    }

    /** returns value to be exported */

    protected Object getExportValue(Object tagValue) throws Exception
    {
        String export = _export;

        if ((export == null) || (export.length() == 0))
        {
            return null;
        }

        if ("_assign".equals(export))
        {
            export = _assign;
        }

        if (export != DEFAULT)
        {
            tagValue = evaluateExpression("export", export, Object.class);
        }

        if (tagValue != null)
        {
            String exportCodec = _exportCodec;

            if (exportCodec != null)
            {
                tagValue = applyCodec(exportCodec, tagValue);
            }
        }

        return tagValue;
    }

    /** does 'Export' stage */

    protected void doExport(Object tagValue) throws Exception
    {
        setPageAttribute(VALUEVAR, tagValue);

        Object exportObject = getExportValue(tagValue);

        if (exportObject != null)
        {
            exportValue(exportObject);
        }
    }

    /** returns the closest ancestor tag of given class */

    protected Object getAncestorTag(String className)
    {
        try
        {
            Class c = Class.forName(className);

            Tag parent = this;

            while ((parent = parent.getParent()) != null)
            {
                if (c.isInstance(parent))
                {
                    return parent;
                }
            }
        }
        catch (Exception e)
        {
        }
            
        return null;
    }

    protected boolean setTargetProperty(Object targetObject, String targetProperty, Object assignValue) throws Exception
    {
        BeanUtil.setProperty(targetObject, targetProperty, assignValue);
        return true;
    }

    /** does 'Assign' stage */

    protected boolean doAssign(Object assignValue) throws Exception
    {
        setPageAttribute(VALUEVAR, assignValue);

        String assign = _assign;

        if ("_export".equals(assign))
        {
            assign = _export;
        }

        if (assign != DEFAULT)
        {
            assignValue = evaluateExpression("assign", assign, Object.class);
        }

        String assignCodec = _assignCodec;

        if (assignCodec != null)
        {
            assignValue = applyCodec(assignCodec, assignValue);
        }

        boolean isAssigned = false;

        String attribute = _attribute;

        if (attribute != null)
        {
            isAssigned = setTargetProperty(getAncestorTag(CLASSNAME), attribute, assignValue);

            if ("value".equals(attribute))
            {
                _valueVarSaved = assignValue;
            }
        }

        Object targetObject = _target;
        String property = _property;

        if (RETURNTARGET.equals(targetObject))
        {
            NestedMap param = (NestedMap)pageContext.getAttribute(ParamTag.IMPLICITPARAMVAR);

            if ((param != null) && (param.get(ParamTag.CALLER) != null))
            {
                targetObject = param;
                property = "@_return.value";

                if (assignValue == null)
                {
                    assignValue = "";
                }
            }
            else
            {
                targetObject = null;
                property = null;
            }
        }

        if (targetObject instanceof String)
        {
            targetObject = evaluateExpression("target", EL.replaceEvalEscape(targetObject.toString()), Object.class);
        }

        String targetProperty = null;

        if (property != null)
        {
            targetProperty = property;

            if (targetObject == null)
            {
                targetObject = getAncestorTag(CLASSNAME);

                if (targetObject != null)
                {
                    targetObject = ((YuzuTag)targetObject)._value;
                }
                else
                {
                    targetObject = null;
                }
            }

            isAssigned = setTargetProperty(targetObject, targetProperty, assignValue);
        }

        String var = _var;
        int scope = _scope;

        if (var != null)
        {
            if (assignValue != null)
            {
                pageContext.setAttribute(var, assignValue, scope);
            }
            else
            {
                pageContext.removeAttribute(var, scope);
            }

            isAssigned = true;
        } 

        return isAssigned;
    }

    /** processes given value and returns 'processedValue' */

    protected Object processValue(Object tagValue) throws Exception
    {
        return tagValue;
    }

    /* does 'Process' stage */
    
    protected Object doProcess(Object processValue) throws Exception
    {
        Object object = processValue(processValue);
        
        String processCodec = _processCodec;
        
        if (_processCodec != null)
        {
            object = applyCodec(processCodec, object);
        }

        return object;
    }

    public void doCatch(Throwable t) throws Throwable
    {
        // pageContext.popBody();

        throw t;
    }

    public void doFinally()
    {
        cleanup();

        restoreLocal();

        init();
    }

    /** Returns true if given object is 'empty' (null, "", empty Collection, or empty array */

    public static boolean isEmpty(Object object)
    {
        return TypeUtil.isEmpty(object);
    }

    /** Returns true if given object is an empty string (null or "") */

    public static boolean isEmptyString(Object object)
    {
        return TypeUtil.isEmptyString(object);
    }

    /** Attribute evaluator to be called inside attribute setters.
        To allow nested attribute setters, returns given expression as-is
        if called inside the body (when _isInBody is true) */

    protected Object evaluateAttribute(String name, Object expression, Class valueClass) throws Exception
    {
        if (_isInBody)
        {
            return expression;
        }
        else
        {
            return evaluateAttributeExpression(name, expression, valueClass);
        }
    }

    public void setVar(Object expression) throws Exception
    {
        _var = (String)evaluateAttribute("var", expression, String.class);
    }

    public void setScope(Object expression) throws Exception
    {
        _scope = Util.getScope((String)evaluateAttribute("scope", expression, String.class));
    }

    public void setTarget(Object expression) throws Exception
    {
        _target = evaluateAttribute("target", expression, Object.class);
    }

    public void setProperty(Object expression) throws Exception
    {
        _property = (String)evaluateAttribute("property", expression, String.class);
    }

    public void setValue(Object expression) throws Exception
    {
        _value = evaluateAttribute("value", expression, Object.class);
    }

    public Object getValue()
    {
        return _value;
    }

    public void setDefault(Object expression) throws Exception
    {
        _defaultValue = evaluateAttribute("default", expression, Object.class);
    }

    public void setDefaultValue(Object expression) throws Exception
    {
        setDefault(expression);
    }

    public void setClassName(Object expression) throws Exception
    {
        _className = (String)evaluateAttribute("className", expression, String.class);
    }

    public void setTest(Object expression) throws Exception
    {
        _test = (String)evaluateAttribute("test", expression, EL.class);
    }

    public void setAttribute(Object expression) throws Exception
    {
        _attribute = (String)evaluateAttribute("attribute", expression, String.class);
    }

    public void setLocal(Object expression) throws Exception
    {
        _local = (String)evaluateAttribute("local", expression, String.class);
    }

    public void setExport(Object expression) throws Exception
    {
        _export = (String)evaluateAttribute("export", expression, EL.class);
    }

    public void setAssign(Object expression) throws Exception
    {
        _assign = (String)evaluateAttribute("assign", expression, EL.class);
    }

    public void setExportCodec(Object expression) throws Exception
    {
        _exportCodec = (String)evaluateAttribute("exportCodec", expression, EL.class);
    }

    public void setAssignCodec(Object expression) throws Exception
    {
        _assignCodec = (String)evaluateAttribute("assignCodec", expression, EL.class);
    }

    public void setImportCodec(Object expression) throws Exception
    {
        _importCodec = (String)evaluateAttribute("importCodec", expression, EL.class);
    }

    public void setProcessCodec(Object expression) throws Exception
    {
        _processCodec = (String)evaluateAttribute("processCodec", expression, EL.class);
    }

    public void setCodec(Object expression) throws Exception
    {
        setProcessCodec(expression);
    }

    public void setPrepareCodec(Object expression) throws Exception
    {
        _prepareCodec = (String)evaluateAttribute("prepareCodec", expression, EL.class);
    }

    public void setCleanupCodec(Object expression) throws Exception
    {
        _cleanupCodec = (String)evaluateAttribute("cleanupCodec", expression, EL.class);
    }

    public void setDoesExport(Object expression) throws Exception
    {
        _doesExport = (String)evaluateAttribute("doesExport", expression, String.class);
    }
}
