/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/

package com.micronova.jsp.tag;

import com.micronova.util.*;
import java.util.*;
import javax.servlet.jsp.*;
import java.lang.reflect.*;
import java.util.regex.*;

/**
   EL evaluation utilities.  Also used to indicate that translation from "#{" to "${" should be made for a tag attribute.
*/

public class EL extends ELRoot implements ObjectSource, ObjectTarget
{
    public static final String DEFAULT = "default";

    /** precompiled patterns */

    public static final String DEFAULTPATTERNEVAL = "[@]\\{([^}]*)\\}";
    public static final String DEFAULTPATTERNQUERY = "[%]\\{([^\\}]*)\\}";
    public static final String DEFAULTPATTERNNAME = "__";

    public static final Pattern defaultPatternEval = Pattern.compile(DEFAULTPATTERNEVAL);
    public static final Pattern defaultPatternQuery = Pattern.compile(DEFAULTPATTERNQUERY);

    public static final Pattern defaultPatternName = Pattern.compile(DEFAULTPATTERNNAME);

    public static final Pattern getPattern(Object patternSpec)
    {
        if (DEFAULTPATTERNEVAL.equals(patternSpec))
        {
            return defaultPatternEval;
        }
        else if (DEFAULTPATTERNQUERY.equals(patternSpec))
        {
            return defaultPatternQuery;
        }
        else if (patternSpec instanceof Pattern)
        {
            return (Pattern)patternSpec;
        }
        else if (patternSpec != null)
        {
            String patternString = (String)patternSpec;

            if (patternString.indexOf('(') == -1)
            {
                patternString = patternString + "\\{([^}]*)\\}";
            }

            return Pattern.compile(patternString);
        }
        else
        {
            return null;
        }
    }

    protected PageContext _pageContext;
    protected String _codec;
    protected String _nameCodec;
    protected String _valueCodec;

    /** built-in codec class prefix */

    public static final String CODECPATH = "com.micronova.util.codec.Codec";

    /** page-scoped variable name for the object a codec is applied on (for codec) */

    public static final String OPERANDVAR = "_operand";

    /** sets a page attribute named 'name' to 'value'.  When 'value' is null, then the attribute is removed. */

    public static void setPageAttribute(PageContext pageContext, String name, Object value)
    {
        if (name != null)
        {
            if (value == null)
            {
                pageContext.removeAttribute(name);
            }
            else
            {
                pageContext.setAttribute(name, value);
            }
        }
    }

    /** gets a page attribute named 'name' */

    public static Object getPageAttribute(PageContext pageContext, String name)
    {
        return pageContext.getAttribute(name);
    }

    /** replace "@{" with "${" in a StringBuffer */

    public final static StringBuffer replaceEvalEscape(StringBuffer buffer)
    {
        boolean isBeforeBrace = false;
        
        for (int i = buffer.length(); --i >= 0;)
        {
            char c = buffer.charAt(i);
            
            if (isBeforeBrace)
            {
                if (c == '@')
                {
                    buffer.setCharAt(i, '$');
                }
                
                isBeforeBrace = false;
            }
            else
            {
                if (c == '{')
                {
                    isBeforeBrace = true;
                }
            }
        }

        return buffer;
    }

    /** String version of replaceEvalEscape() */

    public final static String replaceEvalEscape(String expression)
    {
        if (expression != null)
        {
            expression = replaceEvalEscape(new StringBuffer(expression)).toString();
        }

        return expression;
    }

    /** applies codecs to an Object */

    public static Object applyCodec(PageContext pageContext, String codecs, Object object) throws Exception
    {
        if ((codecs != null) && (codecs.length() > 0))
        {
            List codecList = StringUtil.split(codecs, '|');

            for (int i = 0; i < codecList.size(); i ++)
            {
                String codec = (String)codecList.get(i);

                setPageAttribute(pageContext, OPERANDVAR, object);

                List partList = StringUtil.split(codec, ':');

                int partCount = partList.size();

                if (partCount < 2)
                {
                    object = evaluateExpression(pageContext, "codecExpression", codec.replaceAll("\\\\\\{", "{"), Object.class);
                }
                else
                {
                    String codecName = (String)partList.get(0);

                    List implicitArg = StringUtil.split((String)partList.get(1), ';');

                    boolean doesUseObject = true;

                    if (codecName.startsWith("_"))
                    {
                        codecName = codecName.substring(1);
                        implicitArg.add(1, pageContext);
                    }

                    if (codecName.indexOf('.') < 0)
                    {
                        StringBuffer buffer = new StringBuffer();

                        buffer.append(CODECPATH);
                        buffer.append(codecName);

                        codecName = buffer.toString();
                    }

                    String codecMethod = (String)implicitArg.get(0);

                    if (codecMethod.endsWith("_"))
                    {
                        codecMethod = codecMethod.substring(0, codecMethod.length() - 1);
                        doesUseObject = false;
                    }

                    int implicitArgCount = implicitArg.size() - 1;
                    int numArgs = implicitArgCount + partCount - 1;

                    if (!doesUseObject)
                    {
                        numArgs --;
                    }

                    Class[] types = new Class[numArgs];
                    Object[] args = new Object[numArgs];

                    int j = 0;

                    for (int k = 1; j < implicitArgCount; j ++, k ++)
                    {
                        types[j] = Object.class;

                        Object arg = implicitArg.get(k);

                        if (arg instanceof String)
                        {
                            arg = evaluateExpression(pageContext, "codecArgument", (String)arg, Object.class);
                        }

                        args[j] = arg;
                    }

                    if (doesUseObject)
                    {
                        types[j] = Object.class;
                        args[j] = object;
                        j ++;
                    }

                    for (int k = 2; j < numArgs; j ++, k ++)
                    {
                        types[j] = Object.class;
                        args[j] = evaluateExpression(pageContext, "codecArgument", (String)partList.get(k), Object.class);
                    }

                    Class c = Class.forName(codecName);
                    Method m = c.getDeclaredMethod(codecMethod, types);
                    object = m.invoke(null, args);
                }
            }
        }
            
        return object;
    }

    /** to be used as ObjectSource */

    public EL(PageContext pageContext, String codec)
    {
        _pageContext = pageContext;
        _codec = codec;
    }

    /** to be used as ObjectTarget */

    public EL(PageContext pageContext, String nameCodec, String valueCodec)
    {
        _pageContext = pageContext;
        _nameCodec = nameCodec;
        _valueCodec = valueCodec;
    }

    /** ObjectSource implementation */

    public Object getObject(Object client, Object key)
    {
        try
        {
            PageContext pageContext = _pageContext;

            return applyCodec(pageContext, _codec, evaluateExpression(pageContext, "eval", "${" + key.toString() + "}", Object.class));
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
    }

    /** ObjectTarget implementation (key/value filter for Map client) */

    public Object putObject(Object client, Object key, Object value)
    {
        try
        {
            String nameCodec = _nameCodec;
            String valueCodec = _valueCodec;

            PageContext pageContext = _pageContext;

            if (nameCodec == DEFAULT)
            {
                key = StringUtil.applyPattern(key.toString(), defaultPatternName, null);
            }
            else if (!TypeUtil.isEmptyString(nameCodec))
            {
                key = (String)(applyCodec(pageContext, valueCodec, key.toString()));
            }
            
            if (value instanceof String)
            {
                if (valueCodec == DEFAULT)
                {
                    value = XMLUtil.encode(value.toString());
                }
                else if (!TypeUtil.isEmptyString(valueCodec))
                {
                    value = (String)applyCodec(pageContext, valueCodec, value.toString());
                }
            }
            
            return ((Map)client).put(key, value);
        }
        catch (Exception e)
        {
            throw new RuntimeException(e);
        }
    }

    /** EL filter */

    /** applies filter */

    public static final String INDEXVAR = "_index";
    public static final String ELEMENTVAR = "_element";
    public static final String LENGTHVAR = "_length";
    public static final String INCLUDE = "include";
    public static final String APPLYCODEC = "applyCodec";
    public static final String APPLY = "apply";
    public static final String BREAK = "break";

    public static Object applyFilter(PageContext pageContext, Object object, String include, String breakExpression, String apply, String applyCodec) throws Exception
    {
        if (object != null)
        {
            Object xVar = EL.getPageAttribute(pageContext, ELEMENTVAR);
            Object iVar = EL.getPageAttribute(pageContext, INDEXVAR);
            Object lVar = EL.getPageAttribute(pageContext, LENGTHVAR);
            
            try
            {
                if (object instanceof String)
                {
                    String string = (String)object;
                    StringBuffer buffer = new StringBuffer();
                    
                    int length = string.length();

                    EL.setPageAttribute(pageContext, LENGTHVAR, new Integer(length));
                    
                    for (int i = 0; i < length; i ++)
                    {
                        char x = string.charAt(i);
                        
                        EL.setPageAttribute(pageContext, ELEMENTVAR, "" + x);
                        
                        EL.setPageAttribute(pageContext, INDEXVAR, new Integer(i));
                        
                        Boolean b = (Boolean)EL.evaluateExpression(pageContext, "include", include, Boolean.class);
                        
                        if (b.booleanValue())
                        {
                            if (apply == null)
                            {
                                if (applyCodec == null)
                                {
                                    buffer.append(x);
                                }
                                else
                                {
                                    buffer.append(EL.applyCodec(pageContext, applyCodec, new Character(x)));
                                }
                            }
                            else
                            {
                                Object applyObject = EL.evaluateExpression(pageContext, APPLY, apply, Object.class);
                                
                                if (applyCodec != null)
                                {
                                    applyObject = EL.applyCodec(pageContext, applyCodec, applyObject);
                                }
                                
                                buffer.append(applyObject);
                            }
                        }
                        
                        if (breakExpression != null)
                        {
                            Boolean bBreak = (Boolean)EL.evaluateExpression(pageContext, BREAK, breakExpression, Boolean.class);
                            
                            if (bBreak.booleanValue())
                            {
                                break;
                            }
                        }
                    }
                    
                    object = buffer.toString();
                }
                else if (object instanceof Map)
                {
                    Map map = (Map)object;
                    Map newMap = new NestedMap();
                    
                    Iterator iterator = map.entrySet().iterator();
                    
                    int length = map.size();
                    
                    EL.setPageAttribute(pageContext, LENGTHVAR, new Integer(length));
                    
                    for (int i = 0; iterator.hasNext(); i ++)
                    {
                        Map.Entry x = (Map.Entry)iterator.next();
                        
                        EL.setPageAttribute(pageContext, ELEMENTVAR, x);
                        
                        EL.setPageAttribute(pageContext, INDEXVAR, new Integer(i));
                        
                        Boolean b = (Boolean)EL.evaluateExpression(pageContext, INCLUDE, include, Boolean.class);
                        
                        if (b.booleanValue())
                        {
                            Object key = x.getKey();
                            Object keyValue = x.getValue();
                            
                            if (apply != null)
                            {
                                keyValue = EL.evaluateExpression(pageContext, APPLY, apply, Object.class);
                            }
                            
                            if (applyCodec != null)
                            {
                                keyValue = EL.applyCodec(pageContext, applyCodec, keyValue);
                            }
                            
                            newMap.put(key, keyValue);
                        }
                        
                        if (breakExpression != null)
                        {
                            Boolean bBreak = (Boolean)EL.evaluateExpression(pageContext, BREAK, breakExpression, Boolean.class);
                            
                            if (bBreak.booleanValue())
                            {
                                break;
                            }
                        }
                    }
                    
                    object = newMap;
                }
                else 
                {
                    List list = TypeUtil.isList(object);

                    if (list != null)
                    {
                        List newList = new SparseList();
                        int length = list.size();
                    
                        EL.setPageAttribute(pageContext, LENGTHVAR, new Integer(length));
                    
                        for (int i = 0; i < length; i ++)
                        {
                            Object x = list.get(i);
                            
                            EL.setPageAttribute(pageContext, ELEMENTVAR, x);
                            
                            EL.setPageAttribute(pageContext, INDEXVAR, new Integer(i));
                            
                            Boolean b = (Boolean)EL.evaluateExpression(pageContext, INCLUDE, include, Boolean.class);
                            
                            if (b.booleanValue())
                            {
                                if (apply != null)
                                {
                                    x = EL.evaluateExpression(pageContext, APPLY, apply, Object.class);
                                }
                                
                                if (applyCodec != null)
                                {
                                    x = EL.applyCodec(pageContext, applyCodec, x);
                                }
                                
                                newList.add(x);
                            }
                            
                            if (breakExpression != null)
                            {
                                Boolean bBreak = (Boolean)EL.evaluateExpression(pageContext, BREAK, breakExpression, Boolean.class);
                                
                                if (bBreak.booleanValue())
                                {
                                    break;
                                }
                            }
                        }
                        
                        object = newList;
                    }
                    else
                    {
                        throw new Exception("unsupported filter object type:" + object.getClass().getName());
                    }
                }
            }
            finally
            {
                EL.setPageAttribute(pageContext, ELEMENTVAR, xVar);
                EL.setPageAttribute(pageContext, INDEXVAR, iVar);
                EL.setPageAttribute(pageContext, LENGTHVAR, lVar);
            }
        }
        
        return object;
    }
}
