/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util.codec;

import java.util.*;
import java.util.regex.*;
import com.micronova.util.*;

/** System codecs */

public class CodecSystem extends Codec
{
    private static int _uniqueId = 0;

    /** sleeps for given number of milliseconds */

    public static Object sleep(Object object) throws Exception
    {
        return CodecThread.sleep(object);
    }

    /** yields to other thread */

    public static Object yield()
    {
        return CodecThread.yield();
    }

    /** invokes a java method */

    public static Object invoke(Object mapObject) throws Exception
    {
        NestedMap map = new NestedMap(mapObject);

        return TypeUtil.invoke(map);
    }

    /** invokes a java method on object */

    public static Object invoke(Object object, Object mapObject) throws Exception
    {
        NestedMap map = new NestedMap(mapObject);

        return TypeUtil.invoke(object, map);
    }    

    /** obtain an unique ID */

    public static synchronized Object uniqueId() throws Exception
    {
        return new Integer(_uniqueId ++);
    }

    /** obtain some runtime info */

    public static Object runtime(Object object)
    {
        NestedMap map = TypeUtil.isNestedMap(object);

        Runtime runtime = Runtime.getRuntime();

        if (map.get("gc") != null)
        {
            runtime.gc();
        }

        if (map.get("runFinalization") != null)
        {
            runtime.runFinalization();
        }

        Integer haltValue = TypeUtil.isInteger(map.get("halt"));

        if (haltValue != null)
        {
            runtime.halt(haltValue.intValue());
        }

        Integer exitValue = TypeUtil.isInteger(map.get("exit"));

        if (exitValue != null)
        {
            runtime.exit(exitValue.intValue());
        }

        Boolean traceInstructionsValue = TypeUtil.isBoolean(map.get("traceInstruactions"));

        if (traceInstructionsValue != null)
        {
            runtime.traceInstructions(traceInstructionsValue.booleanValue());
        }

        Boolean traceMethodCallsValue = TypeUtil.isBoolean(map.get("traceMethodCalls"));

        if (traceMethodCallsValue != null)
        {
            runtime.traceMethodCalls(traceMethodCallsValue.booleanValue());
        }

        map.put("freeMemory", new Long(runtime.freeMemory()));
        map.put("maxMemory", new Long(runtime.maxMemory()));
        map.put("totalMemory", new Long(runtime.totalMemory()));
        map.put("availableProcessors", new Long(runtime.availableProcessors()));
        
        return map;
    }

    public static Object runtime() throws Exception
    {
        return runtime(null);
    }

    /** set system property **/

    public static Object setProperty(Object key, Object value) throws Exception
    {
        return System.setProperty(key.toString(), value.toString());
    }

    /** set system properties - map version **/

    public static Object setProperties(Object object) throws Exception
    {
        if (object != null)
        {
            NestedMap map = TypeUtil.isNestedMap(object);

            Iterator iterator = map.exportParams().entrySet().iterator();

            while (iterator.hasNext())
            {
                Map.Entry entry = (Map.Entry)iterator.next();

                System.setProperty(entry.getKey().toString(), entry.getValue().toString());
            }
        }
        
        return object;
    }        

    /** get system property **/

    public static Object getProperty(Object key) throws Exception
    {
        return System.getProperty(key.toString());
    }

    /** get system properties - map version **/

    public static Object getProperties() throws Exception
    {
        return TypeUtil.isNestedMap(System.getProperties()).exportParams();
    }        
    
    /** current system time in milliseconds */

    public static Object currentTimeMillis() throws Exception
    {
        return new Long(System.currentTimeMillis());
    }

    /** current time as date */

    public static Object currentTime() throws Exception
    {
        return new Date(System.currentTimeMillis());
    }

    /** creates a cache */

    public static Object createCache(Object spec) throws Exception
    {
        NestedMap specMap = new NestedMap(spec);

        int initialCapacity = specMap.getInt("initialCapacity", 16);
        float loadFactor = specMap.getFloat("loadFactor", 0.75f);
        String type = specMap.getString("type", "LRU");
        int maxSize = specMap.getInt("maxSize", 16);
        long lifespan = specMap.getLong("lifespan", -1L);

        if (lifespan < 0)
        {
            return new Cache(initialCapacity, loadFactor, (type.equals("LRU")), maxSize);
        }
        else
        {
            return new TimedCache(initialCapacity, loadFactor, (type.equals("LRU")), maxSize, lifespan);
        }
    }
}
