/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.io.*;
import java.text.*;
import java.util.regex.*;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;
import javax.xml.transform.sax.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import java.net.*;

/** XML utilities */

public class XMLUtil
{
    /** tag encoding/decoding maps */

    private static final Pattern patternEncode;
    private static final Map replacementMapEncode;

    private static final Pattern patternDecode;
    private static final Map replacementMapDecode;

    static
    {
        patternDecode = Pattern.compile("(&amp;|&quot;|&lt;|&gt;|&apos;|&#034;|&#039;|&#123;|&#125;)");

        replacementMapDecode = new HashMap();

        replacementMapDecode.put("&amp;", "&");
        replacementMapDecode.put("&quot;", "\"");
        replacementMapDecode.put("&#034;", "\"");
        replacementMapDecode.put("&apos;", "'");
        replacementMapDecode.put("&#039;", "'");
        replacementMapDecode.put("&lt;", "<");
        replacementMapDecode.put("&gt;", ">");
        replacementMapDecode.put("&#123;", "{");
        replacementMapDecode.put("&#125;", "}");

        patternEncode = Pattern.compile("[&\"'<>{}]");

        replacementMapEncode = new HashMap();

        replacementMapEncode.put("&", "&amp;");
        replacementMapEncode.put("\"", "&quot;");
        replacementMapEncode.put("'", "&#039;");
        replacementMapEncode.put("<", "&lt;");
        replacementMapEncode.put(">", "&gt;");
        replacementMapEncode.put("{", "&#123;");
        replacementMapEncode.put("}", "&#125;");
    }

    /** decode tag escapes */

    public static String decode(String string)
    {
        return StringUtil.applyPattern(string, patternDecode, replacementMapDecode);
    }

    /** encode tag characters */

    public static String encode(String string)
    {
        return encode(string, null);
    }

    /** encode tag characters, with specified pattern */

    public static String encode(String string, Pattern pattern)
    {
        if (pattern == null)
        {
            pattern = patternEncode;
        }

        return StringUtil.applyPattern(string, pattern, replacementMapEncode);
    }

    /** create a Source from an Object */

    public static final Source createSource(Object object) throws Exception
    {
        Source source = null;

        if (object instanceof Node)
        {
            source = new DOMSource((Node)object);
        }
        else if (object instanceof String)
        {
            source = new StreamSource(new StringReader(object.toString()));
        }
        else if (object instanceof InputSource)
        {
            source = new SAXSource((InputSource)object);
        }
        else if (object instanceof InputStream)
        {
            source = new StreamSource((InputStream)object);
        }
        else if (object instanceof Reader)
        {
            source = new StreamSource((Reader)object);
        }
        else if (object instanceof URL)
        {
            source = new StreamSource(((URL)object).toString());
        }
        
        return source;
    }

    /** outputs Source */

    public static final Object output(Source source, NestedMap controlMap) throws Exception
    {
        if (source != null)
        {
            Transformer transformer = TransformerFactory.newInstance().newTransformer();
        
            if (controlMap != null)
            {
                Iterator iterator = controlMap.entrySet().iterator();

                while (iterator.hasNext())
                {
                    Map.Entry entry = (Map.Entry)iterator.next();

                    transformer.setOutputProperty(entry.getKey().toString(), entry.getValue().toString());
                }
            }
            
            StringWriter writer = new StringWriter();
            Result result = new StreamResult(writer);
            
            transformer.transform(source, result);

            return writer.toString();
        }
        else
        {
            return null;
        }
    }

    /** parses document according to control map */

    public static final Document parse(Object object, NestedMap controlMap) throws Exception
    {
        Document document = null;

        if (object != null)
        {
            DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();

            if (controlMap != null)
            {
                BeanUtil.fill(documentBuilderFactory, controlMap);
            }
            else
            {
                documentBuilderFactory.setNamespaceAware(true);
                documentBuilderFactory.setIgnoringComments(true);
                documentBuilderFactory.setIgnoringElementContentWhitespace(true);
                documentBuilderFactory.setExpandEntityReferences(true);
                documentBuilderFactory.setCoalescing(true);
                documentBuilderFactory.setValidating(false);
            }

            DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
                
            Reader reader;

            if (object instanceof Reader)
            {
                reader = (Reader)object;
            }
            else
            {
                reader = new StringReader(object.toString());
            }

            document = documentBuilder.parse(new InputSource(reader));
        }

        return document;
    }

    private static Map getParseHtmlMap(Object object)
    {
        Map map = null;

        if (object != null)
        {
            if (object instanceof Map)
            {
                map = (Map)object;
            }
            else
            {
                List list = TypeUtil.isStringList(object, ',', '\\');

                if ((list != null) && (list.size() > 0))
                {
                    map = new HashMap();
                    
                    Iterator iterator = list.iterator();
                    
                    while (iterator.hasNext())
                    {
                        map.put(iterator.next().toString().toLowerCase(), Boolean.TRUE);
                    }
                }
            }
        }

        return map;
    }

    private static int getParseHtmlStrategy(String strategyName)
    {
        int strategy = com.micronova.util.cc.html.Parser.STRATEGY_SINGLE;

        if ("data".equals(strategyName))
        {
            strategy = com.micronova.util.cc.html.Parser.STRATEGY_DATA;
        }
        else if ("none".equals(strategyName))
        {
            strategy = com.micronova.util.cc.html.Parser.STRATEGY_NONE;
        }
        else if ("single".equals(strategyName))
        {
            strategy = com.micronova.util.cc.html.Parser.STRATEGY_SINGLE;
        }

        return strategy;
    }

    public static Document parseHtml(Object object, NestedMap controlMap) throws Exception
    {
        Document document = null;
        Reader reader = null;

        try
        {
            String prefix = controlMap.getString("prefix", "");
            int strategy = getParseHtmlStrategy(controlMap.getString("strategy"));
            String rootName = controlMap.getString("rootName", "root");
            Map exclude = getParseHtmlMap(controlMap.get("exclude"));
            Map include = getParseHtmlMap(controlMap.get("include"));

            if (object instanceof Reader)
            {
                reader = (Reader)object;
            }
            else
            {
                reader = new StringReader(object.toString());
            }

            com.micronova.util.cc.html.Parser parser = new com.micronova.util.cc.html.Parser(reader);
          
            document = parser.buildDocument(prefix, strategy, rootName, exclude, include);
        }
        catch (Exception e)
        {
            try
            {
                reader.close();
            }
            catch (Exception ee)
            {
            }

            throw e;
        }

        return document;
    }
}
