/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.io.*;
import java.util.zip.*;
import java.util.regex.*;

/** string utilities */

public class StringUtil
{
    /** default escape character */

    public static final char DEFAULTESCAPE = '\\';

    /** splits a StringBuffer into a List at separator with escape */

    public static final List split(StringBuffer buffer, char separator, char escape)
    {
        List list = new ArrayList();

        StringBuffer partBuffer = new StringBuffer();
        int bufferLength = buffer.length();

        boolean isEscaped = false;

        for (int i = 0; i < bufferLength; i ++)
        {
            char c = buffer.charAt(i);

            if (isEscaped)
            {
                if (c != separator)
                {
                    partBuffer.append(escape);
                }
                
                partBuffer.append(c);
                
                isEscaped = false;
            }
            else
            {
                if (c == escape)
                {
                    isEscaped = true;
                }
                else if (c == separator)
                {
                    list.add(partBuffer.toString());
                    partBuffer.setLength(0);
                }
                else
                {
                    partBuffer.append(c);
                }
            }
        }

        if (partBuffer.length() != 0)
        {
            list.add(partBuffer.toString());
        }

        return list;
    }

    /** splits a String at separator with escape */

    public static final List split(String string, char separator, char escape)
    {
        return split(new StringBuffer(string), separator, escape);
    }

    /** splits a StringBuffer at separator with default escape "\" */

    public static final List split(StringBuffer buffer, char separator)
    {
        return split(buffer, separator, DEFAULTESCAPE);
    }

    /** splits a String at separator with default escape "\" */

    public static final List split(String string, char separator)
    {
        return split(string, separator, DEFAULTESCAPE);
    }

    /** joins a List using glue */

    public static final String join(List list, String glue)
    {
        StringBuffer output = new StringBuffer();

        boolean needsGlue = false;

        Iterator iterator = list.iterator();

        while (iterator.hasNext())
        {
            if (needsGlue)
            {
                output.append(glue);
            }
            
            output.append(iterator.next());
            
            needsGlue = true;
        }

        return output.toString();
    }

    /** decodes Backslash-encoded ("\n", "\r", etc.) StringBuffer */

    public static final void decodeBackslash(StringBuffer input, StringBuffer output)
    {
        int inputLength = input.length();

        boolean isAfterSlash = false;

        for (int i = 0; i < inputLength; i ++)
        {
            char c = input.charAt(i);

            if (c == '\\')
            {
                if (!isAfterSlash)
                {
                    isAfterSlash = true;
                }
                else
                {
                    output.append(c);
                    isAfterSlash = false;
                }
            }
            else
            {
                if (isAfterSlash)
                {
                    switch (c)
                    {
                        case 'n':
                        {
                            output.append('\n');
                        }
                        break;

                        case 'r':
                        {
                            output.append('\r');
                        }
                        break;

                        case 't':
                        {
                            output.append('\t');
                        }
                        break;

                        case 'b':
                        {
                            output.append('\b');
                        }
                        break;
                    }

                    isAfterSlash = false;
                }
                else
                {
                    output.append(c);
                }                        
            }
        }
    }

    public static final String encodeBackslash(String string)
    {
        StringBuffer input = new StringBuffer(string);
        StringBuffer output = new StringBuffer();

        encodeBackslash(input, output);

        return output.toString();
    }

    /** encodes "\n" etc. using backslash */

    public static final void encodeBackslash(StringBuffer input, StringBuffer output)
    {
        int inputLength = input.length();

        for (int i = 0; i < inputLength; i ++)
        {
            char c = input.charAt(i);
            
            switch (c)
            {
                case '\n':
                {
                    output.append("\\n");
                }
                break;
                
                case '\r':
                {
                    output.append("\\r");
                }
                break;
                
                case '\t':
                {
                    output.append("\\t");
                }
                break;
                
                case '\b':
                {
                    output.append("\\b");
                }
                break;
                    
                case '\\':
                {
                    output.append("\\\\");
                }
                break;
                
                default:
                {
                    output.append(c);
                }
            }
        }
    }

    /** decodes backslash-encoded String */

    public static final String decodeBackslash(String string)
    {
        StringBuffer input = new StringBuffer(string);
        StringBuffer output = new StringBuffer();

        decodeBackslash(input, output);

        return output.toString();
    }

    /** compresses a byte array using GZIP */

    public static byte[] compressGZIP(byte[] input) throws Exception
    {
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
            
        GZIPOutputStream zOut = new GZIPOutputStream(bOut);

        zOut.write(input, 0, input.length);
        
        zOut.close();
        
        return bOut.toByteArray();
    }

    /** decompresses a byte array using GZIP */

    public static byte[] decompressGZIP(byte[] input) throws Exception
    {
        ByteArrayInputStream bIn = new ByteArrayInputStream(input);
            
        GZIPInputStream zIn = new GZIPInputStream(bIn);
        
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        
        int ch;

        while ((ch = zIn.read()) != -1)
        {
            bOut.write(ch);
        }
        
        zIn.close();

        bOut.close();
        
        return bOut.toByteArray();
    }

    /** compresses a byte array using Zip */

    public static byte[] compressZip(byte[] input) throws Exception
    {
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
            
        ZipOutputStream zOut = new ZipOutputStream(bOut);
        
        zOut.putNextEntry(new ZipEntry("ZIP"));

        zOut.write(input, 0, input.length);
        
        zOut.close();
        
        return bOut.toByteArray();
    }

    /** decompresses a byte array using Zip */

    public static byte[] decompressZip(byte[] input) throws Exception
    {
        ByteArrayInputStream bIn = new ByteArrayInputStream(input);
            
        ZipInputStream zIn = new ZipInputStream(bIn);

        zIn.getNextEntry();
        
        ByteArrayOutputStream bOut = new ByteArrayOutputStream();
        
        int ch;

        while ((ch = zIn.read()) != -1)
        {
            bOut.write(ch);
        }
        
        zIn.close();

        bOut.close();
        
        return bOut.toByteArray();
    }


    /** converts byte[] to String using "iso-8859-1" encoding */

    public static final String toBinaryString(byte[] b) throws Exception
    {
        return new String(b, "iso-8859-1");
    }

    /** converts String to byte[] using "iso-8859-1" encoding */

    public static final byte[] fromBinaryString(String s) throws Exception
    {
        return s.getBytes("iso-8859-1");
    }

    /** compresses a String into a BinaryString using GZIP */

    public static final String compressGZIP(String string) throws Exception
    {
        return toBinaryString(compressGZIP(fromBinaryString(string)));
    }

    /** decompresses a BinaryString into a String  using GZIP */

    public static final String decompressGZIP(String string) throws Exception
    {
        return toBinaryString(decompressGZIP(fromBinaryString(string)));
    }

    /** compresses a String into a BinaryString using Zip */

    public static final String compressZip(String string) throws Exception
    {
        return toBinaryString(compressZip(fromBinaryString(string)));
    }

    /** decompresses a BinaryString into a String using Zip */

    public static final String decompressZip(String string) throws Exception
    {
        return toBinaryString(decompressZip(fromBinaryString(string)));
    }

    /** encode String to base64 */

    public static final String encodeBase64(byte[] x) throws Exception
    {
        return (new sun.misc.BASE64Encoder()).encode(x);
    }

    /** decode base64-encoded String */

    public static final byte[] decodeBase64(String s) throws Exception
    {
        return (new sun.misc.BASE64Decoder()).decodeBuffer(s);
    }

    /** converts byte[] to hexadecimal string */

    public static final StringBuffer encodeHex(byte[] in, StringBuffer out)
    {
        for (int i = 0; i < in.length; i ++)
        {
            out.append(NumberUtil.toHexString((256 + in[i]) % 256, "00"));
        }

        return out;
    }

    /** encodes byte[] to String */

    public static final String encodeHex(byte[] in)
    {
        StringBuffer out = new StringBuffer();

        return encodeHex(in, out).toString();
    }

    /** converts hexadecimal string to byte[], assuming the size is known */

    public static final byte[] decodeHex(StringBuffer in, byte[] out) throws Exception
    {
        int j = 0;

        for (int i = 0; i < out.length; i ++)
        {
            out[i] = (byte)Integer.parseInt(in.substring(j, j + 2), 16);
            j += 2;
        }

        return out;
    }

    /** converts hexadecimal string to byte[] */

    public static final byte[] decodeHex(String s) throws Exception
    {
        int sl = s.length();

        if (sl % 2 == 0)
        {
            int bl = sl / 2;
        
            byte[] out = new byte[s.length() / 2];

            return decodeHex(new StringBuffer(s), out);
        }
        else
        {
            throw new Exception("string length must be even");
        }
    }

    /** converts String into Character list */

    public static final List toCharacterList(String s) throws Exception
    {
        return TypeUtil.isList(s.toString().toCharArray());
    }

    /** converts List of Characters to String */

    public static final String fromCharacterList(List list) throws Exception
    {
        int length = list.size();
            
        char[] array = new char[length];
        
        for (int i = length; --i >= 0;)
        {
            array[i] = ((Character)(list.get(i))).charValue();
        }
        
        return new String(array);
    }

    /** apply a regex pattern and replace.  If replaceMap is null, then matching patterns are removed.  If replacementMap is not null, then matching pattern is replaced with the value corresponding to the matching group in the replacementMap.  If the matching group corrensponds to null, then the group itself is used as replacement. */
    
    public static void applyPattern(CharSequence input, StringBuffer output, Pattern pattern, Map replacementMap)
    {
        Matcher matcher = pattern.matcher(input);

        while (matcher.find())
        {
            String group = matcher.group();

            if (replacementMap == null)
            {
                matcher.appendReplacement(output, "");
            }
            else
            {
                String replacement = (String)replacementMap.get(group);

                if (replacement == null)
                {
                    replacement = group;
                }

                matcher.appendReplacement(output, replacement);
            }
        }
        
        matcher.appendTail(output);
    }

    /** apply a regex pattern and replace */

    public static final String applyPattern(String string, Pattern pattern, Map replacementMap)
    {
        StringBuffer input = new StringBuffer(string);
        StringBuffer output = new StringBuffer();

        applyPattern(input, output, pattern, replacementMap);

        return output.toString();
    }

    /** Decompose string by pattern */

    public static List decompose(CharSequence input, Pattern pattern, List list, int groupIndex, boolean includeMatch, boolean includeNoMatch)
    {
        Matcher matcher = pattern.matcher(input);

        int position = 0;
        int end = 0;
        
        while (matcher.find())
        {
            int start = matcher.start();
            end = matcher.end();

            if (start >= position)
            {
                if (includeNoMatch)
                {
                    list.add(input.subSequence(position, start));
                }
            }

            if (includeMatch)
            {
                if (groupIndex >= 0)
                {
                    list.add(matcher.group(groupIndex));
                }
                else
                {
                    List matchingGroups = new SparseList();

                    for (int i = 0; i <= matcher.groupCount(); i ++)
                    {
                        matchingGroups.set(i, matcher.group(i));
                    }

                    list.add(matchingGroups);
                }
            }

            position = end;
        }

        if (includeNoMatch)
        {
            list.add(input.subSequence(end, input.length()));
        }

        return list;
    }

    /** decompose by pattern */
    
    public static void decompose(CharSequence input, Pattern pattern, List list, int groupIndex)
    {
        decompose(input, pattern, list, groupIndex, true, true);
    }

    /** decompose by pattern */

    public static List decompose(String string, Pattern pattern, int groupIndex)
    {
        StringBuffer input = new StringBuffer(string);
        List list = new ArrayList();

        decompose(input, pattern, list, groupIndex);

        return list;
    }

    /** decompose by pattern */

    public static List decompose(String string, Pattern pattern)
    {
        return decompose(string, pattern, 0);
    }

    /** split by pattern */
    
    public static void split(CharSequence input, Pattern pattern, List list, int groupIndex)
    {
        decompose(input, pattern, list, groupIndex, false, true);
    }

    /** split by pattern */

    public static List split(String string, Pattern pattern, int groupIndex)
    {
        StringBuffer input = new StringBuffer(string);
        List list = new ArrayList();

        split(input, pattern, list, groupIndex);

        return list;
    }

    /** decompose by pattern */

    public static List split(String string, Pattern pattern)
    {
        return split(string, pattern, 0);
    }

    /** match pattern group */
    
    public static void matchAll(CharSequence input, Pattern pattern, List list, int groupIndex)
    {
        decompose(input, pattern, list, groupIndex, true, false);
    }

    /** match pattern with group index */

    public static List matchAll(String string, Pattern pattern, int groupIndex)
    {
        StringBuffer input = new StringBuffer(string);
        List list = new ArrayList();

        matchAll(input, pattern, list, groupIndex);

        return list;
    }

    /** match pattern */

    public static List matchAll(String string, Pattern pattern)
    {
        return matchAll(string, pattern, 0);
    }

    /** get list of matching groups */

    public static List matchingGroups(String string, Pattern pattern)
    {
        List list = new SparseList();

        Matcher matcher = pattern.matcher(string);

        if (matcher.matches())
        {
            for (int i = 0; i <= matcher.groupCount(); i ++)
            {
                list.set(i, matcher.group(i));
            }
        }

        return list;
    }
    
    /** capitalize string */

    public static String capitalize(String string)
    {
        StringBuffer buffer = new StringBuffer(string);

        if (buffer.length() > 0)
        {
            char c = buffer.charAt(0);

            if (Character.isLowerCase(c))
            {
                buffer.setCharAt(0, Character.toUpperCase(c));
            }
            
            string = buffer.toString();
        }

        return string;
    }

    /** reverses string */

    public static String reverse(String string)
    {
        StringBuffer buffer = new StringBuffer(string);

        int bufferLength = buffer.length();
        int half = bufferLength / 2;

        for (int i = half, j = bufferLength - half; --i >=0; j ++)
        {
            char c = buffer.charAt(i);
            buffer.setCharAt(i, buffer.charAt(j));
            buffer.setCharAt(j, c);
        }

        return buffer.toString();
    }

    /** multplies (repeats) string given "count" times; returns "" if count is 0 */

    public static String multiply(String string, int count)
    {
        StringBuffer buffer = new StringBuffer();

        for (int i = count; --i >= 0;)
        {
            buffer.append(string);
        }

        return buffer.toString();
    }

    /** return the first matching group, or null */

    public static String match(CharSequence input, Pattern pattern, int groupIndex)
    {
        Matcher matcher = pattern.matcher(input);

        if (matcher.find())
        {
            return matcher.group(groupIndex);
        }
        else
        {
            return null;
        }
    }

    /** returns the first matching pattern */

    public static String match(String string, Pattern pattern)
    {
        return match(string, pattern, 0);
    }

    /** count words */

    public static Map countWords(String string, Pattern pattern)
    {
        String[] array = pattern.split(string);

        Map map = new HashMap();

        for (int i = array.length; --i >= 0;)
        {
            String word = array[i];

            Integer count = (Integer)map.get(word);

            if (count == null)
            {
                count = new Integer(1);
            }
            else
            {
                count = new Integer(count.intValue() + 1);
            }

            map.put(word, count);
        }

        return map;
    }

    /** loose substring */

    public static String substring(String string, int start, int end)
    {
        if (string != null)
        {
            int length = string.length();

            if (start < 0)
            {
                start = length + start;
            }

            if (end <= 0)
            {
                end = length + end;
            }

            if (start < 0)
            {
                start = 0;
            }

            if (end > length)
            {
                end = length;
            }

            if (end <= start)
            {
                string = "";
            }
            else
            {
                string = string.substring(start, end);
            }
        }
        
        return string;
    }

    /** loose substring */

    public static String substring(String string, int start)
    {
        return substring(string, start, 0);
    }

    /** character-by-character replacement (from[i] in string is replaced by to[i]) */

    public static String replaceCharacters(String string, char[] from, char[] to)
    {
        StringBuffer buffer = new StringBuffer(string);

        for (int i = buffer.length(); --i >= 0; )
        {
            char c = buffer.charAt(i);

            for (int j = from.length; --j >= 0; )
            {
                if (from[j] == c)
                {
                    buffer.setCharAt(i, to[j]);
                }
            }
        }

        return buffer.toString();
    }
}
