/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.util;

import java.util.*;
import java.io.*;
import java.beans.*;
import java.lang.reflect.*;

/** bean utilities */

public class BeanUtil
{
    public static String encode(Object object) throws Exception
    {
        if (object != null)
        {
            ByteArrayOutputStream bOut = new ByteArrayOutputStream();
                
            XMLEncoder encoder = new XMLEncoder(bOut);
                
            encoder.writeObject(object);
            
            encoder.close();

            return bOut.toString();
        }

        return null;
    }

    public static Object decode(String string) throws Exception
    {
        Object object = null;

        if (string != null)
        {
            ByteArrayInputStream bIn = new ByteArrayInputStream(string.getBytes("utf-8"));
            XMLDecoder decoder = new XMLDecoder(bIn);
                
            object = decoder.readObject();

            decoder.close();

            return object;
        }

        return null;
    }

    /** get a bean property (strictly without map/list support) */

    public static final Object getBeanProperty(Object object, String name) throws Exception
    {
        PropertyDescriptor pd[] = Introspector.getBeanInfo(object.getClass()).getPropertyDescriptors();
                    
        for (int i = pd.length; --i >= 0;)
        {
            String propertyName = pd[i].getName();
            
            if (name.equals(propertyName))
            {
                Method m = pd[i].getReadMethod();
                
                return m.invoke(object, (Object[])null);
            }
        }
        
        throw new Exception("bean property " + name + " not found");
    }

    /** get a bean property */

    public static final Object getProperty(Object object, String name) throws Exception
    {
        Class objectClass = object.getClass();

        if (object instanceof Map)
        {
            return ((Map)object).get(name);
        }
        else if (object instanceof List)
        {
            List list = (List)object;

            int index = Integer.parseInt(name);

            if (index < 0)
            {
                index = list.size() + index;
            }
            
            return list.get(index);
        }
        else if (objectClass.isArray())
        {
            int index = Integer.parseInt(name);

            if (index < 0)
            {
                index = Array.getLength(object) + index;
            }

            return Array.get(object, index);
        }

        return getBeanProperty(object, name);
    }

    /** sets a bean property (strictly without map/list/array support) */

    public static final Object setBeanProperty(Object object, String name, Object value) throws Exception
    {
        PropertyDescriptor pd[] = Introspector.getBeanInfo(object.getClass()).getPropertyDescriptors();
                    
        for (int i = pd.length; --i >= 0;)
        {
            String propertyName = pd[i].getName();
            
            if (name.equals(propertyName))
            {
                Method m = pd[i].getWriteMethod();
                
                Class[] parameterTypes = m.getParameterTypes();
                
                Object convertedValue = TypeUtil.isClass(value, parameterTypes[0]);
                
                return m.invoke(object, new Object[]{convertedValue});
            }
        }

        throw new Exception("bean property " + name + " not found");
    }


    /** sets a bean property */

    public static final Object setProperty(Object object, String name, Object value) throws Exception
    {
        Class objectClass = object.getClass();

        if (object instanceof Map)
        {
            return ((Map)object).put(name, value);
        }
        else if (object instanceof List)
        {
            List list = (List)object;

            Object returnObject = null;

            int listSize = list.size();

            int index = -1;

            boolean isInsertMode = false;
            boolean isIgnoreNullMode = false;

            if ("".equals(name))
            {
                name = "*";
            }

            if (name.startsWith("*"))
            {
                isInsertMode = true;
                name = name.substring(1);

                if (name.startsWith("*"))
                {
                    isIgnoreNullMode = true;
                    name = name.substring(1);
                }
            }

            if (name.length() == 0)
            {
                index = listSize;
            }
            else
            {
                index = Integer.parseInt(name);

                if (index < 0)
                {
                    index = listSize + index;
                }
            }

            if (isInsertMode)
            {
                if (value != null)
                {
                    if (index == listSize)
                    {
                        list.add(value);
                    }
                    else
                    {
                        list.add(index, value);
                    }
                }
                else if ((listSize > 0) && (!isIgnoreNullMode))
                {
                    if (index == listSize)
                    {
                        returnObject = list.remove(listSize - 1);
                    }
                    else
                    {
                        returnObject = list.remove(index);
                    }
                }
            }
            else
            {
                returnObject = list.set(index, value);
            }

            return returnObject;
        }
        else if (objectClass.isArray())
        {
            Class componentClass = objectClass.getComponentType();
                
            int index = Integer.parseInt(name);
            
            if (index < 0)
            {
                index = Array.getLength(object) + index;
            }
            
            Object currentValue = Array.get(object, index);

            Array.set(object, index, TypeUtil.isClass(value, componentClass));

            return currentValue;
        }

        return setBeanProperty(object, name, value);
    }

    /** fill Object with property map */

    public static Object fill(Object object, Map map) throws Exception
    {
        if (object != null)
        {
            PropertyDescriptor pd[] = Introspector.getBeanInfo(object.getClass()).getPropertyDescriptors();
                    
            Object[] arg = new Object[1];

            for (int i = 0; i < pd.length; i++) 
            {
                String name = pd[i].getName();

                Object propertyValue = map.get(name);

                if (propertyValue != null)
                {
                    Method m = pd[i].getWriteMethod();

                    Class[] parameterTypes = m.getParameterTypes();

                    Object convertedValue = TypeUtil.isClass(propertyValue, parameterTypes[0]);
                    arg[0] = convertedValue;

                    m.invoke(object, arg);
                }
            }
        }

        return object;
    }

    /** creates a map with all defined bean properties */

    public static NestedMap getMap(Object object) throws Exception
    {
        PropertyDescriptor pd[] = Introspector.getBeanInfo(object.getClass()).getPropertyDescriptors();

        NestedMap map = new NestedMap();
        
        for (int i = 0; i < pd.length; i++) 
        {
            String name = pd[i].getName();

            Object propertyValue = map.get(name);
            
            if (propertyValue == null)
            {
                Method m = pd[i].getReadMethod();

                if (m != null)
                {
                    propertyValue = m.invoke(object, (Object[])null);

                    map.put(name, propertyValue);
                }
            }
        }

        return map;
    }
}
