/* 

Copyright 2003-2006 MicroNova (R)
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

    * Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.

    * Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in the
    documentation and/or other materials provided with the distribution.

    * Neither the name of MicroNova nor the names of its contributors
    may be used to endorse or promote products derived from this
    software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.

*/


package com.micronova.jsp.tag;

import java.util.*;
import java.util.regex.*;
import java.io.*;
import javax.servlet.*;
import javax.servlet.http.*;
import javax.servlet.jsp.*;
import com.micronova.util.*;
import com.micronova.util.servlet.*;

public class ParamTag extends MapTag
{
    public static final String IMPLICITPARAMVAR = "__in__";
    public static final String DEFAULTENCODING = "com.micronova.jsp.tag.ParamTag.defaultEncoding";
    public static final String DEFAULTMAXCONTENTLENGTH = "com.micronova.jsp.tag.ParamTag.maxContentLength";
    public static final String DEFAULTSELECTSUFFIX = "com.micronova.jsp.tag.ParamTag.selectSuffix";
    public static final String DEFAULTRADIOSUFFIX = "com.micronova.jsp.tag.ParamTag.radioSuffix";

    public static final String REQUEST = "_request";
    public static final String CALLER = "_caller";
    public static final String MULTIPART = "multipart";

    public static final String MAXCONTENTLENGTH = "maxContentLength";
    public static final String MULTIPATTERN = "multiPattern";
    public static final String SELECTPATTERN = "selectPattern";
    public static final String RADIOPATTERN = "radioPattern";
    public static final String FILEPATTERN = "filePattern";
    public static final String VALUECODEC = "valueCodec";
    public static final String NAMECODEC = "nameCodec";
    public static final String ENCODING = "encoding";
    public static final String EMPTYPATTERN = "emptyPattern";

    public static final String DEFAULTEMPTYPATTERN = "--";

    public static final String DEFAULTFILEPATTERN = ".*[Ff]ile.*";

    protected Object _parameterMap;
    protected NestedMap _controlMap;

    /** default name codec pattern */

    protected static Pattern defaultNamePattern = Pattern.compile("__");

    protected void init()
    {
        super.init();

        _parameterMap = null;
        _controlMap = null;
    }

    protected NestedMap getControlMap()
    {
        NestedMap controlMap = _controlMap;

        if (controlMap == null)
        {
            controlMap = new NestedMap();
            _controlMap = controlMap;
        }

        return controlMap;
    }

    public void setControl(Object expression) throws Exception
    {
        NestedMap controlMap = getControlMap();

        controlMap.copyFromSource(evaluateAttribute("control", expression, Object.class));
    }

    public void setNameCodec(Object expression) throws Exception
    {
        getControlMap().put(NAMECODEC, evaluateAttribute(NAMECODEC, expression, String.class));

    }

    public void setValueCodec(Object expression) throws Exception
    {
        getControlMap().put(VALUECODEC, evaluateAttribute(VALUECODEC, expression, String.class));

    }

    public void setEncoding(Object expression) throws Exception
    {
        getControlMap().put(ENCODING, evaluateAttribute(ENCODING, expression, String.class));

    }

    public void setMaxContentLength(Object expression) throws Exception
    {
        getControlMap().put(MAXCONTENTLENGTH, evaluateAttribute(MAXCONTENTLENGTH, expression, String.class));

    }

    public void setMultiPattern(Object expression) throws Exception
    {
        getControlMap().put(MULTIPATTERN, evaluateAttribute(MULTIPATTERN, expression, String.class));

    }

    public void setSelectPattern(Object expression) throws Exception
    {
        getControlMap().put(SELECTPATTERN, evaluateAttribute(SELECTPATTERN, expression, String.class));

    }

    public void setRadioPattern(Object expression) throws Exception
    {
        getControlMap().put(RADIOPATTERN, evaluateAttribute(RADIOPATTERN, expression, String.class));

    }

    public void setFilePattern(Object expression) throws Exception
    {
        getControlMap().put(FILEPATTERN, evaluateAttribute(FILEPATTERN, expression, String.class));

    }

    public void setEmptyPattern(Object expression) throws Exception
    {
        getControlMap().put(EMPTYPATTERN, evaluateAttribute(EMPTYPATTERN, expression, String.class));
    }

    public void setParameterMap(Object expression) throws Exception
    {
        _parameterMap = evaluateAttribute("parameterMap", expression, Object.class);
    }

    /** apply nameCodec */

    protected String applyNameCodec(String nameCodec, String name) throws Exception
    {
        if (nameCodec == DEFAULT)
        {
            name = StringUtil.applyPattern(name, defaultNamePattern, null);
        }
        else if (!isEmptyString(nameCodec))
        {
            name= (String)(applyCodec(nameCodec, name));
        }

        return name;
    }

    /** apply valueCodec, if value is a String */

    protected Object applyValueCodec(String valueCodec, Object value) throws Exception
    {
        if (value instanceof String)
        {
            if (valueCodec == DEFAULT)
            {
                value = XMLUtil.encode(value.toString());
            }
            else if (!isEmptyString(valueCodec))
            {
                value = applyCodec(valueCodec, value);
            }
        }

        return value;
    }

    /** processes parameter map */

    protected void processParameterMap(ServletRequest request, NestedMap map, NestedMap controlMap, Map parameters) throws Exception
    {
        Pattern selectPattern = TypeUtil.isPattern(controlMap.get(SELECTPATTERN));
        Pattern multiPattern = TypeUtil.isPattern(controlMap.get(MULTIPATTERN));
        Pattern radioPattern = TypeUtil.isPattern(controlMap.get(RADIOPATTERN));
        Pattern emptyPattern = TypeUtil.isPattern(controlMap.get(EMPTYPATTERN, DEFAULTEMPTYPATTERN));

        String valueCodec = controlMap.getString(VALUECODEC, DEFAULT);
        String nameCodec = controlMap.getString(NAMECODEC, DEFAULT);

        String encoding = controlMap.getString(ENCODING);

        if ("*".equals(encoding))
        {
            String defaultEncoding = (String)getConfiguration(DEFAULTENCODING, "utf-8");

            String requestEncoding = pageContext.getRequest().getCharacterEncoding();
            if (requestEncoding != null)
            {
                encoding = requestEncoding;
            }
            else
            {
                encoding = defaultEncoding;
            }
        }
        
        String[] parameterArray = null;

        Iterator iterator = parameters.entrySet().iterator();
        
        while (iterator.hasNext())
        {
            Map.Entry entry = (Map.Entry)iterator.next();
            
            String parameterName = applyNameCodec(nameCodec, (String)(entry.getKey()));
            
            Object entryValue = entry.getValue();
            
            String[] parameterValues = null;
            
            if (entryValue instanceof String)
            {
                if (parameterArray == null)
                {
                    parameterArray = new String[1];
                }
                
                parameterArray[0] = (String)entryValue;
                
                parameterValues = parameterArray;
            }
            else if (entryValue instanceof String[])
            {
                parameterValues = (String[])entryValue;
            }
            else
            {
                map.setElement(parameterName, entryValue);
            }
            
            if (parameterValues != null)
            {
                for (int i = 0; i < parameterValues.length; i ++)
                {
                    Object parameterValue = parameterValues[i];

                    if (!isEmptyString(encoding))
                    {
                        parameterValue = new String(parameterValue.toString().getBytes("iso-8859-1"), encoding);
                        parameterName = new String(parameterName.getBytes("iso-8859-1"), encoding);
                    }
                    
                    parameterValue = applyValueCodec(valueCodec, parameterValue);

                    if (parameterValue != null)
                    {
                        if ((emptyPattern != null) && emptyPattern.matcher(parameterValue.toString()).matches())
                        {
                            parameterValue = "";
                        }
                    }
                   
                    if (isEmptyString(parameterValue))
                    {
                        map.setElement(parameterName, parameterValue);
                    }
                    else
                    {
                        boolean isMulti = ((multiPattern != null) && multiPattern.matcher(parameterName).matches());
                        
                        if ((selectPattern != null) && selectPattern.matcher(parameterName).matches())
                        {
                            String selectSuffix = (String)getConfiguration(DEFAULTSELECTSUFFIX, "_SELECTED");
                            String selectName = parameterName + selectSuffix;

                            parameterValue = applyNameCodec(nameCodec, parameterValue.toString());

                            map.setElement(parameterName, parameterValue);

                            if (!isMulti)
                            {
                                map.setElement(selectName, null);
                            }

                            map.setElement(selectName + "." + parameterValue, "SELECTED");
                        }
                        else if ((radioPattern != null) && radioPattern.matcher(parameterName).matches())
                        {
                            String radioSuffix = (String)getConfiguration(DEFAULTRADIOSUFFIX, "_CHECKED");
                            String radioName = parameterName + radioSuffix;

                            parameterValue = applyNameCodec(nameCodec, parameterValue.toString());

                            map.setElement(parameterName, parameterValue);

                            if (!isMulti)
                            {
                                map.setElement(radioName, null);
                            }

                            map.setElement(radioName + "." + parameterValue, "CHECKED");
                        }
                        else
                        {
                            if (isMulti)
                            {
                                map.setElement(parameterName + "._.*", parameterValue);
                            }
                            else
                            {
                                map.setElement(parameterName, parameterValue);
                            }
                        }
                    }
                }
            }
        }
    }

    protected Object processValue(Object tagValue) throws Exception
    {
        if (tagValue != null)
        {
            NestedMap map = (NestedMap)tagValue;
            HttpServletRequest request = (HttpServletRequest)pageContext.getRequest();
            NestedMap controlMap = getControlMap();

            Object parameterMap = _parameterMap;

            boolean doesRethrow = false;

            try
            {
                if ((request instanceof DispatchHttpServletRequest) && (parameterMap == null))
                {
                    DispatchHttpServletRequest dispatchHttpServletRequest = (DispatchHttpServletRequest)request;

                    map.put(CALLER, dispatchHttpServletRequest.getDispatchCaller());

                    Object dispatchObject = dispatchHttpServletRequest.getDispatchObject();

                    map.copyFromSource(dispatchObject);

                    pageContext.setAttribute(IMPLICITPARAMVAR, map);

                    if (dispatchObject instanceof NestedMap)
                    {
                        map.put(CallTag.RETURNPROPERTY, ((NestedMap)dispatchObject).get(CallTag.RETURNPROPERTY));
                    }
                }
                else
                {
                    Object maxContentLengthSpec = controlMap.get(MAXCONTENTLENGTH);

                    if (maxContentLengthSpec == null)
                    {
                        maxContentLengthSpec = getConfiguration(DEFAULTMAXCONTENTLENGTH, null);
                    }

                    if (maxContentLengthSpec != null)
                    {
                        int maxContentLength = Integer.parseInt(maxContentLengthSpec.toString());

                        boolean doesThrow = false;

                        if (maxContentLength < 0)
                        {
                            doesThrow = true;
                            maxContentLength = -maxContentLength;
                        }

                        if (maxContentLength > 0)
                        {
                            if (request.getContentLength() > maxContentLength)
                            {
                                doesRethrow = doesThrow;
                                throw new Exception("max content length exceeded");
                            }
                        }
                    }

                    processParameterMap(request, map, controlMap, (Map)map.get("__param"));

                    NestedMap multipartMap = null;
                    Map parameters = request.getParameterMap();

                    if (parameterMap != null)
                    {
                        processParameterMap(request, map, controlMap, (Map)((new NestedMap(parameterMap)).get("__param")));
                    }
                    else
                    {
                        processParameterMap(request, map, controlMap, parameters);
                    }

                    String contentType = request.getHeader("Content-Type");

                    if ((contentType != null) && (contentType.startsWith("multipart/form-data")))
                    {
                        String filePattern = controlMap.getString(FILEPATTERN, DEFAULTFILEPATTERN);
                        
                        multipartMap = MailFolder.decodeMultipart(request.getInputStream(), contentType, controlMap);
                        
                        NestedMap partNameMap = multipartMap.getSubMap(MailFolder.PARTNAME, false);

                        NestedMap multipartParameterMap = new NestedMap();

                        if (partNameMap != null)
                        {
                            NestedMap newPartNameMap = new NestedMap();
                        
                            String nameCodec = controlMap.getString(NAMECODEC, DEFAULT);

                            Iterator iterator = partNameMap.entrySet().iterator();
                            while (iterator.hasNext())
                            {
                                Map.Entry entry = (Map.Entry)iterator.next();
                                
                                String keyName = applyNameCodec(nameCodec, entry.getKey().toString());
                                
                                NestedMap valueMap = (NestedMap)entry.getValue();
                                
                                Object content = valueMap.get(MailFolder.CONTENT);
                                
                                if ((!isEmptyString(filePattern)) && (keyName.matches(filePattern)))
                                {
                                    if (content instanceof String)
                                    {
                                        content = new StringReader(content.toString());
                                    }
                                }

                                // multipartParameterMap.put(keyName, content);

                                multipartParameterMap.setElement(keyName, content);
                                newPartNameMap.setElement(keyName, valueMap);
                            }

                            multipartMap.put(MailFolder.PARTNAME, newPartNameMap);
                        }
                        
                        processParameterMap(request, map, controlMap, (Map)multipartParameterMap.get("__param"));
                    }

                    map.put(CALLER, null);
                    map.put(MULTIPART, multipartMap);
                }
            }
            catch (Exception e)
            {
                if (doesRethrow)
                {
                    throw e;
                }

                map.clear();
            }
            finally
            {
                map.put(REQUEST, request);
            }
        }

        return tagValue;
    }
}
