//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * XbVgX[^
 * @author Junpee
 */

#include <LampBasic.h>
#include "XTranslator/XMeshTranslator.h"
#include "XTranslator/XMaterialTranslator.h"
#include <Core/InputOutput/FilePath.h>
#include <Graphics/System/LampGraphics.h>
#include <Graphics/SceneNode/SceneNodeManager.h>
#include <Graphics/Model/ModelManager.h>
#include <Graphics/Mesh/MeshManager.h>
#include <Graphics/MeshData/MeshDataManager.h>

// V[m[hTtBbNX
const String XMeshTranslator::sceneNodeSuffix = "SceneNode";
// fTtBbNX
const String XMeshTranslator::modelSuffix = "Model";
// bVTtBbNX
const String XMeshTranslator::meshSuffix = "Mesh";
// bVf[^TtBbNX
const String XMeshTranslator::meshDataSuffix = "MeshData";

//------------------------------------------------------------------------------
// RXgN^
XMeshTranslator::XMeshTranslator() : xMesh_(NULL), xMaterialsBuffer_(NULL),
	xMaterialCount_(0), xAttributes_(NULL), xAttributeCount_(0),
	vertexSize_(0), positionOffset_(-1), normalOffset_(-1), uvOffset_(-1),
	scene_(NULL), name_(""), sceneNode_(NULL), model_(NULL){
}
//------------------------------------------------------------------------------
// fXgN^
XMeshTranslator::~XMeshTranslator(){
	SafeArrayDelete(xAttributes_);
	SafeRelease(xMaterialsBuffer_);
	SafeRelease(xMesh_);
}
//------------------------------------------------------------------------------
// Xt@C̕ϊ
bool XMeshTranslator::translateXFile(const FilePath& xFilePath, Scene* scene){
	if(!xFilePath.existFile()){
		ErrorOut("XFileConverter::initialize() Xt@C܂");
		return false;
	}
	name_ = xFilePath.getName();
	scene_ = scene;
	
	// Xt@C̃[h
	// VXeɃ[h
	u_int options = (D3DXMESH_SYSTEMMEM);
	// אڏƃGtFNg͎擾Ȃ
	DirectXCheck(D3DXLoadMeshFromX(xFilePath.getPath().getBytes(), options,
		LampGraphics::getDirect3DDevice(), NULL, &xMaterialsBuffer_,
		NULL, (u_long*)&xMaterialCount_, &xMesh_));
	// @ꍇ͎
	if((xMesh_->GetFVF() & D3DFVF_NORMAL) == 0){
		Direct3DXMesh* tempMesh;
		DirectXCheck(xMesh_->CloneMeshFVF(xMesh_->GetOptions(),
			(xMesh_->GetFVF() | D3DFVF_NORMAL),
			LampGraphics::getDirect3DDevice(), &tempMesh));
		D3DXComputeNormals(tempMesh, NULL);
		SafeRelease(xMesh_);
		xMesh_ = tempMesh;
	}
	// Agr[g쐬
	DirectXCheck(xMesh_->OptimizeInplace(
		(D3DXMESHOPT_ATTRSORT | D3DXMESHOPT_COMPACT),
		NULL, NULL, NULL, NULL));

	// Agr[g̉
	if(!analyzeAttribute()){ return false; }

	// _Lq̉
	if(!analyzeDeclaration()){ return false; }

	// f̏
	initializeModel();

	// bV̍쐬
	D3DXMATERIAL* xMaterials;
	xMaterials = (D3DXMATERIAL*)xMaterialsBuffer_->GetBufferPointer();
	for(u_int i = 0; i < xAttributeCount_; i++){
		const D3DXATTRIBUTERANGE& attribute = xAttributes_[i];
		const D3DXMATERIAL& material = xMaterials[attribute.AttribId];
		if(!analyzeMesh(xFilePath, attribute, material)){ return false; }
	}
	return true;
}
//------------------------------------------------------------------------------
// f̏
void XMeshTranslator::initializeModel(){
	Assert(scene_ != NULL);
	Assert(!name_.isEmpty());
	Assert(xMesh_ != NULL);
	// V[m[h
	SceneNodeManager* sceneNodeManager = scene_->getSceneNodeManager();
	String sceneNodeName = sceneNodeManager->rename(name_ + sceneNodeSuffix);
	sceneNode_ = sceneNodeManager->createSceneNode(sceneNodeName);
	scene_->getRootNode()->addSceneNode(sceneNode_);
	// f
	ModelManager* modelManager = scene_->getModelManager();
	String modelName = modelManager->rename(name_ + modelSuffix);
	model_ = modelManager->createStandardModel(modelName);
	sceneNode_->addSceneLeaf(model_);
}
//------------------------------------------------------------------------------
// Agr[g̉
bool XMeshTranslator::analyzeAttribute(){
	Assert(xMesh_ != NULL);
	DirectXCheck(xMesh_->GetAttributeTable(NULL, (u_long*)&xAttributeCount_));
	xAttributes_ = new D3DXATTRIBUTERANGE[xAttributeCount_];
	DirectXCheck(xMesh_->GetAttributeTable(
		xAttributes_, (u_long*)&xAttributeCount_));
	return true;
}
//------------------------------------------------------------------------------
// _Lq̉
bool XMeshTranslator::analyzeDeclaration(){
	Assert(xMesh_ != NULL);
	// _TCỶ
	vertexSize_ = xMesh_->GetNumBytesPerVertex();
	// e_Lq̉
	D3DVertexElement elements[MAX_FVF_DECL_SIZE];
	DirectXCheck(xMesh_->GetDeclaration(elements));
	for(int i = 0; i < MAX_FVF_DECL_SIZE; i++){
		const D3DVertexElement& element = elements[i];
		// D3DDECL_END
		if(element.Stream == 0xff){ break; }
		// Xg[0ȊO͖
		if(element.Stream != 0){ continue; }
		// vZbVO\bhDefault̂
		if(element.Method != D3DDECLMETHOD_DEFAULT){ continue; }
		// ʒủ
		if(element.Usage == D3DDECLUSAGE_POSITION){
			// float3ȊO͖
			if(element.Type != D3DDECLTYPE_FLOAT3){ continue; }
			// gp@CfbNX0̂
			if(element.UsageIndex != 0){ continue; }
			positionOffset_ = element.Offset;
		// @̉
		}else if(element.Usage == D3DDECLUSAGE_NORMAL){
			// float3ȊO͖
			if(element.Type != D3DDECLTYPE_FLOAT3){ continue; }
			// gp@CfbNX0̂
			if(element.UsageIndex != 0){ continue; }
			normalOffset_ = element.Offset;
		// UV̉
		}else if(element.Usage == D3DDECLUSAGE_TEXCOORD){
			// float2ȊO͖
			if(element.Type != D3DDECLTYPE_FLOAT2){ continue; }
			// gp@CfbNX0̂
			if(element.UsageIndex != 0){ continue; }
			uvOffset_ = element.Offset;
		}
	}
	if(positionOffset_ == -1){
		ErrorOut("XMeshTranslator::analyzeDeclaration() "
			"_LqɈʒu܂");
		return false;
	}
	return true;
}
//------------------------------------------------------------------------------
// bV̉
bool XMeshTranslator::analyzeMesh(const FilePath& xFilePath,
	const D3DXATTRIBUTERANGE& attribute, const D3DXMATERIAL& material){
	// bV쐬
	MeshManager* meshManager = scene_->getMeshManager();
	String meshName = meshManager->rename(name_ + meshSuffix);
	Mesh* mesh = meshManager->createRigidMesh(meshName);
	model_->addMesh(mesh);
	// bVf[^쐬
	MeshDataManager* meshDataManager = scene_->getMeshDataManager();
	String meshDataName = meshDataManager->rename(name_ + meshDataSuffix);
	MeshData* meshData = meshDataManager->createMeshData(meshDataName);
	mesh->setMeshData(meshData);
	// }eA쐬
	XMaterialTranslator xMaterialTranslator;
	if(!xMaterialTranslator.translate(xFilePath, scene_, name_, material)){
		return false;
	}
	mesh->setMaterial(xMaterialTranslator.getMaterial());

	// bV̉
	meshData->setPrimitiveType(Mesh::indexedTriangleList);
	// CfbNXobt@̉
	u_short* indexAddress;
	DirectXCheck(xMesh_->LockIndexBuffer(
		D3DLOCK_READONLY, (void**)&indexAddress));
	indexAddress += (attribute.FaceStart * 3);
	u_int faceCount = attribute.FaceCount;
	u_int indexOffset = attribute.VertexStart;
	meshData->setVertexIndexCount(faceCount * 3);
	for(u_int i = 0; i < faceCount; i++){
		int offset = i * 3;
		// ʂ̉]𔽓]
		meshData->setVertexIndex(offset + 2, (*indexAddress) - indexOffset);
		indexAddress++;
		meshData->setVertexIndex(offset + 1, (*indexAddress) - indexOffset);
		indexAddress++;
		meshData->setVertexIndex(offset + 0, (*indexAddress) - indexOffset);
		indexAddress++;
	}
	DirectXCheck(xMesh_->UnlockIndexBuffer());

	// _obt@̉
	u_char* vertexAddress;
	DirectXCheck(xMesh_->LockVertexBuffer(
		D3DLOCK_READONLY, (void**)&vertexAddress));
	vertexAddress += (attribute.VertexStart * vertexSize_);
	u_int vertexCount = attribute.VertexCount;
	meshData->setVertexCount(vertexCount);
	Assert(positionOffset_ != -1);
	if(normalOffset_ != -1){ meshData->enableNormal(true); }
	if(uvOffset_ != -1){
		meshData->setTexCoordSetCount(1);
		meshData->setTexCoordType(0, TexCoord::type2);
	}
	for(u_int i = 0; i < vertexCount; i++){
		// ʒu
		Vector3 position = *(Vector3*)(vertexAddress + positionOffset_);
		position.z = -position.z; // Z]
		meshData->setPosition(i, position);
		// @
		if(normalOffset_ != -1){
			Vector3 normal = *(Vector3*)(vertexAddress + normalOffset_);
			normal.z = -normal.z; // Z]
			meshData->setNormal(i, normal);
		}
		// UV
		if(uvOffset_ != -1){
			TexCoord2 uv = *(TexCoord2*)(vertexAddress + uvOffset_);
			meshData->setTexCoord2(i, 0, uv);
		}
		vertexAddress += vertexSize_;
	}
	DirectXCheck(xMesh_->UnlockVertexBuffer());
	return true;
}
//------------------------------------------------------------------------------
