//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * WIgTv RW
 * @author Junpee
 */

#include <LampBasic.h>
#include <Framework/System/SceneFramework.h>
#include <Graphics/PrimitiveRenderer/PrimitiveRenderer.h>
#include <Graphics/Renderer/InformationRenderer.h>
#include <Graphics/Scene/Scene.h>
#include <Graphics/Camera/Camera.h>
#include <Input/Pad/PS2Pad.h>

#include "Geometry/Primitive/Capsule.h"
#include "Geometry/Primitive/Cone.h"
#include "Geometry/Primitive/OrientedBox.h"
#include "Geometry/Primitive/Triangle.h"

//------------------------------------------------------------------------------
// vvZX萔
//------------------------------------------------------------------------------
// POINT͉Ev~eBuł̂ݎgp\
#define POINT				(0)
#define AXIS_ALIGNED_BOX	(POINT + 1)
#define CAPSULE				(AXIS_ALIGNED_BOX + 1)
#define CONE				(CAPSULE + 1)
#define LINE				(CONE + 1)
#define ORIENTED_BOX		(LINE + 1)
#define PLANE				(ORIENTED_BOX + 1)
#define RAY					(PLANE + 1)
#define SEGMENT				(RAY + 1)
#define SPHERE				(SEGMENT + 1)
#define TRIANGLE			(SPHERE + 1)

//------------------------------------------------------------------------------
// vO[h
//------------------------------------------------------------------------------
/// XeBbNő삷v~eBu
#define LEFT_PRIMITIVE CAPSULE
/// EXeBbNő삷v~eBu
#define RIGHT_PRIMITIVE POINT

//------------------------------------------------------------------------------

// Lamp̃l[Xy[XLɂ
using namespace Lamp;

//------------------------------------------------------------------------------
// NX`Aʏ̓wb_t@CɊi[
//------------------------------------------------------------------------------
/**
 * WIgTv RW
 */
class SampleGeometryCollision : public SceneFramework{
public:
	//--------------------------------------------------------------------------
	/**
	 * RXgN^
	 */
	SampleGeometryCollision();

	/**
	 * 
	 * @return ɐtrueԂ
	 */
	virtual bool initialize();

	/**
	 * n
	 */
	virtual void finalize();

	/**
	 * s
	 */
	virtual void run();

	/**
	 * _O
	 */
	virtual void renderSetup();

	/**
	 * _O
	 */
	virtual void render();

	/**
	 * `
	 */
	virtual void drawInformation();

private:
	//--------------------------------------------------------------------------
	// pbh
	virtual void padControll();

	// v~eBu_
	PrimitiveRenderer* primitiveRenderer_;
	// 
	bool intersected_;
	// s
	Matrix34 leftMatrix_;
	// XP[
	Vector3 leftScale_;
	// ]
	Quaternion leftRotation_;
	// ʒu
	Vector3 leftPosition_;
	// [h
	int leftMode_;
	// Es
	Matrix34 rightMatrix_;
	// EXP[
	Vector3 rightScale_;
	// E]
	Quaternion rightRotation_;
	// Eʒu
	Vector3 rightPosition_;
	// E[h
	int rightMode_;

	//--------------------------------------------------------------------------
	// v~eBuR[h
	//--------------------------------------------------------------------------
	// v~eBu
	virtual void initializePrimitive(Vector3* primitive){
		primitive->set(0.f, 5.f, 0.f);
	}

	// v~eBu
	virtual void initializePrimitive(AxisAlignedBox* primitive){
		primitive->set(-4.f, -5.f, -6.f, 6.f, 5.f, 4.f);
	}

	// v~eBu
	virtual void initializePrimitive(Capsule* primitive){
		primitive->set(0.f, 0.f, 0.f, 0.f, 10.f, 0.f, 3.f);
	}

	// v~eBu
	virtual void initializePrimitive(Cone* primitive){
		primitive->set(0.f, 5.f, 0.f, 5.f, -10.f, 0.f, Math::quadrantPI);
	}

	// v~eBu
	virtual void initializePrimitive(Line* primitive){
		primitive->set(0.f, 0.f, 0.f, 2.f, 2.f, 0.f);
	}

	// v~eBu
	virtual void initializePrimitive(OrientedBox* primitive){
		primitive->set(Matrix33::unit, Vector3(0.f, 3.f, 0.f),
			Vector3(4.f, 5.f, 6.f));
	}

	// v~eBu
	virtual void initializePrimitive(Plane* primitive){
		primitive->set(0.f, 1.f, 0.f, -5.f);
	}

	// v~eBu
	virtual void initializePrimitive(Ray* primitive){
		primitive->set(0.f, 0.f, 0.f, -0.707f, 0.707f, 0.f);
	}

	// v~eBu
	virtual void initializePrimitive(Segment* primitive){
		primitive->set(0.f, 0.f, 0.f, 5.f, 5.f, 0.f);
	}

	// v~eBu
	virtual void initializePrimitive(Sphere* primitive){
		primitive->set(0.f, 0.f, 0.f, 5.f);
	}

	// v~eBu
	virtual void initializePrimitive(Triangle* primitive){
		primitive->set(0.f, 0.f, 0.f, 5.f, 0.f, 0.f, 0.f, 0.f, -5.f);
	}

	//--------------------------------------------------------------------------
	// [hv~eBǔvZ
	virtual Vector3 calcWorldPrimitive(
		const Matrix34& matrix, const Vector3& primitive){
		return matrix * primitive;
	}

	// [hv~eBǔvZ
	virtual AxisAlignedBox calcWorldPrimitive(
		const Matrix34& matrix, const AxisAlignedBox& primitive){
		return primitive.transform(matrix);
	}

	// [hv~eBǔvZ
	virtual Capsule calcWorldPrimitive(
		const Matrix34& matrix, const Capsule& primitive){
		return primitive.scaledTransform(matrix);
	}

	// [hv~eBǔvZ
	virtual Cone calcWorldPrimitive(
		const Matrix34& matrix, const Cone& primitive){
		return primitive.transform(matrix);
	}

	// [hv~eBǔvZ
	virtual Line calcWorldPrimitive(
		const Matrix34& matrix, const Line& primitive){
		return primitive.transform(matrix);
	}

	// [hv~eBǔvZ
	virtual OrientedBox calcWorldPrimitive(
		const Matrix34& matrix, const OrientedBox& primitive){
		return primitive.transform(matrix);
	}

	// [hv~eBǔvZ
	virtual Plane calcWorldPrimitive(
		const Matrix34& matrix, const Plane& primitive){
			return primitive.scaledTransform(matrix).normalize();
	}

	// [hv~eBǔvZ
	virtual Ray calcWorldPrimitive(
		const Matrix34& matrix, const Ray& primitive){
		return primitive.transform(matrix);
	}

	// [hv~eBǔvZ
	virtual Segment calcWorldPrimitive(
		const Matrix34& matrix, const Segment& primitive){
		return primitive.transform(matrix);
	}

	// [hv~eBǔvZ
	virtual Sphere calcWorldPrimitive(
		const Matrix34& matrix, const Sphere& primitive){
		return primitive.scaledTransform(matrix);
	}

	// [hv~eBǔvZ
	virtual Triangle calcWorldPrimitive(
		const Matrix34& matrix, const Triangle& primitive){
		return primitive.transform(matrix);
	}

	//--------------------------------------------------------------------------
	// v~eBu̕`
	virtual void drawPrimitive(const Vector3& world, const Vector3& local,
		const Matrix34& worldMatrix, Color4c color){
		primitiveRenderer_->requestPoint(
			Vector3::unitScale, Vector3::zero, world, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const AxisAlignedBox& world,
		const AxisAlignedBox& local, const Matrix34& worldMatrix,
		Color4c color){
		// [h{bNX
		primitiveRenderer_->requestBox(world.getSize(), Vector3::zero,
			world.getCenter(), color); 
		// [h
		primitiveRenderer_->requestAxis(Vector3::unitScale, Vector3::zero,
			world.getCenter(), color);
		// [J{bNX
		Matrix34 matrix;
		matrix.setScale(local.getSize());
		matrix.addTranslation(local.getCenter());
		matrix = worldMatrix * matrix;
		color.a = 128;
		primitiveRenderer_->requestBox(matrix, color);
		// [J
		matrix.setTranslation(local.getCenter());
		matrix = worldMatrix * matrix;
		color.a = 255;
		primitiveRenderer_->requestAxis(worldMatrix, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Capsule& world, const Capsule& local,
		const Matrix34& worldMatrix, Color4c color){
		float radius = world.getRadius();
		primitiveRenderer_->requestSphere(Vector3(radius, radius, radius),
			Vector3::zero, world.getSourcePosition(), color);
		primitiveRenderer_->requestSphere(Vector3(radius, radius, radius),
			Vector3::zero, world.getTargetPosition(), color);

		Matrix34 matrix;
		const Vector3& direction = world.getDirection();
		float length = direction.getLength();
		matrix.setScale(radius, length, radius);
		matrix.addRotationX(Math::halfPI);
		matrix.addRotationXYZ(Math3D::lookAtZ(direction));
		matrix.addTranslation(world.getOrigin());
		primitiveRenderer_->requestCylinder(matrix, color);

/*
		Matrix34 matrix;
		matrix.setTranslation(Vector3(0.f, -1.f, 0.f));
		const Vector3& direction = world.getDirection();
		float length = direction.getLength();
		float cos = world.getCos();
		matrix.addScale(length * cos, length, length * cos);
		matrix.addRotationX(-Math::halfPI);
		matrix.addRotationXYZ(Math3D::lookAtZ(direction));
		matrix.addTranslation(world.getOrigin());
		primitiveRenderer_->requestCone(matrix, color);
*/
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Cone& world, const Cone& local,
		const Matrix34& worldMatrix, Color4c color){
		Matrix34 matrix;
		matrix.setTranslation(Vector3(0.f, -1.f, 0.f));
		const Vector3& direction = world.getDirection();
		float length = direction.getLength();
		float cos = world.getCos();
		matrix.addScale(length * cos, length, length * cos);
		matrix.addRotationX(-Math::halfPI);
		matrix.addRotationXYZ(Math3D::lookAtZ(direction));
		matrix.addTranslation(world.getOrigin());
		primitiveRenderer_->requestCone(matrix, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Line& world, const Line& local,
		const Matrix34& worldMatrix, Color4c color){
		Vector3 positions[2];
		positions[0] = world.getOrigin() - world.getDirection() * 10000.f;
		positions[1] = world.getOrigin() + world.getDirection() * 10000.f;
		primitiveRenderer_->requestLine(2, positions, Matrix34::unit, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const OrientedBox& world,
		const OrientedBox& local, const Matrix34& worldMatrix, Color4c color){
		Matrix34 matrix, tempMatrix;
		matrix.setScale(world.getSize());
		tempMatrix.set(world.getRotationMatrix());
		matrix = tempMatrix * matrix;
		matrix.addTranslation(world.getCenter());
		primitiveRenderer_->requestBox(matrix, color);

		matrix.setScale(Vector3::unitScale);
		matrix = tempMatrix * matrix;
		matrix.addTranslation(world.getCenter());
		primitiveRenderer_->requestAxis(matrix, color);

		AxisAlignedBox box = local.scaledTransform(worldMatrix);
		primitiveRenderer_->requestBox(box.getSize(), Vector3::zero,
			box.getCenter(), color); 
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Plane& world, const Plane& local,
		const Matrix34& worldMatrix, Color4c color){
		Plane plane(world);
		plane.normalize();
		Vector3 normal = plane.getNormal();
		primitiveRenderer_->requestPlane(Vector3::unitScale,
			Math3D::lookAtZ(normal), normal * -plane.getConstant(), color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Ray& world, const Ray& local,
		const Matrix34& worldMatrix, Color4c color){
		Vector3 positions[2];
		positions[0] = world.getOrigin();
		positions[1] = world.getOrigin() + world.getDirection() * 10000.f;
		primitiveRenderer_->requestLine(2, positions, Matrix34::unit, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Segment& world, const Segment& local,
		const Matrix34& worldMatrix, Color4c color){
		Vector3 positions[2];
		positions[0] = world.getSourcePosition();
		positions[1] = world.getTargetPosition();
		primitiveRenderer_->requestLine(2, positions, Matrix34::unit, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Sphere& world, const Sphere& local,
		const Matrix34& worldMatrix, Color4c color){
		float radius = world.getRadius();
		Vector3 scale(radius, radius, radius);
		primitiveRenderer_->requestSphere(
			scale, Vector3::zero, world.getCenter(), color);
		primitiveRenderer_->requestAxis(worldMatrix, color);
	}

	// v~eBu̕`
	virtual void drawPrimitive(const Triangle& world, const Triangle& local,
		const Matrix34& worldMatrix, Color4c color){
		Vector3 positions[6];
		positions[0] = world.getVertex(0);
		positions[1] = world.getVertex(1);
		positions[2] = world.getVertex(1);
		positions[3] = world.getVertex(2);
		positions[4] = world.getVertex(0);
		positions[5] = world.getVertex(2);
		primitiveRenderer_->requestLine(6, positions, Matrix34::unit, color);
	}

	//--------------------------------------------------------------------------
	// v~eBu
#if (LEFT_PRIMITIVE == POINT)
	Vector3 left_;
	Vector3 worldLeft_;
#elif (LEFT_PRIMITIVE == AXIS_ALIGNED_BOX)
	AxisAlignedBox left_;
	AxisAlignedBox worldLeft_;
#elif (LEFT_PRIMITIVE == CAPSULE)
	Capsule left_;
	Capsule worldLeft_;
#elif (LEFT_PRIMITIVE == CONE)
	Cone left_;
	Cone worldLeft_;
#elif (LEFT_PRIMITIVE == LINE)
	Line left_;
	Line worldLeft_;
#elif (LEFT_PRIMITIVE == ORIENTED_BOX)
	OrientedBox left_;
	OrientedBox worldLeft_;
#elif (LEFT_PRIMITIVE == PLANE)
	Plane left_;
	Plane worldLeft_;
#elif (LEFT_PRIMITIVE == RAY)
	Ray left_;
	Ray worldLeft_;
#elif (LEFT_PRIMITIVE == SEGMENT)
	Segment left_;
	Segment worldLeft_;
#elif (LEFT_PRIMITIVE == SPHERE)
	Sphere left_;
	Sphere worldLeft_;
#elif (LEFT_PRIMITIVE == TRIANGLE)
	Triangle left_;
	Triangle worldLeft_;
#endif

	// Ev~eBu
#if (RIGHT_PRIMITIVE == POINT)
	Vector3 right_;
	Vector3 worldRight_;
#elif (RIGHT_PRIMITIVE == AXIS_ALIGNED_BOX)
	AxisAlignedBox right_;
	AxisAlignedBox worldRight_;
#elif (RIGHT_PRIMITIVE == CAPSULE)
	Capsule right_;
	Capsule worldRight_;
#elif (RIGHT_PRIMITIVE == CONE)
	Cone right_;
	Cone worldRight_;
#elif (RIGHT_PRIMITIVE == LINE)
	Line right_;
	Line worldRight_;
#elif (RIGHT_PRIMITIVE == ORIENTED_BOX)
	OrientedBox right_;
	OrientedBox worldRight_;
#elif (RIGHT_PRIMITIVE == PLANE)
	Plane right_;
	Plane worldRight_;
#elif (RIGHT_PRIMITIVE == RAY)
	Ray right_;
	Ray worldRight_;
#elif (RIGHT_PRIMITIVE == SEGMENT)
	Segment right_;
	Segment worldRight_;
#elif (RIGHT_PRIMITIVE == SPHERE)
	Sphere right_;
	Sphere worldRight_;
#elif (RIGHT_PRIMITIVE == TRIANGLE)
	Triangle right_;
	Triangle worldRight_;
#endif

};
//------------------------------------------------------------------------------
// vOGg|Cg
//------------------------------------------------------------------------------
// C֐
int WINAPI WinMain(
	HINSTANCE instance, HINSTANCE previousInstance, LPSTR command, int show){
	SampleGeometryCollision sample;
	return sample.execute(instance);
}
//------------------------------------------------------------------------------
// NX
//------------------------------------------------------------------------------
// RXgN^
SampleGeometryCollision::SampleGeometryCollision() :
	SceneFramework("Tutorial1_LoadScene"), primitiveRenderer_(NULL),
	leftScale_(Vector3::unitScale), leftRotation_(Quaternion::identity),
	leftPosition_(Vector3::zero), leftMode_(0),
	rightScale_(Vector3::unitScale), rightRotation_(Quaternion::identity),
	rightPosition_(Vector3::zero), rightMode_(0){
}
//------------------------------------------------------------------------------
// 
bool SampleGeometryCollision::initialize(){
	primitiveRenderer_ = new PrimitiveRenderer();
	// Jʒuݒ
	scene_->getCurrentCamera()->setTransformation(
		Vector3::zero, Vector3(0.f, 5.f, 50.f));
	// \
	informationRenderer_->setDrawnAxis(true);
	informationRenderer_->setDrawnGrid(true);
	// v~eBu
	initializePrimitive(&left_);
	initializePrimitive(&right_);
	return true;
}
//------------------------------------------------------------------------------
// n
void SampleGeometryCollision::finalize(){
	SafeDelete(primitiveRenderer_);
}
//------------------------------------------------------------------------------
// s
void SampleGeometryCollision::run(){
	// pbh
	padControll();

	// [hv~eBu̎Zo
	leftMatrix_.setTransformationQuaternion(
		leftScale_, leftRotation_, leftPosition_);
	worldLeft_ = calcWorldPrimitive(leftMatrix_, left_);
	rightMatrix_.setTransformationQuaternion(
		rightScale_, rightRotation_, rightPosition_);
	worldRight_ = calcWorldPrimitive(rightMatrix_, right_);

	// `FbN
	intersected_ = worldLeft_.intersect(worldRight_);
}
//------------------------------------------------------------------------------
// pbh
void SampleGeometryCollision::padControll(){
	if(pad_ == NULL){ return; }
	float positionSens = 0.25f;
	float rotationSens = 0.1f;
	float scaleSens = 0.01f;
	// v~eBu
	if(pad_->buttonDown(PS2Pad::buttonL3)){
		leftMode_++;
		if(leftMode_ == 3){ leftMode_ = 0; }
	}
	if(leftMode_ == 0){
		leftPosition_.x += pad_->getLeftXAxis() * positionSens;
		leftPosition_.z += pad_->getLeftYAxis() * positionSens;
		if(pad_->buttonPressed(PS2Pad::buttonL1)){
			leftPosition_.y += positionSens;
		}
		if(pad_->buttonPressed(PS2Pad::buttonL2)){
			leftPosition_.y -= positionSens;
		}
	}else if(leftMode_ == 1){
		Quaternion xRotation, yRotation;
		yRotation.setRotationAxis(Vector3::unitY,
			-pad_->getLeftXAxis() * rotationSens);
		leftRotation_ = yRotation * leftRotation_;
		xRotation.setRotationAxis(Vector3::unitX,
			-pad_->getLeftYAxis() * rotationSens);
		leftRotation_ = leftRotation_ * xRotation;
		leftRotation_.normalize();
	}else if(leftMode_ == 2){
		leftScale_.x += pad_->getLeftXAxis() * scaleSens;
		leftScale_.z += -pad_->getLeftYAxis() * scaleSens;
		if(pad_->buttonPressed(PS2Pad::buttonL1)){
			leftScale_.y += scaleSens;
		}
		if(pad_->buttonPressed(PS2Pad::buttonL2)){
			leftScale_.y -= scaleSens;
		}
	}
	// Ev~eBu
	if(pad_->buttonDown(PS2Pad::buttonR3)){
		rightMode_++;
		if(rightMode_ == 3){ rightMode_ = 0; }
	}
	if(rightMode_ == 0){
		rightPosition_.x += pad_->getRightXAxis() * positionSens;
		rightPosition_.z += pad_->getRightYAxis() * positionSens;
		if(pad_->buttonPressed(PS2Pad::buttonR1)){
			rightPosition_.y += positionSens;
		}
		if(pad_->buttonPressed(PS2Pad::buttonR2)){
			rightPosition_.y -= positionSens;
		}
	}else if(rightMode_ == 1){
		Quaternion xRotation, yRotation;
		yRotation.setRotationAxis(Vector3::unitY,
			-pad_->getRightXAxis() * rotationSens);
		rightRotation_ = yRotation * rightRotation_;
		xRotation.setRotationAxis(Vector3::unitX,
			-pad_->getRightYAxis() * rotationSens);
		rightRotation_ = rightRotation_ * xRotation;
		rightRotation_.normalize();
	}else if(rightMode_ == 2){
		rightScale_.x += pad_->getRightXAxis() * scaleSens;
		rightScale_.z += -pad_->getRightYAxis() * scaleSens;
		if(pad_->buttonPressed(PS2Pad::buttonR1)){
			rightScale_.y += scaleSens;
		}
		if(pad_->buttonPressed(PS2Pad::buttonR2)){
			rightScale_.y -= scaleSens;
		}
	}
}
//------------------------------------------------------------------------------
// _O
void SampleGeometryCollision::renderSetup(){
	// v~eBu`
	Color4c color(Color4c::white);
	if(intersected_){ color = Color4c(255, 128, 128, 255); }
	drawPrimitive(worldLeft_, left_, leftMatrix_, color);
	drawPrimitive(worldRight_, right_, rightMatrix_, color);
}
//------------------------------------------------------------------------------
// _O
void SampleGeometryCollision::render(){
	primitiveRenderer_->render(scene_->getCurrentCamera());
}
//------------------------------------------------------------------------------
// `
void SampleGeometryCollision::drawInformation(){
	BasicFramework::drawInformation();
	String output, temp;
	String mode[] = { "Position", "Rotation", "Scale" };
	output += "Left " + mode[leftMode_] + "\n";
	output += worldLeft_.toString() + "\n";
	temp.format(" Scale    ( %5.2f %5.2f %5.2f )\n",
		leftScale_.x, leftScale_.y, leftScale_.z);
	output += temp;
	temp.format(" Rotation ( %5.2f %5.2f %5.2f %5.2f )\n",
		leftRotation_.x, leftRotation_.y, leftRotation_.z, leftRotation_.w);
	output += temp;
	temp.format(" Position ( %5.2f %5.2f %5.2f )\n",
		leftPosition_.x, leftPosition_.y, leftPosition_.z);
	output += temp;

	output += "Right " + mode[rightMode_] + "\n";
	output += worldRight_.toString() + "\n";
	temp.format(" Scale    ( %5.2f %5.2f %5.2f )\n",
		rightScale_.x, rightScale_.y, rightScale_.z);
	output += temp;
	temp.format(" Rotation ( %5.2f %5.2f %5.2f %5.2f )\n",
		rightRotation_.x, rightRotation_.y, rightRotation_.z, rightRotation_.w);
	output += temp;
	temp.format(" Position ( %5.2f %5.2f %5.2f )\n",
		rightPosition_.x, rightPosition_.y, rightPosition_.z);
	output += temp;

	drawInformationString(output);
}
//------------------------------------------------------------------------------
