//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * LampgX[^
 * @author Junpee
 */

#include "System/stdafx.h"
#include "Translator/Core/LampTranslator.h"
#include "Translator/Scene/TranslationScene.h"
#include <Core/Utility/Timer.h>
#include <Core/Utility/StringTokenizer.h>
#include <Graphics/System/LampGraphics.h>
#include <Graphics/InputOutput/TextSceneSaver.h>
#include <Graphics/InputOutput/BinarySceneSaver.h>
#include <Graphics/SceneFilter/SceneFilter.h>
#include <Animation/System/AnimationManager.h>
#include <Animation/System/AnimationSet.h>
#include <Animation/Utility/AnimationCompressor.h>
#include <Animation/InputOutput/TextAnimationSaver.h>
#include <Animation/InputOutput/BinaryAnimationSaver.h>

namespace LampForMaya{

//------------------------------------------------------------------------------
// RXgN^
LampTranslator::LampTranslator() :
	translationScene_(NULL), scene_(NULL), animationManager_(NULL),
	animationSet_(NULL), exportSceneFlag_(true), picturePath_("pictures/"),
	meshOptimizeFlag_(true), characterScale_(1.f), exportAnimationFlag_(true),
	deleteStaticChannelsFlag_(false){
}
//------------------------------------------------------------------------------
// fXgN^
LampTranslator::~LampTranslator(){
	Assert(translationScene_ == NULL);
	Assert(scene_ == NULL);
	Assert(animationManager_ == NULL);
	Assert(animationSet_ == NULL);
}
//------------------------------------------------------------------------------
// o
MStatus LampTranslator::write(const MFileObject& file,
	const MString& optionsString, MPxFileTranslator::FileAccessMode mode,
	bool textFlag){
	Timer::Tick startTime = Timer::getTick();
	textExportFlag_ = textFlag;
	// SăGNX|[ĝݎ󂯕t
	if(mode != MPxFileTranslator::kExportAccessMode){
		MayaErrorOut("\"Export All\"T|[gĂ܂B");
		return errorCleanup();
	}
	// IvV
	analyzeOption(String(optionsString.asChar()));
	// JnbZ[Wo
	outputStartMessage();

	// ÓI`l폜
	if(exportAnimationFlag_ && deleteStaticChannelsFlag_){
		MGlobal::executeCommand("delete -all -staticChannels;", false, true);
	}

	//--------------------------------------------------------------------------
	// ϊpV[ɂ̎W
	Timer::Tick startSceneCollectionTime = Timer::getTick();
	translationScene_ = new TranslationScene();
	bool result = translationScene_->collection();
	if(!result){ return errorCleanup(); }
	float sceneCollectionTime =
		Timer::getInterval(startSceneCollectionTime)  * 0.001f;


	// ϊpV[ɂAj[V̎W
	Timer::Tick startAnimationCollectionTime = Timer::getTick();
	result = translationScene_->collectAnimation();
	if(!result){ return errorCleanup(); }
	float animationCollectionTime =
		Timer::getInterval(startAnimationCollectionTime)  * 0.001f;

	//--------------------------------------------------------------------------
	// LampV[ւ̕ϊ
	Timer::Tick startSceneTranslateTime = Timer::getTick();
	scene_ = LampGraphics::createScene("TranslateScene");
	result = translationScene_->convertToLamp(scene_);
	if(!result){ return errorCleanup(); }
	float sceneTranslateTime =
		Timer::getInterval(startSceneTranslateTime)  * 0.001f;

	// LampAj[Vւ̕ϊ
	Timer::Tick startAnimationTranslateTime = Timer::getTick();
	animationManager_ = new AnimationManager();
	String animationName = file.name().asChar();
	if(textFlag){
		Assert(animationName.endsWith("." + getTextSceneExtension()));
	}else{
		Assert(animationName.endsWith("." + getBinarySceneExtension()));
	}
	animationName = animationName.getSubstring(0, animationName.getSize() - 4);
	animationSet_ = animationManager_->createAnimationSet(animationName);
	result = translationScene_->convertAnimation(
		animationManager_, animationSet_);
	if(!result){ return errorCleanup(); }
	float animationTranslateTime =
		Timer::getInterval(startAnimationTranslateTime)  * 0.001f;

	// Aj[V̈k
	Timer::Tick startAnimationCompressTime = Timer::getTick();
	AnimationCompressor animationCompressor;
	animationCompressor.compress(animationSet_);
	float animationCompressTime =
		Timer::getInterval(startAnimationCompressTime)  * 0.001f;

	//--------------------------------------------------------------------------
	// ϊpV[̍폜
	translationScene_->clear();
	delete translationScene_;
	translationScene_ = NULL;

	//--------------------------------------------------------------------------
	// V[ւ̃tB^
	Timer::Tick startFilterTime = Timer::getTick();
	SceneFilter sceneFilter(scene_);
	// sN`pXύX
	if(exportSceneFlag_){
		if(!sceneFilter.filter(String("ChangePicturePath ") + picturePath_)){
			return errorCleanup();
		}
		// oEfBOZo
		String scaleString;
		scaleString.format("characterScale %.8f", characterScale_);
		// oEfBO{bNXZo
		if(!sceneFilter.filter("CalculateBoundingBox " + scaleString)){
			return errorCleanup();
		}
		// oEfBOXtBAZo
		if(!sceneFilter.filter("CalculateBoundingSphere " + scaleString)){
			return errorCleanup();
		}
		// bVœK
		if(meshOptimizeFlag_){
			// CfbNXgCAO\z
			if(!sceneFilter.filter("BuildIndexedTriangle")){
				return errorCleanup();
			}
		}
	}
	// V[̘_`FbN
	if(!sceneFilter.filter("SceneLogicCheck")){ return errorCleanup(); }
	float filterTime = Timer::getInterval(startFilterTime)  * 0.001f;

	//--------------------------------------------------------------------------
	// V[o
	Timer::Tick startSceneExportTime = Timer::getTick();
	if(exportSceneFlag_){
		if(textExportFlag_){
			TextSceneSaver* textSceneSaver = new TextSceneSaver();
			textSceneSaver->save(file.fullName().asChar(), scene_);
			delete textSceneSaver;
		}else{
			BinarySceneSaver* binarySceneSaver = new BinarySceneSaver();
			binarySceneSaver->save(file.fullName().asChar(), scene_);
			delete binarySceneSaver;
		}
	}
	float sceneExportTime = Timer::getInterval(startSceneExportTime)  * 0.001f;

	// LampV[̔j
	scene_->clear();
	LampGraphics::destroyScene(scene_);
	scene_ = NULL;

	//--------------------------------------------------------------------------
	// Aj[Vo
	Timer::Tick startAnimationExportTime = Timer::getTick();
	if(exportAnimationFlag_ && (animationSet_->getAnimationCount() != 0)){
		String animationFileName = file.fullName().asChar();
		if(textExportFlag_){
			Assert(animationFileName.endsWith("." + getTextSceneExtension()));
			animationFileName = animationFileName.getSubstring(
				0, animationFileName.getSize() - 3);
			animationFileName += getTextAnimationExtension();
			TextAnimationSaver* textAnimationSaver = new TextAnimationSaver();
			textAnimationSaver->save(animationFileName, animationManager_);
			delete textAnimationSaver;
		}else{
			Assert(animationFileName.endsWith("." + getBinarySceneExtension()));
			animationFileName = animationFileName.getSubstring(
				0, animationFileName.getSize() - 3);
			animationFileName += getBinaryAnimationExtension();
			BinaryAnimationSaver* binaryAnimationSaver =
				new BinaryAnimationSaver();
			binaryAnimationSaver->save(animationFileName, animationManager_);
			delete binaryAnimationSaver;
		}
	}
	float animationExportTime =
		Timer::getInterval(startAnimationExportTime)  * 0.001f;

	// LampAj[V̔j
	animationManager_->clear();
	delete animationManager_;
	animationSet_ = NULL;
	animationManager_ = NULL;

	//--------------------------------------------------------------------------
	// IbZ[W
	float totalTime = Timer::getInterval(startTime)  * 0.001f;
	float otherTime = totalTime - (
		sceneCollectionTime + animationCollectionTime +
		sceneTranslateTime + animationTranslateTime + animationCompressTime +
		filterTime + sceneExportTime + animationExportTime);
	String finishMessage;
	finishMessage.format("\nGNX|[gI  g[^@%.2fb\n"
		"@V[W  %.2fb\n"
		"@Aj[VW  %.2fb\n"
		"@V[ϊ  %.2fb\n"
		"@Aj[Vϊ  %.2fb\n"
		"@Aj[Vk  %.2fb\n"
		"@tB^  %.2fb\n"
		"@V[o  %.2fb\n"
		"@Aj[Vo  %.2fb\n"
		"@̑  %.2fb\n",
		totalTime, sceneCollectionTime, animationCollectionTime,
		sceneTranslateTime, animationTranslateTime, animationCompressTime,
		filterTime, sceneExportTime, animationExportTime, otherTime);
	MayaMessageOut(finishMessage);
	return MStatus(MStatus::kSuccess);
}
//------------------------------------------------------------------------------
// G[n
MStatus LampTranslator::errorCleanup(){
	if(translationScene_ != NULL){
		translationScene_->clear();
		delete translationScene_;
		translationScene_ = NULL;
	}
	if(scene_ != NULL){
		scene_->clear();
		LampGraphics::destroyScene(scene_);
		scene_ = NULL;
	}
	if(animationManager_ != NULL){
		animationManager_->clear();
		delete animationManager_;
		animationSet_ = NULL;
		animationManager_ = NULL;
	}
	return MStatus(MStatus::kFailure);
}
//------------------------------------------------------------------------------
// IvV
void LampTranslator::analyzeOption(const String& options){
	StringTokenizer optionList(options, ";");
	while(optionList.hasMoreTokens()){
		StringTokenizer option(optionList.getNextToken(), "=");
		if(!option.hasMoreTokens()){ continue; }
		String optionName = option.getNextToken();
		if(!option.hasMoreTokens()){ continue; }
		String optionData = option.getNextToken();
		if(optionName == "scene"){
			exportSceneFlag_ = (optionData != "0");
		}else if(optionName == "picturePath"){
			picturePath_ = optionData;
			if(!picturePath_.endsWith("/")){ picturePath_ += "/"; }
		}else if(optionName == "meshOptimize"){
			meshOptimizeFlag_ = (optionData != "0");
		}else if(optionName == "characterScale"){
			if(!optionData.parseFloat(&characterScale_)){
				characterScale_ = 1.f;
			}
		}else if(optionName == "animation"){
			exportAnimationFlag_ = (optionData != "0");
		}else if(optionName == "deleteStaticChannnels"){
			deleteStaticChannelsFlag_ = (optionData != "0");
		}
	}
}
//------------------------------------------------------------------------------
// JnbZ[Wo
void LampTranslator::outputStartMessage(){
	String startMessage;
	if(textExportFlag_){ startMessage += "\neLXgGNX|[gJn\n"; }
	else{ startMessage += "\noCiGNX|[gJn\n"; }
	if(exportSceneFlag_){ startMessage += "@V[ó@L\n"; }
	else{ startMessage += "@V[ó@\n"; }
	if(exportSceneFlag_){
		startMessage += "@@sN`o̓pX@" + picturePath_ + "\n";
		if(meshOptimizeFlag_){ startMessage += "@@bVœK@L\n"; }
		else{ startMessage += "@@bVœK@\n"; }
		String characterScaleString;
		characterScaleString.format(
			"@@LN^oEfBOXP[@%.8f\n", characterScale_);
		startMessage += characterScaleString;
	}
	if(exportAnimationFlag_){ startMessage += "@Aj[Vó@L\n"; }
	else{ startMessage += "@Aj[Vó@\n"; }
	if(exportAnimationFlag_){
		if(deleteStaticChannelsFlag_){
			startMessage += "@@ÓI`l폜@L\n";
		}else{
			startMessage += "@@ÓI`l폜@\n";
		}
	}
	startMessage += "\n";
	MayaMessageOut(startMessage);
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
