//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * eXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Geometry/Primitive/SphereTest.h"
#include "Geometry/Primitive/Sphere.h"

//------------------------------------------------------------------------------
// RXgN^
SphereTest::SphereTest(String name) : TestCase(name){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* SphereTest::suite(){
	TestSuite* suite = new TestSuite("SphereTest");
	suite->addTest(new TestCaller(SphereTest, testConstructor));
	suite->addTest(new TestCaller(SphereTest, testFixedNumber));
	suite->addTest(new TestCaller(SphereTest, testSetValue));
	suite->addTest(new TestCaller(SphereTest, testGetValue));
	suite->addTest(new TestCaller(SphereTest, testSphereArithmetic));
	suite->addTest(new TestCaller(SphereTest, testTransform));
	suite->addTest(new TestCaller(SphereTest, testLogicalOperation));
	suite->addTest(new TestCaller(SphereTest, testToString));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void SphereTest::testConstructor(){
	TestEquals(16, (int)sizeof(Sphere));
	Sphere sphere0(-1.f, -2.f, -3.f, 1.f);
	Sphere sphere1(Vector3(-1.f, -2.f, -3.f), 1.f);
	TestAssert(sphere0 == sphere1);
	Sphere sphere2(sphere1);
	TestAssert(sphere0 == sphere2);
	sphere1 = sphere2;
	TestAssert(sphere0 == sphere1);
	Sphere sphere3(Sphere::zero);
	TestAssert(sphere3 == Sphere(0.f, 0.f, 0.f, 0.f));
	float array[] = { -1.f, -2.f, -3.f, 1.f };
	Sphere sphere4(array);
	TestAssert(sphere0 == sphere4);
}
//------------------------------------------------------------------------------
// 萔eXg
void SphereTest::testFixedNumber(){
	Sphere sphere0(0.f, 0.f, 0.f, 0.f);
	TestAssert(sphere0 == Sphere::zero);
	Sphere sphere1(0.f, 0.f, 0.f, 1.f);
	TestAssert(sphere1 == Sphere::unit);
}
//------------------------------------------------------------------------------
// l̐ݒeXg
void SphereTest::testSetValue(){
	Sphere sphere0(-1.f, -2.f, -3.f, 1.f);
	Sphere sphere1;
	sphere1.set(-1.f, -2.f, -3.f, 1.f);
	TestAssert(sphere0 == sphere1);
	sphere1.set(Vector3(-1.f, -2.f, -3.f), 1.f);
	TestAssert(sphere0 == sphere1);
	float array[] = { -1.f, -2.f, -3.f, 1.f };
	sphere1.set(array);
	TestAssert(sphere0 == sphere1);
	sphere1.setCenter(-1.f, -2.f, -3.f);
	TestAssert(sphere0 == sphere1);
	sphere1.setCenter(Vector3(-1.f, -2.f, -3.f));
	TestAssert(sphere0 == sphere1);
	sphere1.setRadius(1.f);
	TestAssert(sphere0 == sphere1);
}
//------------------------------------------------------------------------------
// l̎擾eXg
void SphereTest::testGetValue(){
	Sphere sphere0(-1.f, -2.f, -3.f, 1.f);
	TestAssert(sphere0.getCenter() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(sphere0.getRadius() == 1.f);
}
//------------------------------------------------------------------------------
// Z
void SphereTest::testSphereArithmetic(){
	// Ƌ̃}[W
	Sphere sphere0(0.f, 0.f, 0.f, 5.f);
	Sphere sphere1(10.f, 0.f, 0.f, 1.f);
	sphere1.merge(sphere0);
	TestAssert(sphere1 == Sphere(3.f, 0.f, 0.f, 8.f));
	sphere1.set(0.f, 0.f, 0.f, 5.f);
	sphere1.merge(sphere0);
	TestAssert(sphere1 == Sphere(0.f, 0.f, 0.f, 5.f));
	sphere1.set(0.f, 0.f, 0.f, 5.f + Math::epsilon);
	sphere1.merge(sphere0);
	TestAssert(sphere1 == Sphere(0.f, 0.f, 0.f, 5.f + Math::epsilon));
	sphere1.set(1.f, 0.f, 0.f, 1.f);
	sphere1.merge(sphere0);
	TestAssert(sphere1 == Sphere(0.f, 0.f, 0.f, 5.f));
	sphere1.set(1.f, 0.f, 0.f, 10.f);
	sphere1.merge(sphere0);
	TestAssert(sphere1 == Sphere(1.f, 0.f, 0.f, 10.f));
	// Ɠ_̃}[W
	sphere1.set(0.f, 0.f, 0.f, 5.f);
	sphere1.merge(Vector3(11.f, 0.f, 0.f));
	TestAssert(sphere1 == Sphere(3.f, 0.f, 0.f, 8.f));
	sphere1.set(0.f, 0.f, 0.f, 5.f);
	sphere1.merge(Vector3(1.f, 0.f, 0.f));
	TestAssert(sphere1 == Sphere(0.f, 0.f, 0.f, 5.f));
	// ɓ_ǉ
	sphere1.set(0.f, 0.f, 0.f, 5.f);
	sphere1.append(Vector3(1.f, 0.f, 0.f));
	TestAssert(sphere1 == Sphere(0.f, 0.f, 0.f, 5.f));
	sphere1.append(Vector3(10.f, 0.f, 0.f));
	TestAssert(sphere1 == Sphere(0.f, 0.f, 0.f, 10.f));

	TestAssert(Sphere(0.f, 0.f, 0.f, 0.f).isZero());
	TestAssert(Sphere(0.f, 0.f, 0.f, 1.f).isUnit());
}
//------------------------------------------------------------------------------
// ϊeXg
void SphereTest::testTransform(){
	Sphere sphere0(1.f, 2.f, 3.f, 6.f);
	Sphere sphere1;

	Matrix33 mtx33;
	mtx33.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	sphere1 = sphere0.transform(mtx33);
	TestAssert(sphere1.epsilonEquals(
		Sphere(3.f, 2.f, -1.f, 6.f), Math::epsilon));

	Matrix34 mtx43;
	mtx43.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	sphere1 = sphere0.transform(mtx43);
	TestAssert(sphere1.epsilonEquals(
		Sphere(4.f, 3.f, 0.f, 6.f), Math::epsilon));

	Matrix44 mtx44;
	mtx44.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	sphere1 = sphere0.transform(mtx44);
	TestAssert(sphere1.epsilonEquals(
		Sphere(4.f, 1.f, 0.f, 6.f), Math::epsilon));

	// XP[
	mtx33.setScale(Vector3(1.f, 2.f, 1.f));
	mtx33.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	sphere1 = sphere0.scaledTransform(mtx33);
	TestAssert(sphere1.epsilonEquals(
		Sphere(3.f, 4.f, -1.f, 12.f), Math::epsilon));

	mtx43.setScale(Vector3(1.f, 2.f, 1.f));
	mtx43.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	sphere1 = sphere0.scaledTransform(mtx43);
	TestAssert(sphere1.epsilonEquals(
		Sphere(4.f, 5.f, 0.f, 12.f), Math::epsilon));

	mtx44.setScale(Vector3(1.f, 2.f, 1.f));
	mtx44.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	sphere1 = sphere0.scaledTransform(mtx44);
	TestAssert(sphere1.epsilonEquals(
		Sphere(4.f, 3.f, 0.f, 12.f), Math::epsilon));
}
//------------------------------------------------------------------------------
// _ZeXg
void SphereTest::testLogicalOperation(){
	Sphere sphere0(0.f, 0.f, 0.f, 1.f),
		sphere1(1.f, 1.f, 1.f, 2.f),
		sphere2(0.f, 0.f, 0.f, 1.f);
	TestAssert(!(sphere0 == sphere1));
	TestAssert(sphere0 == sphere2);
	TestAssert(sphere0 != sphere1);
	TestAssert(!(sphere0 != sphere2));
	TestAssert(sphere0.epsilonEquals(sphere1, 1.f));
	TestAssert(sphere0 != sphere1);
	TestAssert(sphere0.notEpsilonEquals(sphere1, 0.5f));
	TestAssert(!sphere0.notEpsilonEquals(sphere1, 1.f));
	TestAssert(!sphere0.notEpsilonEquals(sphere2, 0.f));
}
//------------------------------------------------------------------------------
// eXg
void SphereTest::testToString(){
	Sphere sphere0(12345678.f, 1234.5678f, 0.12345678f, 12345678.f);
	TestEquals("{ ( 12345678.00000000, 1234.56774902, 0.12345678 ) "
		"12345678.00000000 }", sphere0.toString());
}
//------------------------------------------------------------------------------
