//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * ZOgeXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Geometry/Primitive/SegmentTest.h"
#include "Geometry/Primitive/Segment.h"

//------------------------------------------------------------------------------
// RXgN^
SegmentTest::SegmentTest(String name) : TestCase(name){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* SegmentTest::suite(){
	TestSuite* suite = new TestSuite("SegmentTest");
	suite->addTest(new TestCaller(SegmentTest, testConstructor));
	suite->addTest(new TestCaller(SegmentTest, testFixedNumber));
	suite->addTest(new TestCaller(SegmentTest, testSetValue));
	suite->addTest(new TestCaller(SegmentTest, testGetValue));
	suite->addTest(new TestCaller(SegmentTest, testSegmentArithmetic));
	suite->addTest(new TestCaller(SegmentTest, testTransform));
	suite->addTest(new TestCaller(SegmentTest, testLogicalOperation));
	suite->addTest(new TestCaller(SegmentTest, testToString));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void SegmentTest::testConstructor(){
	TestEquals(24, (int)sizeof(Segment));
	Segment segment0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f);
	Segment segment1(Vector3(-1.f, -2.f, -3.f), Vector3(1.f, 2.f, 3.f));
	TestAssert(segment0 == segment1);
	Segment segment2(segment1);
	TestAssert(segment0 == segment2);
	segment1 = segment2;
	TestAssert(segment0 == segment1);
	Segment segment3(Segment::zero);
	TestAssert(segment3 == Segment(0.f, 0.f, 0.f, 0.f, 0.f, 0.f));
	float array[] = { -1.f, -2.f, -3.f, 1.f, 2.f, 3.f };
	Segment segment4(array);
	TestAssert(segment0 == segment4);
}
//------------------------------------------------------------------------------
// 萔eXg
void SegmentTest::testFixedNumber(){
	Segment segment0(0.f, 0.f, 0.f, 0.f, 0.f, 0.f);
	TestAssert(segment0 == Segment::zero);
}
//------------------------------------------------------------------------------
// l̐ݒeXg
void SegmentTest::testSetValue(){
	Segment segment0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f);
	Segment segment1;
	segment1.set(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f);
	TestAssert(segment0 == segment1);
	segment1.set(Vector3(-1.f, -2.f, -3.f), Vector3(1.f, 2.f, 3.f));
	TestAssert(segment0 == segment1);
	float array[] = { -1.f, -2.f, -3.f, 1.f, 2.f, 3.f };
	segment1.set(array);
	TestAssert(segment0 == segment1);
	segment1.setPositions(Vector3(-1.f, -2.f, -3.f), Vector3(0.f, 0.f, 0.f));
	TestAssert(segment0 == segment1);
}
//------------------------------------------------------------------------------
// l̎擾eXg
void SegmentTest::testGetValue(){
	Segment segment0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f);
	TestAssert(segment0.getOrigin() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(segment0.getDirection() == Vector3(1.f, 2.f, 3.f));
	TestAssert(segment0.getSourcePosition() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(segment0.getTargetPosition() == Vector3(0.f, 0.f, 0.f));
}
//------------------------------------------------------------------------------
// ZOgZ
void SegmentTest::testSegmentArithmetic(){
	TestAssert(Segment(0.f, 0.f, 0.f, 0.f, 0.f, 0.f).isZero());
	TestAssert(Segment(1.f, 2.f, 3.f, 0.f, 0.f, 0.f).isZero());
}
//------------------------------------------------------------------------------
// ϊeXg
void SegmentTest::testTransform(){
	Segment segment0(1.f, 2.f, 3.f, 1.f, 2.f, 3.f);
	Segment segment1(Segment::zero);

	Matrix33 mtx33;
	mtx33.setScale(Vector3(1.f, 2.f, 1.f));
	mtx33.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	segment1 = segment0.transform(mtx33);
	TestAssert(segment1.epsilonEquals(
		Segment(3.f, 4.f, -1.f, 3.f, 4.f, -1.f), Math::epsilon));

	Matrix34 mtx43;
	mtx43.setScale(Vector3(1.f, 2.f, 1.f));
	mtx43.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	segment1 = segment0.transform(mtx43);
	TestAssert(segment1.epsilonEquals(
		Segment(4.f, 5.f, 0.f, 3.f, 4.f, -1.f), Math::epsilon));

	Matrix44 mtx44;
	mtx44.setScale(Vector3(1.f, 2.f, 1.f));
	mtx44.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	segment1 = segment0.transform(mtx44);
	TestAssert(segment1.epsilonEquals(
		Segment(4.f, 3.f, 0.f, 3.f, 4.f, -1.f), Math::epsilon));
}
//------------------------------------------------------------------------------
// _ZeXg
void SegmentTest::testLogicalOperation(){
	Segment segment0(0.f, 0.f, 0.f, 1.f, 1.f, 1.f),
		segment1(1.f, 1.f, 1.f, 2.f, 2.f, 2.f),
		segment2(0.f, 0.f, 0.f, 1.f, 1.f, 1.f);
	TestAssert(!(segment0 == segment1));
	TestAssert(segment0 == segment2);
	TestAssert(segment0 != segment1);
	TestAssert(!(segment0 != segment2));
	TestAssert(segment0.epsilonEquals(segment1, 1.f));
	TestAssert(segment0 != segment1);
	TestAssert(segment0.notEpsilonEquals(segment1, 0.5f));
	TestAssert(!segment0.notEpsilonEquals(segment1, 1.f));
	TestAssert(!segment0.notEpsilonEquals(segment2, 0.f));
}
//------------------------------------------------------------------------------
// eXg
void SegmentTest::testToString(){
	Segment segment0(12345678.f, 1234.5678f, 0.12345678f,
		12345678.f, 1234.5678f, 0.12345678f);
	TestEquals("{ ( 12345678.00000000, 1234.56774902, 0.12345678 ) "
		"( 12345678.00000000, 1234.56774902, 0.12345678 ) }",
		segment0.toString());
}
//------------------------------------------------------------------------------
