//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * ʃeXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Geometry/Primitive/PlaneTest.h"
#include "Geometry/Primitive/Plane.h"

//------------------------------------------------------------------------------
// RXgN^
PlaneTest::PlaneTest(String name) : TestCase(name){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* PlaneTest::suite(){
	TestSuite* suite = new TestSuite("PlaneTest");
	suite->addTest(new TestCaller(PlaneTest, testConstructor));
	suite->addTest(new TestCaller(PlaneTest, testFixedNumber));
	suite->addTest(new TestCaller(PlaneTest, testSetValue));
	suite->addTest(new TestCaller(PlaneTest, testGetValue));
	suite->addTest(new TestCaller(PlaneTest, testPlaneArithmetic));
	suite->addTest(new TestCaller(PlaneTest, testTransform));
	suite->addTest(new TestCaller(PlaneTest, testLogicalOperation));
	suite->addTest(new TestCaller(PlaneTest, testToString));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void PlaneTest::testConstructor(){
	TestEquals(16, (int)sizeof(Plane));
	Plane plane0(-1.f, -2.f, -3.f, 1.f);
	Plane plane1(Vector3(-1.f, -2.f, -3.f), 1.f);
	TestAssert(plane0 == plane1);
	Plane plane2(plane1);
	TestAssert(plane0 == plane2);
	plane1 = plane2;
	TestAssert(plane0 == plane1);
	Plane plane3(Plane::zero);
	TestAssert(plane3 == Plane(0.f, 0.f, 0.f, 0.f));
	float array[] = { -1.f, -2.f, -3.f, 1.f };
	Plane plane4(array);
	TestAssert(plane0 == plane4);
	Plane plane5(Vector3(0.f, 1.f, 0.f),
		Vector3(0.f, 1.f, 1.f), Vector3(1.f, 1.f, 0.f));
	TestAssert(plane5 == Plane(0.f, 1.f, 0.f, -1.f));
	Plane plane6(Vector3(0.f, 1.f, 0.f), Vector3(1.f, 1.f, 1.f));
	TestAssert(plane6 == Plane(0.f, 1.f, 0.f, -1.f));
}
//------------------------------------------------------------------------------
// 萔eXg
void PlaneTest::testFixedNumber(){
	Plane plane0(0.f, 0.f, 0.f, 0.f);
	TestAssert(plane0 == Plane::zero);
	Plane plane1(1.f, 0.f, 0.f, 0.f);
	TestAssert(plane1 == Plane::unitX);
	Plane plane2(0.f, 1.f, 0.f, 0.f);
	TestAssert(plane2 == Plane::unitY);
	Plane plane3(0.f, 0.f, 1.f, 0.f);
	TestAssert(plane3 == Plane::unitZ);
}
//------------------------------------------------------------------------------
// l̐ݒeXg
void PlaneTest::testSetValue(){
	Plane plane0(-1.f, -2.f, -3.f, 1.f);
	Plane plane1;
	plane1.set(-1.f, -2.f, -3.f, 1.f);
	TestAssert(plane0 == plane1);
	plane1.set(Vector3(-1.f, -2.f, -3.f), 1.f);
	TestAssert(plane0 == plane1);
	float array[] = { -1.f, -2.f, -3.f, 1.f };
	plane1.set(array);
	TestAssert(plane0 == plane1);
	plane1.setNormal(-1.f, -2.f, -3.f);
	TestAssert(plane0 == plane1);
	plane1.setNormal(Vector3(-1.f, -2.f, -3.f));
	TestAssert(plane0 == plane1);
	plane1.setConstant(1.f);
	TestAssert(plane0 == plane1);
	plane1.set(1.f, 0.f, 0.f, 1.f);
	plane1.setLength(10.f);
	TestAssert(plane1 == Plane(10.f, 0.f, 0.f, 10.f));
	plane1.set(Vector3(0.f, 1.f, 0.f),
		Vector3(0.f, 1.f, 1.f), Vector3(1.f, 1.f, 0.f));
	TestAssert(plane1 == Plane(0.f, 1.f, 0.f, -1.f));
	plane1.set(Vector3(0.f, 1.f, 0.f), Vector3(1.f, 1.f, 1.f));
	TestAssert(plane1 == Plane(0.f, 1.f, 0.f, -1.f));
}
//------------------------------------------------------------------------------
// l̎擾eXg
void PlaneTest::testGetValue(){
	Plane plane0(-1.f, -2.f, -3.f, 1.f);
	TestAssert(plane0.getNormal() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(plane0.getConstant() == 1.f);
	TestEquals(3.7416575f, plane0.getLength());
	TestEquals(14.f, plane0.getSquaredLength());
}
//------------------------------------------------------------------------------
// ʉZ
void PlaneTest::testPlaneArithmetic(){
	Plane plane0(0.f, 10.f, 0.f, -10.f);
	plane0.normalize();
	TestAssert(plane0 == Plane(0.f, 1.f, 0.f, -1.f));

	TestAssert(Plane(0.f, 0.f, 0.f, 1.f).isZero());
	TestAssert(Plane(1.f, 0.f, 0.f, 1.f).isUnit());
	TestAssert(Plane(0.f, 1.f, 0.f, 1.f).isUnit());
	TestAssert(Plane(0.f, 0.f, 1.f, 1.f).isUnit());
}
//------------------------------------------------------------------------------
// ϊeXg
void PlaneTest::testTransform(){
	Plane plane0(1.f, 2.f, 3.f, 6.f);
	Plane plane1;

	Matrix33 mtx33;
	mtx33.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	plane1 = plane0.transform(mtx33);
	TestAssert(plane1.epsilonEquals(Plane(3.f, 2.f, -1.f, 6.f), Math::epsilon));

	Matrix34 mtx43;
	mtx43.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	plane1 = plane0.transform(mtx43);
	TestAssert(plane1.epsilonEquals(Plane(3.f, 2.f, -1.f, 2.f), Math::epsilon));

	Matrix44 mtx44;
	mtx44.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	plane1 = plane0.transform(mtx44);
	TestAssert(plane1.epsilonEquals(Plane(3.f, 2.f, -1.f, 6.f), Math::epsilon));

	// XP[
	mtx33.setScale(Vector3(1.f, 2.f, 1.f));
	mtx33.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	plane1 = plane0.scaledTransform(mtx33);
	TestAssert(plane1.epsilonEquals(Plane(3.f, 1.f, -1.f, 6.f), Math::epsilon));

	mtx43.setScale(Vector3(1.f, 2.f, 1.f));
	mtx43.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	plane1 = plane0.scaledTransform(mtx43);
	TestAssert(plane1.epsilonEquals(Plane(3.f, 1.f, -1.f, 3.f), Math::epsilon));

	mtx44.setScale(Vector3(1.f, 2.f, 1.f));
	mtx44.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	plane1 = plane0.scaledTransform(mtx44);
	TestAssert(plane1.epsilonEquals(Plane(3.f, 1.f, -1.f, 5.f), Math::epsilon));
}
//------------------------------------------------------------------------------
// _ZeXg
void PlaneTest::testLogicalOperation(){
	Plane plane0(0.f, 0.f, 0.f, 1.f),
		plane1(1.f, 1.f, 1.f, 2.f),
		plane2(0.f, 0.f, 0.f, 1.f);
	TestAssert(!(plane0 == plane1));
	TestAssert(plane0 == plane2);
	TestAssert(plane0 != plane1);
	TestAssert(!(plane0 != plane2));
	TestAssert(plane0.epsilonEquals(plane1, 1.f));
	TestAssert(plane0 != plane1);
	TestAssert(plane0.notEpsilonEquals(plane1, 0.5f));
	TestAssert(!plane0.notEpsilonEquals(plane1, 1.f));
	TestAssert(!plane0.notEpsilonEquals(plane2, 0.f));
}
//------------------------------------------------------------------------------
// eXg
void PlaneTest::testToString(){
	Plane plane0(12345678.f, 1234.5678f, 0.12345678f, 12345678.f);
	TestEquals("{ ( 12345678.00000000, 1234.56774902, 0.12345678 ) "
		"12345678.00000000 }", plane0.toString());
}
//------------------------------------------------------------------------------
