//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * R[eXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Geometry/Primitive/ConeTest.h"
#include "Geometry/Primitive/Cone.h"

//------------------------------------------------------------------------------
// RXgN^
ConeTest::ConeTest(String name) : TestCase(name){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* ConeTest::suite(){
	TestSuite* suite = new TestSuite("ConeTest");
	suite->addTest(new TestCaller(ConeTest, testConstructor));
	suite->addTest(new TestCaller(ConeTest, testFixedNumber));
	suite->addTest(new TestCaller(ConeTest, testSetValue));
	suite->addTest(new TestCaller(ConeTest, testGetValue));
	suite->addTest(new TestCaller(ConeTest, testConeArithmetic));
	suite->addTest(new TestCaller(ConeTest, testTransform));
	suite->addTest(new TestCaller(ConeTest, testLogicalOperation));
	suite->addTest(new TestCaller(ConeTest, testToString));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void ConeTest::testConstructor(){
	TestEquals(28, (int)sizeof(Cone));
	Cone cone0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 1.f);
	Cone cone1(Vector3(-1.f, -2.f, -3.f), Vector3(1.f, 2.f, 3.f), 1.f);
	TestAssert(cone0 == cone1);
	Cone cone2(cone1);
	TestAssert(cone0 == cone2);
	cone1 = cone2;
	TestAssert(cone0 == cone1);
	Cone cone3(Cone::zero);
	TestAssert(cone3 == Cone(0.f, 0.f, 0.f, 0.f, 0.f, 0.f, 0.f));
	float array[] = { -1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 1.f};
	Cone cone4(array);
	TestAssert(cone0 == cone4);
}
//------------------------------------------------------------------------------
// 萔eXg
void ConeTest::testFixedNumber(){
	Cone cone0(0.f, 0.f, 0.f, 0.f, 0.f, 0.f, 0.f);
	TestAssert(cone0 == Cone::zero);
}
//------------------------------------------------------------------------------
// l̐ݒeXg
void ConeTest::testSetValue(){
	Cone cone0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 1.f);
	Cone cone1;
	cone1.set(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 1.f);
	TestAssert(cone0 == cone1);
	cone1.set(Vector3(-1.f, -2.f, -3.f), Vector3(1.f, 2.f, 3.f), 1.f);
	TestAssert(cone0 == cone1);
	float array[] = { -1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 1.f };
	cone1.set(array);
	TestAssert(cone0 == cone1);
	cone1.setTheta(1.f);
	TestAssert(cone0 == cone1);
	cone1.setAngle(2.f);
	TestAssert(cone0 == cone1);
	cone1.setPositions(
		Vector3(-1.f, -2.f, -3.f), Vector3(0.f, 0.f, 0.f));
	TestAssert(cone0 == cone1);
}
//------------------------------------------------------------------------------
// l̎擾eXg
void ConeTest::testGetValue(){
	Cone cone0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 1.f);
	TestAssert(cone0.getOrigin() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(cone0.getDirection() == Vector3(1.f, 2.f, 3.f));
	TestEquals(1.f, cone0.getTheta());
	TestEquals(2.f, cone0.getAngle());
	TestAssert(cone0.getSourcePosition() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(cone0.getTargetPosition() == Vector3(0.f, 0.f, 0.f));
}
//------------------------------------------------------------------------------
// R[Z
void ConeTest::testConeArithmetic(){
	TestAssert(Cone(0.f, 0.f, 0.f, 0.f, 0.f, 0.f, 0.f).isZero());
	TestAssert(Cone(1.f, 2.f, 3.f, 0.f, 0.f, 0.f, 1.f).isZero());
}
//------------------------------------------------------------------------------
// ϊeXg
void ConeTest::testTransform(){
	Cone cone0(1.f, 2.f, 3.f, 1.f, 2.f, 3.f, 1.f);
	Cone cone1(Cone::zero);

	Matrix33 mtx33;
	mtx33.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	cone1 = cone0.transform(mtx33);
	TestAssert(cone1.epsilonEquals(
		Cone(3.f, 2.f, -1.f, 3.f, 2.f, -1.f, 1.f), Math::epsilon));

	Matrix34 mtx43;
	mtx43.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	cone1 = cone0.transform(mtx43);
	TestAssert(cone1.epsilonEquals(
		Cone(4.f, 3.f, 0.f, 3.f, 2.f, -1.f, 1.f), Math::epsilon));

	Matrix44 mtx44;
	mtx44.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	cone1 = cone0.transform(mtx44);
	TestAssert(cone1.epsilonEquals(
		Cone(4.f, 1.f, 0.f, 3.f, 2.f, -1.f, 1.f), Math::epsilon));
}
//------------------------------------------------------------------------------
// _ZeXg
void ConeTest::testLogicalOperation(){
	Cone cone0(0.f, 0.f, 0.f, 1.f, 1.f, 1.f, 0.f),
		cone1(1.f, 1.f, 1.f, 2.f, 2.f, 2.f, 1.f),
		cone2(0.f, 0.f, 0.f, 1.f, 1.f, 1.f, 0.f);
	TestAssert(!(cone0 == cone1));
	TestAssert(cone0 == cone2);
	TestAssert(cone0 != cone1);
	TestAssert(!(cone0 != cone2));
	TestAssert(cone0.epsilonEquals(cone1, 1.f));
	TestAssert(cone0 != cone1);
	TestAssert(cone0.notEpsilonEquals(cone1, 0.5f));
	TestAssert(!cone0.notEpsilonEquals(cone1, 1.f));
	TestAssert(!cone0.notEpsilonEquals(cone2, 0.f));
}
//------------------------------------------------------------------------------
// eXg
void ConeTest::testToString(){
	Cone cone0(12345678.f, 1234.5678f, 0.12345678f,
		12345678.f, 1234.5678f, 0.12345678f, 0.54321f);
	TestEquals("{ ( 12345678.00000000, 1234.56774902, 0.12345678 ) "
		"( 12345678.00000000, 1234.56774902, 0.12345678 ) 0.54321003 }",
		cone0.toString());
}
//------------------------------------------------------------------------------
