//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * JvZeXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Geometry/Primitive/CapsuleTest.h"
#include "Geometry/Primitive/Capsule.h"

//------------------------------------------------------------------------------
// RXgN^
CapsuleTest::CapsuleTest(String name) : TestCase(name){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* CapsuleTest::suite(){
	TestSuite* suite = new TestSuite("CapsuleTest");
	suite->addTest(new TestCaller(CapsuleTest, testConstructor));
	suite->addTest(new TestCaller(CapsuleTest, testFixedNumber));
	suite->addTest(new TestCaller(CapsuleTest, testSetValue));
	suite->addTest(new TestCaller(CapsuleTest, testGetValue));
	suite->addTest(new TestCaller(CapsuleTest, testCapsuleArithmetic));
	suite->addTest(new TestCaller(CapsuleTest, testTransform));
	suite->addTest(new TestCaller(CapsuleTest, testLogicalOperation));
	suite->addTest(new TestCaller(CapsuleTest, testToString));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void CapsuleTest::testConstructor(){
	TestEquals(28, (int)sizeof(Capsule));
	Capsule capsule0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 4.f);
	Capsule capsule1(Vector3(-1.f, -2.f, -3.f), Vector3(1.f, 2.f, 3.f), 4.f);
	TestAssert(capsule0 == capsule1);
	Capsule capsule2(capsule1);
	TestAssert(capsule0 == capsule2);
	capsule1 = capsule2;
	TestAssert(capsule0 == capsule1);
	Capsule capsule3(Capsule::zero);
	TestAssert(capsule3 == Capsule(0.f, 0.f, 0.f, 0.f, 0.f, 0.f, 0.f));
	float array[] = { -1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 4.f};
	Capsule capsule4(array);
	TestAssert(capsule0 == capsule4);
}
//------------------------------------------------------------------------------
// 萔eXg
void CapsuleTest::testFixedNumber(){
	Capsule capsule0(0.f, 0.f, 0.f, 0.f, 0.f, 0.f, 0.f);
	TestAssert(capsule0 == Capsule::zero);
}
//------------------------------------------------------------------------------
// l̐ݒeXg
void CapsuleTest::testSetValue(){
	Capsule capsule0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 4.f);
	Capsule capsule1;
	capsule1.set(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 4.f);
	TestAssert(capsule0 == capsule1);
	capsule1.set(Vector3(-1.f, -2.f, -3.f), Vector3(1.f, 2.f, 3.f), 4.f);
	TestAssert(capsule0 == capsule1);
	float array[] = { -1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 4.f };
	capsule1.set(array);
	TestAssert(capsule0 == capsule1);
	capsule1.setRadius(4.f);
	TestAssert(capsule0 == capsule1);
	capsule1.setPositions(
		Vector3(-1.f, -2.f, -3.f), Vector3(0.f, 0.f, 0.f));
	TestAssert(capsule0 == capsule1);
}
//------------------------------------------------------------------------------
// l̎擾eXg
void CapsuleTest::testGetValue(){
	Capsule capsule0(-1.f, -2.f, -3.f, 1.f, 2.f, 3.f, 4.f);
	TestAssert(capsule0.getOrigin() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(capsule0.getDirection() == Vector3(1.f, 2.f, 3.f));
	TestEquals(4.f, capsule0.getRadius());
	TestAssert(capsule0.getSourcePosition() == Vector3(-1.f, -2.f, -3.f));
	TestAssert(capsule0.getTargetPosition() == Vector3(0.f, 0.f, 0.f));
}
//------------------------------------------------------------------------------
// JvZZ
void CapsuleTest::testCapsuleArithmetic(){
	TestAssert(Capsule(0.f, 0.f, 0.f, 0.f, 0.f, 0.f, 0.f).isZero());
	TestAssert(Capsule(1.f, 2.f, 3.f, 0.f, 0.f, 0.f, 0.f).isZero());
}
//------------------------------------------------------------------------------
// ϊeXg
void CapsuleTest::testTransform(){
	Capsule capsule0(1.f, 2.f, 3.f, 1.f, 2.f, 3.f, 4.f);
	Capsule capsule1(Capsule::zero);

	Matrix33 mtx33;
	mtx33.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	capsule1 = capsule0.transform(mtx33);
	TestAssert(capsule1.epsilonEquals(
		Capsule(3.f, 2.f, -1.f, 3.f, 2.f, -1.f, 4.f), Math::epsilon));

	Matrix34 mtx43;
	mtx43.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	capsule1 = capsule0.transform(mtx43);
	TestAssert(capsule1.epsilonEquals(
		Capsule(4.f, 3.f, 0.f, 3.f, 2.f, -1.f, 4.f), Math::epsilon));

	Matrix44 mtx44;
	mtx44.setRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	capsule1 = capsule0.transform(mtx44);
	TestAssert(capsule1.epsilonEquals(
		Capsule(4.f, 1.f, 0.f, 3.f, 2.f, -1.f, 4.f), Math::epsilon));

	// XP[
	mtx33.setScale(Vector3(1.f, 2.f, 1.f));
	mtx33.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	capsule1 = capsule0.scaledTransform(mtx33);
	TestAssert(capsule1.epsilonEquals(
		Capsule(3.f, 4.f, -1.f, 3.f, 4.f, -1.f, 8.f), Math::epsilon));

	mtx43.setScale(Vector3(1.f, 2.f, 1.f));
	mtx43.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx43.addTranslation(1.f, 1.f, 1.f);
	capsule1 = capsule0.scaledTransform(mtx43);
	TestAssert(capsule1.epsilonEquals(
		Capsule(4.f, 5.f, 0.f, 3.f, 4.f, -1.f, 8.f), Math::epsilon));

	mtx44.setScale(Vector3(1.f, 2.f, 1.f));
	mtx44.addRotationXYZ(Vector3(Math::halfPI, Math::halfPI, Math::halfPI));
	mtx44.addTranslation(1.f, -1.f, 1.f);
	capsule1 = capsule0.scaledTransform(mtx44);
	TestAssert(capsule1.epsilonEquals(
		Capsule(4.f, 3.f, 0.f, 3.f, 4.f, -1.f, 8.f), Math::epsilon));
}
//------------------------------------------------------------------------------
// _ZeXg
void CapsuleTest::testLogicalOperation(){
	Capsule capsule0(0.f, 0.f, 0.f, 1.f, 1.f, 1.f, 2.f),
		capsule1(1.f, 1.f, 1.f, 2.f, 2.f, 2.f, 3.f),
		capsule2(0.f, 0.f, 0.f, 1.f, 1.f, 1.f, 2.f);
	TestAssert(!(capsule0 == capsule1));
	TestAssert(capsule0 == capsule2);
	TestAssert(capsule0 != capsule1);
	TestAssert(!(capsule0 != capsule2));
	TestAssert(capsule0.epsilonEquals(capsule1, 1.f));
	TestAssert(capsule0 != capsule1);
	TestAssert(capsule0.notEpsilonEquals(capsule1, 0.5f));
	TestAssert(!capsule0.notEpsilonEquals(capsule1, 1.f));
	TestAssert(!capsule0.notEpsilonEquals(capsule2, 0.f));
}
//------------------------------------------------------------------------------
// eXg
void CapsuleTest::testToString(){
	Capsule capsule0(12345678.f, 1234.5678f, 0.12345678f,
		12345678.f, 1234.5678f, 0.12345678f, 12345678.f);
	TestEquals("{ ( 12345678.00000000, 1234.56774902, 0.12345678 ) "
		"( 12345678.00000000, 1234.56774902, 0.12345678 ) 12345678.00000000 }",
		capsule0.toString());
}
//------------------------------------------------------------------------------
