//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * eXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Core/Primitive/StringTest.h"
#include "Core/Primitive/String.h"

//------------------------------------------------------------------------------
// RXgN^
StringTest::StringTest(String name) : TestCase(name), aiu_("aiuaiu\n"){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* StringTest::suite(){
	TestSuite* suite = new TestSuite("StringTest");
	suite->addTest(new TestCaller(StringTest, testConstructor));
	suite->addTest(new TestCaller(StringTest, testParse));
	suite->addTest(new TestCaller(StringTest, testAppend));
	suite->addTest(new TestCaller(StringTest, testFormat));
	suite->addTest(new TestCaller(StringTest, testSubstring));
	suite->addTest(new TestCaller(StringTest, testUpperLower));
	suite->addTest(new TestCaller(StringTest, testCompare));
	suite->addTest(new TestCaller(StringTest, testCompareIgnoreCase));
	suite->addTest(new TestCaller(StringTest, testStartsWithEndsWith));
	suite->addTest(new TestCaller(StringTest, testIndex));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void StringTest::testConstructor(){
	// NULL
	TestEquals(0, null_.getSize());
	TestEquals("", null_);
	TestAssert(null_.isEmpty());
	String constructorNull(null_);
	TestEquals(0, constructorNull.getSize());
	TestEquals(0, constructorNull.getCharacterCount());
	TestEquals("", constructorNull);
	String copyNull;
	copyNull = null_;
	TestEquals(0, copyNull.getSize());
	TestEquals(0, copyNull.getCharacterCount());
	TestEquals("", copyNull);
	String equalNull;
	equalNull = "";
	TestEquals(0, equalNull.getSize());
	TestEquals(0, equalNull.getCharacterCount());
	TestEquals("", equalNull);
	// 
	TestAssert(!aiu_.isEmpty());
	TestEquals(13, aiu_.getSize());
	TestEquals(10, aiu_.getCharacterCount());
	TestEquals("aiuaiu\n", aiu_);
	TestEquals('a', aiu_.charAt(0));
	TestEquals('u', aiu_.charAt(11));
	String constructorAiu(aiu_);
	TestEquals(13, constructorAiu.getSize());
	TestEquals(10, constructorAiu.getCharacterCount());
	TestEquals("aiuaiu\n", constructorAiu);
	String copyAiu;
	copyAiu = aiu_;
	TestEquals(13, copyAiu.getSize());
	TestEquals(10, copyAiu.getCharacterCount());
	TestEquals("aiuaiu\n", copyAiu);
	String equalAiu;
	equalAiu = "aiuaiu\n";
	TestEquals(13, equalAiu.getSize());
	TestEquals(10, equalAiu.getCharacterCount());
	TestEquals("aiuaiu\n", equalAiu);
}
//------------------------------------------------------------------------------
// p[XeXg
void StringTest::testParse(){
	String target;
	// chareXg
	char charVal;
	target = "";
	TestAssert(!target.parseChar(&charVal));
	target = "127";
	TestAssert(target.parseChar(&charVal));
	TestEquals(127, charVal);
	target = "-128";
	TestAssert(target.parseChar(&charVal));
	TestEquals(-128, charVal);
	target = "128";
	TestAssert(!target.parseChar(&charVal));
	target = "-129";
	TestAssert(!target.parseChar(&charVal));
	// u_chareXg
	u_char uCharVal;
	target = "";
	TestAssert(!target.parseUChar(&uCharVal));
	target = "255";
	TestAssert(target.parseUChar(&uCharVal));
	TestEquals(255, uCharVal);
	target = "0";
	TestAssert(target.parseUChar(&uCharVal));
	TestEquals(0, uCharVal);
	target = "256";
	TestAssert(!target.parseUChar(&uCharVal));
	target = "-1";
	TestAssert(!target.parseUChar(&uCharVal));
	// shorteXg
	short shortVal;
	target = "";
	TestAssert(!target.parseShort(&shortVal));
	target = "32767";
	TestAssert(target.parseShort(&shortVal));
	TestEquals(32767, shortVal);
	target = "-32768";
	TestAssert(target.parseShort(&shortVal));
	TestEquals(-32768, shortVal);
	target = "32768";
	TestAssert(!target.parseShort(&shortVal));
	target = "-32769";
	TestAssert(!target.parseShort(&shortVal));
	// u_shorteXg
	u_short uShortVal;
	target = "";
	TestAssert(!target.parseUShort(&uShortVal));
	target = "65535";
	TestAssert(target.parseUShort(&uShortVal));
	TestEquals(65535, uShortVal);
	target = "0";
	TestAssert(target.parseUShort(&uShortVal));
	TestEquals(0, uShortVal);
	target = "65536";
	TestAssert(!target.parseUShort(&uShortVal));
	target = "-1";
	TestAssert(!target.parseUShort(&uShortVal));
	// inteXg
	int intVal;
	target = "";
	TestAssert(!target.parseInt(&intVal));
	target = "-2147483648";
	TestAssert(target.parseInt(&intVal));
	TestEquals(-2147483647 - 1, intVal);
	target = "2147483647";
	TestAssert(target.parseInt(&intVal));
	TestEquals(2147483647, intVal);
	target = "4294967296";
	TestAssert(target.parseInt(&intVal));
	TestEquals(0, intVal);
	// u_inteXg
	u_int uIntVal;
	target = "";
	TestAssert(!target.parseUInt(&uIntVal));
	target = "0";
	TestAssert(target.parseUInt(&uIntVal));
	TestEquals(0u, uIntVal);
	target = "4294967295";
	TestAssert(target.parseUInt(&uIntVal));
	TestEquals(4294967295, uIntVal);
	target = "-1";
	TestAssert(target.parseUInt(&uIntVal));
	TestEquals(4294967295, uIntVal);
	target = "-2147483648";
	TestAssert(target.parseUInt(&uIntVal));
	TestEquals(2147483648, uIntVal);
	// floateXg
	float floatVal;
	target = "";
	TestAssert(!target.parseFloat(&floatVal));
	target = "0";
	TestAssert(target.parseFloat(&floatVal));
	TestEquals(0.f, floatVal);
	target = "-1e-5";
	TestAssert(target.parseFloat(&floatVal));
	TestEquals(-0.00001f, floatVal);
	target = "100000.f";
	TestAssert(target.parseFloat(&floatVal));
	TestEquals(100000.f, floatVal);
	// doubleeXg
	double doubleVal;
	target = "";
	TestAssert(!target.parseDouble(&doubleVal));
	target = "0";
	TestAssert(target.parseDouble(&doubleVal));
	TestEquals(0., doubleVal);
	target = "-1e-15";
	TestAssert(target.parseDouble(&doubleVal));
	TestEquals(-0.000000000000001, doubleVal);
	target = "1000000000000000.";
	TestAssert(target.parseDouble(&doubleVal));
	TestEquals(1000000000000000., doubleVal);
}
//------------------------------------------------------------------------------
// AeXg
void StringTest::testAppend(){
	String pre, post;
	// 󕶎
	pre = post = "";
	pre.append(post);
	TestEquals(0, pre.getSize());
	TestEquals("", pre);
	// pre󕶎
	post = "a\n";
	pre.append(post);
	TestEquals(4, pre.getSize());
	TestEquals("a\n", pre);
	// ʏ핶
	pre.append(post);
	TestEquals(8, pre.getSize());
	TestEquals("a\na\n", pre);
	// post󕶎
	post = "";
	pre.append(post);
	TestEquals(8, pre.getSize());
	TestEquals("a\na\n", pre);

	// 󕶎ւ̋󕶎񑫂
	String addString;
	addString.append("");
	TestEquals(0, addString.getSize());
	TestEquals("", addString);
	// 󕶎ւ̑
	addString.append("a\n");
	TestEquals(4, addString.getSize());
	TestEquals("a\n", addString);
	// ʏ̑
	addString.append("a\n");
	TestEquals(8, addString.getSize());
	TestEquals(6, addString.getCharacterCount());
	TestEquals("a\na\n", addString);

	// ǉZq
	String opeString;
	opeString += "a";
	opeString += "\t";
	opeString += String("c");
	opeString += "";
	opeString += String("");
	TestEquals(7, opeString.getSize());
	TestEquals("a\tc", opeString);
	opeString = "b" + String("a") + "" + "\t";
	TestEquals("ba\t", opeString);
	opeString = opeString + "";
	TestEquals("ba\t", opeString);
	opeString = "" + opeString;
	TestEquals("ba\t", opeString);

}
//------------------------------------------------------------------------------
// tH[}bgeXg
void StringTest::testFormat(){
	String format;
	TestEquals(0, format.getSize());
	TestEquals("", format);
	// 󕶎
	format.format("");
	TestEquals("", format);
	// ʏ핶
	format.format("12345678");
	TestEquals("12345678", format);
	// tH[}bg
	format.format("%d %.2f %s",
		5, 1.f / 3.f , aiu_.getBytes());
	TestEquals(17, format.getCharacterCount());
	TestEquals("5 0.33 aiuaiu\n", format);
	format.format("%s", "01234567890123456789");
	TestEquals("01234567890123456789", format);
	format.format("%s", "");
	TestEquals("", format);
}
//------------------------------------------------------------------------------
// eXg
void StringTest::testSubstring(){
	// JnCfbNX
	String source("0123456789");
	source = source.getSubstring(0);
	TestEquals(10, source.getSize());
	TestEquals("0123456789", source);
	source = source.getSubstring(5);
	TestEquals(5, source.getSize());
	TestEquals("56789", source);
	source = source.getSubstring(5);
	TestEquals(0, source.getSize());
	TestEquals("", source);
	source = source.getSubstring(0);
	TestEquals(0, source.getSize());
	TestEquals("", source);
	// JnAICfbNX
	source = "0123456789";
	source = source.getSubstring(0, 10);
	TestEquals(10, source.getSize());
	TestEquals("0123456789", source);
	source = source.getSubstring(0, 9);
	TestEquals(9, source.getSize());
	TestEquals("012345678", source);
	source = source.getSubstring(0, 1);
	TestEquals(1, source.getSize());
	TestEquals("0", source);
	source = source.getSubstring(1, 1);
	TestEquals(0, source.getSize());
	TestEquals("", source);
	source = "0123456789";
	source = source.getSubstring(9, 10);
	TestEquals(1, source.getSize());
	TestEquals("9", source);
}
//------------------------------------------------------------------------------
// 啶eXg
void StringTest::testUpperLower(){
	String source(""), destination;
	source = source.getUpperCase();
	TestEquals(0, source.getSize());
	TestEquals("", source);
	source = source.getLowerCase();
	TestEquals(0, source.getSize());
	TestEquals("", source);
	source = "0";
	source = source.getUpperCase();
	TestEquals(1, source.getSize());
	TestEquals("0", source);
	source = source.getLowerCase();
	TestEquals(1, source.getSize());
	TestEquals("0", source);
	source = "aBcDe";
	source = source.getUpperCase();
	TestEquals(5, source.getSize());
	TestEquals("ABCDE", source);
	source = "aBcDe";
	source = source.getLowerCase();
	TestEquals(5, source.getSize());
	TestEquals("abcde", source);
}
//------------------------------------------------------------------------------
// reXg
void StringTest::testCompare(){
	// reXg
	String a("\ta"), b("\tb");
	int comp;
	comp = a.compareTo(b);
	TestEquals(-1, comp);
	comp = a.compareTo("\tb");
	TestEquals(-1, comp);
	comp = b.compareTo(a);
	TestEquals(1, comp);
	comp = b.compareTo("\ta");
	TestEquals(1, comp);
	comp = a.compareTo(a);
	TestEquals(0, comp);
	comp = a.compareTo("\ta");
	TestEquals(0, comp);
	// reXg
	TestAssert(!a.equals(b));
	TestAssert(!a.equals("\tb"));
	b = "\ta";
	TestAssert(a.equals(b));
	TestAssert(a.equals("\ta"));
}
//------------------------------------------------------------------------------
// 啶𖳎reXg
void StringTest::testCompareIgnoreCase(){
	// reXg
	String source("aBcDe0(\t"), target("AbCdE0(\t");
	TestEquals(-1, source.compareToIgnoreCase(target));
	TestEquals(-1, source.compareToIgnoreCase("AbCdE0(\t"));
	TestEquals(1, target.compareToIgnoreCase(source));
	TestEquals(1, target.compareToIgnoreCase("aBcDe0(\t"));
	target = "AbCdE0(\t";
	TestEquals(0, target.compareToIgnoreCase(source));
	TestEquals(0, target.compareToIgnoreCase("aBcDe0(\t"));
	// reXg
	TestAssert(source.equalsIsIgnoreCase(target));
	TestAssert(source.equalsIsIgnoreCase("aBcDe0(\t"));
	target = "AbCdE0(\t";
	TestAssert(!source.equalsIsIgnoreCase(target));
	TestAssert(!source.equalsIsIgnoreCase("AbCdE0(\t"));
}
//------------------------------------------------------------------------------
// JnvAIveXg
void StringTest::testStartsWithEndsWith(){
	// JneXg
	String target("ab"), compare("abc");
	TestAssert(!target.startsWith(compare));
	TestAssert(!target.startsWith("abc"));
	target = "abcd";
	TestAssert(target.startsWith(compare));
	TestAssert(target.startsWith("abc"));
	target = "acbd";
	TestAssert(!target.startsWith(compare));
	TestAssert(!target.startsWith("abc"));
	target = "";
	TestAssert(!target.startsWith(compare));
	TestAssert(!target.startsWith("abc"));
	compare = "";
	TestAssert(target.startsWith(compare));
	TestAssert(target.startsWith(""));
	// IeXg
	compare = "abc";
	target = "bc";
	TestAssert(!target.endsWith(compare));
	TestAssert(!target.endsWith("abc"));
	target = "dabc";
	TestAssert(target.endsWith(compare));
	TestAssert(target.endsWith("abc"));
	target = "abcd";
	TestAssert(!target.endsWith(compare));
	TestAssert(!target.endsWith("abc"));
	target = "";
	TestAssert(!target.endsWith(compare));
	TestAssert(!target.endsWith("abc"));
	compare = "";
	TestAssert(target.endsWith(compare));
	TestAssert(target.endsWith(""));
}
//------------------------------------------------------------------------------
// CfbNXeXg
void StringTest::testIndex(){
	String source("01234567890123456789"),
		target012("012"), target789("789"), targetAiu(""), targetNull("");
	// ŏ
	TestEquals(0, source.getIndexOf('0'));
	TestEquals(9, source.getIndexOf('9'));
	TestEquals(-1, source.getIndexOf('a'));
	TestEquals(0, source.getIndexOf("012"));
	TestEquals(7, source.getIndexOf("789"));
	TestEquals(-1, source.getIndexOf(""));
	TestEquals(0, source.getIndexOf(""));
	TestEquals(0, source.getIndexOf(target012));
	TestEquals(7, source.getIndexOf(target789));
	TestEquals(-1, source.getIndexOf(targetAiu));
	TestEquals(0, source.getIndexOf(targetNull));
	// Ō
	TestEquals(10, source.getLastIndexOf('0'));
	TestEquals(19, source.getLastIndexOf('9'));
	TestEquals(-1, source.getLastIndexOf('a'));
	TestEquals(10, source.getLastIndexOf("012"));
	TestEquals(17, source.getLastIndexOf("789"));
	TestEquals(-1, source.getLastIndexOf(""));
	TestEquals(20, source.getLastIndexOf(""));
	TestEquals(10, source.getLastIndexOf(target012));
	TestEquals(17, source.getLastIndexOf(target789));
	TestEquals(-1, source.getLastIndexOf(targetAiu));
	TestEquals(20, source.getLastIndexOf(targetNull));
}
//------------------------------------------------------------------------------
