//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * OvfJ[eXg
 * @author Junpee
 */

#include "Test/stdafx.h"
#include "Core/Primitive/Color3fTest.h"
#include "Core/Primitive/Color3f.h"
#include "Core/Primitive/Color3c.h"
#include "Core/Primitive/Color4c.h"
#include "Core/Primitive/Color4f.h"

//------------------------------------------------------------------------------
// RXgN^
Color3fTest::Color3fTest(String name) : TestCase(name){
}
//------------------------------------------------------------------------------
// eXgXB[g̎擾
Test* Color3fTest::suite(){
	TestSuite* suite = new TestSuite("Color3fTest");
	suite->addTest(new TestCaller(Color3fTest, testConstructor));
	suite->addTest(new TestCaller(Color3fTest, testFixedNumber));
	suite->addTest(new TestCaller(Color3fTest, testSetValue));
	suite->addTest(new TestCaller(Color3fTest, testArithmetic));
	suite->addTest(new TestCaller(Color3fTest, testSubstitutionArithmetic));
	suite->addTest(new TestCaller(Color3fTest, testColorArithmetic));
	suite->addTest(new TestCaller(Color3fTest, testHSV));
	suite->addTest(new TestCaller(Color3fTest, testLogicalOperation));
	suite->addTest(new TestCaller(Color3fTest, testToString));
	return suite;
}
//------------------------------------------------------------------------------
// RXgN^eXg
void Color3fTest::testConstructor(){
	TestEquals(12, (int)sizeof(Color3f));
	Color3f col0(0.5f, 1.f, 2.f);
	TestEquals(0.5f, col0.r);
	TestEquals(1.f, col0.g);
	TestEquals(2.f, col0.b);
	TestEquals(0.5f, col0.array[0]);
	TestEquals(1.f, col0.array[1]);
	TestEquals(2.f, col0.array[2]);
	Color3f col1(0.f, 1.f, 2.f);
	TestEquals(0.f, col1.r);
	TestEquals(1.f, col1.g);
	TestEquals(2.f, col1.b);
	col1 = col0;
	TestEquals(0.5f, col1.r);
	TestEquals(1.f, col1.g);
	TestEquals(2.f, col1.b);
	Color3f col2(Color3c(64, 128, 192));
	TestAssert(Color3f(64.f / 255.f, 128.f / 255.f, 192.f / 255.f).
		epsilonEquals(col2, Math::epsilon));
	Color3f col3(Color4c(64, 128, 192, 64));
	TestAssert(Color3f(64.f / 255.f, 128.f / 255.f, 192.f / 255.f).
		epsilonEquals(col3, Math::epsilon));
	Color3f col4(Color4f(0.5f, 1.f, 2.f, 3.f));
	TestAssert(Color3f(0.5f, 1.f, 2.f) == col4);
}
//------------------------------------------------------------------------------
// 萔eXg
void Color3fTest::testFixedNumber(){
	TestAssert(Color3f(1.f, 1.f, 1.f) == Color3f::white);
	TestAssert(Color3f(0.5f, 0.5f, 0.5f) == Color3f::gray);
	TestAssert(Color3f(0.f, 0.f, 0.f) == Color3f::black);
	TestAssert(Color3f(1.f, 0.f, 0.f) == Color3f::red);
	TestAssert(Color3f(0.f, 1.f, 0.f) == Color3f::green);
	TestAssert(Color3f(0.f, 0.f, 1.f) == Color3f::blue);
	TestAssert(Color3f(1.f, 1.f, 0.f) == Color3f::yellow);
	TestAssert(Color3f(0.f, 1.f, 1.f) == Color3f::cyan);
	TestAssert(Color3f(1.f, 0.f, 1.f) == Color3f::magenta);
}
//------------------------------------------------------------------------------
// l̐ݒeXg
void Color3fTest::testSetValue(){
	Color3f col0;
	col0.set(1.f, 1.f, 1.f);
	TestAssert(Color3f::white == col0);
	col0.set(1.f, 1.f, 1.f - Math::epsilon);
	TestAssert(Color3f::white != col0);
	// OvfLN^J[̐ݒ
	col0.set(Color3c(64, 128, 192));
	TestAssert(Color3f(64.f / 255.f, 128.f / 255.f, 192.f / 255.f).
		epsilonEquals(col0, Math::epsilon));
	// lvfLN^J[̐ݒ
	col0.set(Color4c(64, 128, 192, 64));
	TestAssert(Color3f(64.f / 255.f, 128.f / 255.f, 192.f / 255.f).
		epsilonEquals(col0, Math::epsilon));
	// lvfJ[̐ݒ
	col0.set(Color4f(0.5f, 1.f, 2.f, 3.f));
	TestAssert(Color3f(0.5f, 1.f, 2.f) == col0);
}
//------------------------------------------------------------------------------
// Z
void Color3fTest::testArithmetic(){
	// Z
	Color3f col0(0.25f, 0.25f, 0.25f), col1(0.25f, 0.5f, 1.f), col2;
	col2 = col0 + col1;
	TestAssert(Color3f(0.5f, 0.75f, 1.25f) == col2);
	// Z
	col2 = col2 - col1;
	TestAssert(col0 == col2);
	// Z
	col0.set(0.5f, 0.5f, 0.5f);
	col0 = col0 * col0;
	TestAssert(Color3f(0.25f, 0.25f, 0.25f) == col0);
	col0 = col0 * 2.f;
	TestAssert(Color3f(0.5f, 0.5f, 0.5f) == col0);
	col0 = 0.5f * col0;
	TestAssert(Color3f(0.25f, 0.25f, 0.25f) == col0);
	// +-Zq
	TestAssert(col0 == +col0);
	TestAssert(Color3f(-0.25f, -0.25f, -0.25f) == -col0);
}
//------------------------------------------------------------------------------
// Z
void Color3fTest::testSubstitutionArithmetic(){
	// Z
	Color3f col0(0.25f, 0.25f, 0.25f), col1(0.25f, 0.5f, 1.f), col2;
	col2 = col0;
	col0 += col1;
	TestAssert(Color3f(0.5f, 0.75f, 1.25f) == col0);
	// Z
	col0 -= col1;
	TestAssert(col2 == col0);
	// Z
	col0.set(0.5f, 0.5f, 0.5f);
	col0 *= col0;
	TestAssert(Color3f(0.25f, 0.25f, 0.25f) == col0);
	col0 *= 2.f;
	TestAssert(Color3f(0.5f, 0.5f, 0.5f) == col0);
}
//------------------------------------------------------------------------------
// FZ
void Color3fTest::testColorArithmetic(){
	// Nv
	Color3f clamp(-1.f, 2.f, 0.8f);
	clamp.clamp();
	TestAssert(Color3f(0.f, 1.f, 0.8f) == clamp);
	clamp.upperClamp(0.6f);
	TestAssert(Color3f(0.f, 0.6f, 0.6f) == clamp);
	clamp.lowerClamp(0.4f);
	TestAssert(Color3f(0.4f, 0.6f, 0.6f) == clamp);
	// ΐF
	Color3f negColor(0.25f, 0.5f, 0.75f);
	negColor.negative();
	TestAssert(Color3f(0.75f, 0.5f, 0.25f) == negColor);
	// `
	Color3f col0(0, 0, 0), col1(1.f, 1.f, 1.f);
	for(int i = 0; i < 11; i++){
		float alpha = i * 0.1f;
		Color3f answer(alpha, alpha, alpha);
		Color3f result = Color3f::lerp(col0, col1, alpha);
		TestAssert(answer == result);
	}
}
//------------------------------------------------------------------------------
// HSVeXg
void Color3fTest::testHSV(){
	Color3f col0;
	col0 = Color3f::white.getHSV();
	TestAssert(col0 == Color3f(0.f, 0.f, 1.f));
	col0.setHSV(col0);
	TestAssert(col0 == Color3f::white);

	col0 = Color3f::gray.getHSV();
	TestAssert(col0 == Color3f(0.f, 0.f, 0.5f));
	col0.setHSV(col0);
	TestAssert(col0 == Color3f::gray);

	col0 = Color3f::black.getHSV();
	TestAssert(col0 == Color3f(0.f, 0.f, 0.f));
	col0.setHSV(col0);
	TestAssert(col0 == Color3f::black);

	col0 = Color3f::red.getHSV();
	TestAssert(col0 == Color3f(0.f, 1.f, 1.f));
	col0.setHSV(col0);
	TestAssert(col0 == Color3f::red);

	col0 = Color3f::green.getHSV();
	TestAssert(col0.epsilonEquals(Color3f(0.333333f, 1.f, 1.f), Math::epsilon));
	col0.setHSV(col0);
	TestAssert(col0.epsilonEquals(Color3f::green, Math::epsilon));

	col0 = Color3f::blue.getHSV();
	TestAssert(col0.epsilonEquals(Color3f(0.666666f, 1.f, 1.f), Math::epsilon));
	col0.setHSV(col0);
	TestAssert(col0.epsilonEquals(Color3f::blue, Math::epsilon));

	col0 = Color3f::yellow.getHSV();
	TestAssert(col0.epsilonEquals(Color3f(0.166666f, 1.f, 1.f), Math::epsilon));
	col0.setHSV(col0);
	TestAssert(col0.epsilonEquals(Color3f::yellow, Math::epsilon));

	col0 = Color3f::cyan.getHSV();
	TestAssert(col0.epsilonEquals(Color3f(0.5f, 1.f, 1.f), Math::epsilon));
	col0.setHSV(col0);
	TestAssert(col0.epsilonEquals(Color3f::cyan, Math::epsilon));

	col0 = Color3f::magenta.getHSV();
	TestAssert(col0 == Color3f(0.8333333f, 1.f, 1.f));
	col0.setHSV(col0);
	TestAssert(col0.epsilonEquals(Color3f::magenta, Math::epsilon));
}
//------------------------------------------------------------------------------
// _ZeXg
void Color3fTest::testLogicalOperation(){
	Color3f col0(0.5f, 1.f, 2.f), col1(0.5, 1.f, 2.f);
	TestAssert(col0 == col1);
	TestAssert(!(col0 != col1));
	col1.r += 1.f;
	TestAssert(!(col0 == col1));
	TestAssert(col0 != col1);
	TestAssert(col0.epsilonEquals(col1, 1.f));
	TestAssert(!col0.notEpsilonEquals(col1, 1.f));
	col1.r += Math::epsilon;
	TestAssert(!col0.epsilonEquals(col1, 1.f));
	TestAssert(col0.notEpsilonEquals(col1, 1.f));
}
//------------------------------------------------------------------------------
// eXg
void Color3fTest::testToString(){
	Color3f col0(0.5f, 1.f, 2.f);
	TestEquals(
		"( 0.50000000, 1.00000000, 2.00000000 )",
		col0.toString());
}
//------------------------------------------------------------------------------
