//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * TEhXg
 * @author Junpee
 */

#include "LampBasic.h"
#include "Sound/Utility/SoundList.h"
#include "Sound/Utility/SoundCache.h"
#include "Core/InputOutput/TextFileReader.h"
#include "Core/InputOutput/FilePath.h"
#include "Core/Utility/StringTokenizer.h"

namespace Lamp{

//------------------------------------------------------------------------------
// [h
bool SoundList::load(SoundCache* soundCache, const String& filePath){
	FilePath path(filePath);
	if(!path.existFile()){ return false; }
	TextFileReader* reader = new TextFileReader(filePath);
	bool result = load(soundCache, reader);
	delete reader;
	return result;
}
//------------------------------------------------------------------------------
// [h
bool SoundList::load(SoundCache* soundCache, TextReader* textReader){
	String line;
	while(!textReader->isEnd()){
		String line = textReader->readLine();
		if(!loadSound(soundCache, line)){ return false; }
	}
	return true;
}
//------------------------------------------------------------------------------
// TEh̃[h
bool SoundList::loadSound(SoundCache* soundCache, const String& line){
	String temp, errorMethod("SoundList::loadSound() ");
	StringTokenizer tokenizer(line, "\t");
	// s`FbN
	if(!tokenizer.hasMoreTokens()){ return true; }
	temp = tokenizer.getNextToken();
	// Rg`FbN
	if(temp.startsWith("//")){ return true; }

	// t@C
	String fileName;
	fileName = temp;

	// XeIA3D
	bool isStereo;
	if(!tokenizer.hasMoreTokens()){
		ErrorOut(errorMethod + "XeIA3Dw肪܂B" + line);
		return false;
	}
	temp = tokenizer.getNextToken();
	if(temp.equalsIsIgnoreCase("stereo")){
		isStereo = true;
	}else if(temp.equalsIsIgnoreCase("3d")){
		isStereo = false;
	}else{
		ErrorOut(errorMethod + "XeIA3Dwɂ"
			"uStereovu3Dvw肵ĉB" + line);
		return false;
	}

	// [v
	bool isLoop;
	if(!tokenizer.hasMoreTokens()){
		ErrorOut(errorMethod + "[vw肪܂B" + line);
		return false;
	}
	temp = tokenizer.getNextToken();
	if(temp.equalsIsIgnoreCase("loop")){
		isLoop = true;
	}else if(temp.equalsIsIgnoreCase("once")){
		isLoop = false;
	}else{
		ErrorOut(errorMethod + "[vwɂ"
			"uLoopvuOncevw肵ĉB" + line);
		return false;
	}

	// vCIeB
	int priority;
	if(!tokenizer.hasMoreTokens()){
		ErrorOut(errorMethod + "vCIeBw肪܂B" + line);
		return false;
	}
	temp = tokenizer.getNextToken();
	if((!temp.parseInt(&priority)) ||
		(priority > Limit::shortMax) || (priority < Limit::shortMin)){
		ErrorOut(errorMethod + "vCIeB32767`-32768̐"
			"w肵ĂB" + line);
		return false;
	}

	// ő~LVO
	int maxMixingCount;
	if(!tokenizer.hasMoreTokens()){
		ErrorOut(errorMethod + "ő~LVOw肪܂B" + line);
		return false;
	}
	temp = tokenizer.getNextToken();
	if((!temp.parseInt(&maxMixingCount)) || (maxMixingCount < 0)){
		ErrorOut(errorMethod + "ő~LVO0ȏ̐"
			"w肵ĂB" + line);
		return false;
	}

	// ŏ3DAő3D
	float minimum3DDistance;
	float maximum3DDistance;
	if(!isStereo){
		// ŏ3D
		if(!tokenizer.hasMoreTokens()){
			ErrorOut(errorMethod + "ŏ3Dw肪܂B" + line);
			return false;
		}
		temp = tokenizer.getNextToken();
		if((!temp.parseFloat(&minimum3DDistance)) ||
			(minimum3DDistance <= 0.f)){
			ErrorOut(errorMethod + "ŏ3D͐̋"
				"w肵ĂB" + line);
			return false;
		}

		// ő3D
		if(!tokenizer.hasMoreTokens()){
			ErrorOut(errorMethod + "ŏ3Dw肪܂B" + line);
			return false;
		}
		temp = tokenizer.getNextToken();
		if((!temp.parseFloat(&maximum3DDistance)) ||
			(maximum3DDistance <= 0.f)){
			ErrorOut(errorMethod + "ő3D͐̋"
				"w肵ĂB" + line);
			return false;
		}
		if(maximum3DDistance < minimum3DDistance){
			ErrorOut(errorMethod + "ő3D͍ŏ3D傫Ȏ"
				"w肵ĂB" + line);
			return false;
		}
	}

	// TEh̃[h
	if(isStereo){
		if(!soundCache->loadStaticSound(
			fileName, isLoop, priority, maxMixingCount)){
			ErrorOut(errorMethod + "ÓITEh"
				"[hɎs܂B" + line);
			return false;
		}
	}else{
		if(!soundCache->loadStaticSound3D(fileName, isLoop, priority,
			minimum3DDistance, maximum3DDistance, maxMixingCount)){
			ErrorOut(errorMethod + "ÓI3DTEh"
				"[hɎs܂B" + line);
			return false;
		}
	}
	return true;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
