//------------------------------------------------------------------------------
// Lamp : Open source game middleware
// Copyright (C) 2004  Junpei Ohtani ( Email : junpee@users.sourceforge.jp )
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//------------------------------------------------------------------------------

/** @file
 * TEhLbV
 * @author Junpee
 */

#include "LampBasic.h"
#include "Sound/Utility/SoundCache.h"
#include "Sound/System/LampSound.h"
#include "Sound/System/SoundManager.h"
#include "Sound/Stereo/StaticSound.h"
#include "Sound/3D/StaticSound3D.h"
#include "Sound/Utility/SoundList.h"

namespace Lamp{

//------------------------------------------------------------------------------
// Aj
//------------------------------------------------------------------------------
// RXgN^
SoundCache::SoundCache(const String& basePath, const String& extension,
	Sound::Focus focus, int defaultMaxMixingCount) :
	basePath_(basePath), extension_(extension),
	focus_(focus), defaultMaxMixingCount_(defaultMaxMixingCount){
	Assert(defaultMaxMixingCount_ > 0);
}
//------------------------------------------------------------------------------
// fXgN^
SoundCache::~SoundCache(){
	unloadAll();
}
//------------------------------------------------------------------------------
// TEh̃[h
//------------------------------------------------------------------------------
// ÓITEh̃[h
bool SoundCache::loadStaticSound(
	const String& name, bool loop, int priority, int maxMixingCount){
	// łɑ݂ĂȂ`FbN
	if(cache_.get(name) != NULL){ Assert(false); return false; }
	// TEh쐬
	StaticSound* sound = LampSound::getSoundManager()->createStaticSound(
		basePath_ + name + extension_, focus_);
	if(sound == NULL){ Assert(false); return false; }
	sound->setLoop(loop);
	sound->setPriority(priority);
	// LbV쐬
	int soundCount = maxMixingCount;
	if(soundCount == 0){ soundCount = defaultMaxMixingCount_; }
	createCache(name, sound, soundCount);
	return true;
}
//------------------------------------------------------------------------------
// ÓI3DTEh̃[h
bool SoundCache::loadStaticSound3D(const String& name, bool loop, int priority,
	float minimumDistance, float maximumDistance, int maxMixingCount){
	// łɑ݂ĂȂ`FbN
	if(cache_.get(name) != NULL){ Assert(false); return false; }
	// TEh쐬
	StaticSound3D* sound = LampSound::getSoundManager()->createStaticSound3D(
		basePath_ + name + extension_, focus_);
	if(sound == NULL){ Assert(false); return false; }
	sound->setLoop(loop);
	sound->setPriority(priority);
	sound->setDistance(minimumDistance, maximumDistance);
	// LbV쐬
	int mixingCount = maxMixingCount;
	if(mixingCount == 0){ mixingCount = defaultMaxMixingCount_; }
	createCache(name, sound, mixingCount);
	return true;
}
//------------------------------------------------------------------------------
// LbV쐬
void SoundCache::createCache(
	const String& name, Sound* sound, int maxMixingCount){
	SoundArray* soundArray = new SoundArray(sound, maxMixingCount);
	cache_.put(name, soundArray);
	list_.add(soundArray);
}
//------------------------------------------------------------------------------
// TEh̃A[h
bool SoundCache::unloadSound(const String& name){
	SoundArray* soundArray = cache_.remove(name);
	if(soundArray == NULL){ return false; }
	list_.removeByValue(soundArray);
	delete soundArray;
	return true;
}
//------------------------------------------------------------------------------
// STEh̃A[h
void SoundCache::unloadAll(){
	int count = list_.getCount();
	for(int i = count - 1; i >= 0; i--){ delete list_[i]; }
	list_.clear();
	cache_.clear();
}
//------------------------------------------------------------------------------
// TEhXg
//------------------------------------------------------------------------------
// TEhXg̃[h
bool SoundCache::loadSoundList(const String& filePath){
	return SoundList::load(this, basePath_ + filePath);
}
//------------------------------------------------------------------------------
// TEhXg̃[h
bool SoundCache::loadSoundList(TextReader* textReader){
	return SoundList::load(this, textReader);
}
//------------------------------------------------------------------------------
// TEh̍Đ
//------------------------------------------------------------------------------
// TEh̍Đ
Sound* SoundCache::playSound(const String& name, float volume, int frequency){
	// TEȟ
	SoundArray* array = cache_.get(name);
	if(array == NULL){ return NULL; }
	Sound* sound = array->getFreeSound();
	if(sound == NULL){ return NULL; }
	// TEh̍Đ
	sound->reset(Sound::resetRuntime);
	sound->setVolume(volume);
	if(frequency != 0){ sound->setFrequency(frequency); }
	sound->play();
	return sound;
}
//------------------------------------------------------------------------------
// XeITEh̍Đ
StereoSound* SoundCache::playStereoSound(
	const String& name, float volume, int frequency, float pan){
	// TEȟ
	SoundArray* array = cache_.get(name);
	if(array == NULL){ return NULL; }
	Sound* sound = array->getFreeSound();
	if(sound == NULL){ return NULL; }
	StereoSound* stereoSound = sound->castStereoSound();
	if(stereoSound == NULL){ return NULL; }
	// TEh̍Đ
	stereoSound->reset(Sound::resetRuntime);
	stereoSound->setVolume(volume);
	if(frequency != 0){ stereoSound->setFrequency(frequency); }
	stereoSound->setPan(pan);
	stereoSound->play();
	return stereoSound;
}
//------------------------------------------------------------------------------
// 3DTEh̍Đ
Sound3D* SoundCache::playSound3D(
	const String& name, float volume, int frequency, const Vector3& position){
	// TEȟ
	SoundArray* array = cache_.get(name);
	if(array == NULL){ return NULL; }
	Sound* sound = array->getFreeSound();
	if(sound == NULL){ return NULL; }
	Sound3D* sound3D = sound->castSound3D();
	if(sound3D == NULL){ return NULL; }
	// TEh̍Đ
	sound3D->reset(Sound::resetRuntime);
	sound3D->setVolume(volume);
	if(frequency != 0){ sound3D->setFrequency(frequency); }
	sound3D->setPosition(position);
	sound3D->play();
	return sound3D;
}
//------------------------------------------------------------------------------
// TEh̎擾A
//------------------------------------------------------------------------------
// TEh̎擾
Sound* SoundCache::getSound(const String& name, Sound::Reset resetFlag){
	// TEȟ
	SoundArray* array = cache_.get(name);
	if(array == NULL){ Assert(false); return NULL; }
	Sound* sound = array->getFreeSound();
	if(sound == NULL){ Assert(false); return NULL; }
	// L𖳂
	Assert(sound->hasOwnership());
	sound->setOwnership(false);
	sound->reset(resetFlag);
	return sound;
}
//------------------------------------------------------------------------------
// XeITEh̎擾
StereoSound* SoundCache::getStereoSound(
	const String& name, Sound::Reset resetFlag){
	Sound* sound = getSound(name, resetFlag);
	if(sound == NULL){ return NULL; }
	StereoSound* stereoSound = sound->castStereoSound();
	if(stereoSound == NULL){
		releaseSound(sound);
		return NULL;
	}
	return stereoSound;
}
//------------------------------------------------------------------------------
// 3DTEh̎擾
Sound3D* SoundCache::getSound3D(const String& name, Sound::Reset resetFlag){
	Sound* sound = getSound(name, resetFlag);
	if(sound == NULL){ return NULL; }
	Sound3D* sound3D = sound->castSound3D();
	if(sound3D == NULL){
		releaseSound(sound);
		return NULL;
	}
	return sound3D;
}
//------------------------------------------------------------------------------
// TEh̉
void SoundCache::releaseSound(Sound* sound){
	Assert(sound != NULL);
	// Lǂ
	Assert(!sound->hasOwnership());
	sound->setOwnership(true);
}
//------------------------------------------------------------------------------
// ̑
//------------------------------------------------------------------------------
// STEh̍Đꎞ~
void SoundCache::suspendAll(){
	int listCount = list_.getCount();
	for(int i = 0; i < listCount; i++){
		SoundArray* soundArray = list_[i];
		int soundCount = soundArray->getSoundCount();
		for(int j = 0; j < soundCount; j++){
			soundArray->getSound(j)->suspend();
		}
	}
}
//------------------------------------------------------------------------------
// STEh̍ĐĊJ
void SoundCache::resumeAll(){
	int listCount = list_.getCount();
	for(int i = 0; i < listCount; i++){
		SoundArray* soundArray = list_[i];
		int soundCount = soundArray->getSoundCount();
		for(int j = 0; j < soundCount; j++){
			soundArray->getSound(j)->resume();
		}
	}
}
//------------------------------------------------------------------------------
// ւ̕ϊ
String SoundCache::toString() const{
	String result;
	int listCount = list_.getCount();
	int cacheCount = 0;
	for(int i = 0; i < listCount; i++){
		cacheCount += list_[i]->getSoundCount();
	}
	result.format("Cache/List (%d/%d)  ", cacheCount, listCount);
	if(focus_ == Sound::focusNormal){
		result += "Normal\n";
	}else if(focus_ == Sound::focusSticky){
		result += "Sticky\n";
	}else if(focus_ == Sound::focusGlobal){
		result += "Global\n";
	}
	result += "Path " + basePath_ + "  Extension " + extension_ + "\n";
	for(int i = 0; i < listCount; i++){
		result += list_[i]->toString();
	}
	return result;
}
//------------------------------------------------------------------------------
// TEhz
//------------------------------------------------------------------------------
// RXgN^
SoundCache::SoundArray::SoundArray(Sound* sound, int maxMixingCount) :
	sounds_(maxMixingCount), maxMixingCount_(maxMixingCount){
	Assert(sound != NULL);
	sounds_.add(sound);
}
//------------------------------------------------------------------------------
// fXgN^
SoundCache::SoundArray::~SoundArray(){
	// STEhj
	int count = sounds_.getCount();
	SoundManager* manager = LampSound::getSoundManager();
	for(int i = count - 1; i >= 0; i--){ manager->destroy(sounds_[i]); }
	sounds_.clear();
}
//------------------------------------------------------------------------------
// t[TEh擾
Sound* SoundCache::SoundArray::getFreeSound(){
	Assert(!sounds_.isEmpty());
	Sound* topSound = sounds_[0];
	// TEh̎擾
	int count = sounds_.getCount();
	for(int i = 0; i < count; i++){
		Sound* sound = sounds_[i];
		// LΖ
		if(!sound->hasOwnership()){ continue; }
		Sound::State state = sound->getState();
		// ĐATXyh͖
		// ŃXg𖳎ƁATEhʂɐ
		if((state == Sound::statePlay) ||
			(state == Sound::stateSuspend)){ continue; }
		return sound;
	}

	// 
	int soundCount = sounds_.getCount();
	if(soundCount == maxMixingCount_){
		u_int maxCursor = 0, maxIndex = -1;
		for(int i = 0; i < soundCount; i++){
			Sound* sound = sounds_[i];
			// LΖ
			if(!sound->hasOwnership()){ continue; }
			u_int cursor = sound->getCursor();
			if(cursor > maxCursor){
				maxCursor = cursor;
				maxIndex = i;
			}
		}
		if(maxIndex == -1){ return NULL; }
		return sounds_[maxIndex];
	}

	// 󂫂Ȃ̂ŕ쐬
	Sound* clone = topSound->clone();
	if(clone != NULL){
		sounds_.add(clone);
		return clone;
	}
	return NULL;
}
//------------------------------------------------------------------------------
// ւ̕ϊ
String SoundCache::SoundArray::toString() const{
	String result, temp;
	Sound* sound = sounds_[0];
	result.format("  %2d/%2d ", sounds_.getCount(), maxMixingCount_);
	if(sound->isStereoSound()){
		result += "Stereo ";
	}else{
		result += "3D     ";
	}
	if(sound->isLoop()){
		result += "Loop ";
	}else{
		result += "Once ";
	}
	temp.format("Pri%4d ", sound->getPriority());
	result += temp;

	// t@C̑OɂĂpX͏o͂Ȃ
	FilePath path(sound->getName());
	result += path.getName();

	Sound3D* sound3D = sound->castSound3D();
	if(sound3D != NULL){
		temp.format(" Dist %.1f-%.1f ",
			sound3D->getMinimumDistance(), sound3D->getMaximumDistance());
		result += temp;
	}
	result += "\n";
	return result;
}
//------------------------------------------------------------------------------
} // End of namespace Lamp
//------------------------------------------------------------------------------
